/*
 * str.h --
 *
 *      Housing conversions to and from strings etc.
 *
 * Copyright (c) 1997-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef STR_H
#define STR_H

#include "tclcl.h"

extern "C"
{
#include <string.h>
#include <stdlib.h>
#include <tcl.h>
};

inline int str2double(const char* szDouble, double& result) {
        result=0.0;
        return (TCL_OK==Tcl_GetDouble(Tcl::instance().interp(),
                                  (char*)szDouble, &result));
}

/* note: there is no range checking! */
inline int str2float(const char* szFloat, float& result)
{
        double d;
        if (!str2double(szFloat,d))
                return 0;
        else {
                result=(float)d;
                return 1;
        }
}

inline int str2int(const char* szInt, int& result) {
        return (TCL_OK==Tcl_GetInt(Tcl::instance().interp(),
                               (char*)szInt, &result));
}

inline int str2ulong(const char* szULong, unsigned long& result) {
	char* szEnd;
	unsigned long ul = strtoul(szULong, &szEnd, 10);
	if (szEnd == szULong) return 0;
	result = ul;
	return 1;
}

// some commonly used characters
const char cchSpc = ' ';
const char cchNull = '\0';
const char cchDel = '\b';
const char cchCR = '\n';
const char cchTab = '\t';

/*
 * Allocate space in pszDest, and copy szSrc to it
 * Note: this procedure does not delete *ppDest !!!
 */
inline void AllocNCopy(char **pszDest, const char *szSrc)
{
    if (!szSrc) return;

    *pszDest = new char[strlen(szSrc)+1];
    strcpy(*pszDest, szSrc);
}

inline void AllocNNCopy(char **pszDest, const char *szSrc, int len)
{
    if (!szSrc) return;

    *pszDest = new char[len+1];
    strncpy(*pszDest, szSrc, len);
    (*pszDest)[len]=cchNull;
}

/*
 * Concatenate 5 strings and return the result
 */
inline char* Concat5(const char *sz1, const char *sz2, const char *sz3,
                     const char *sz4, const char *sz5)
{
    char *szNew = new char[strlen(sz1)+strlen(sz2)+
                          strlen(sz3)+strlen(sz4)+strlen(sz5)+1];
    strcpy(szNew, sz1);
    strcat(szNew, sz2);
    strcat(szNew, sz3);
    strcat(szNew, sz4);
    strcat(szNew, sz5);
    return szNew;
}


/*
 * Concatenate 4 strings and return the result
 */
inline char* Concat4(const char *sz1, const char *sz2, const char *sz3,
                     const char *sz4)
{
   char *szNew = new char[strlen(sz1)+strlen(sz2)+strlen(sz3)+strlen(sz4)+1];
   strcpy(szNew, sz1);
   strcat(szNew, sz2);
   strcat(szNew, sz3);
   strcat(szNew, sz4);
   return szNew;
}

/*
 * Concatenate 3 strings and return the result
 */
inline char* Concat3(const char *sz1, const char *sz2, const char *sz3)
{
   char *szNew = new char[strlen(sz1)+strlen(sz2)+strlen(sz3)+3];
   strcpy(szNew, sz1);
   strcat(szNew, sz2);
   strcat(szNew, sz3);
   return szNew;
}

/*
 * Concatenate 2 strings and return the result
 */
inline char* Concat(const char *sz1, const char *sz2)
{
   char *szNew = new char[strlen(sz1)+strlen(sz2)+2];
   strcpy(szNew, sz1);
   strcat(szNew, sz2);
   return szNew;
}

#endif /* #ifdef STR_H */
