/***************************************************************************
                          jdxnumbers.h  -  description
                             -------------------
    begin                : Mon Jul 12 2004
    copyright            : (C) 2001 by Thies H. Jochimsen
    email                : jochimse@cns.mpg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef JDXNUMBERS_H
#define JDXNUMBERS_H


#include <odinpara/jdxbase.h>
#include <tjutils/tjtypes.h>


/**
  * @addtogroup jcampdx
  * @{
  */
//////////////////////////////////////////////////////////////////

/**
  *
  *  JCAMP-DX template class for representing built-in numbers
  */
template<class T>
class JDXnumber : public virtual JcampDxClass  {

 public:

/**
  *  Default constructor
  */
  JDXnumber() {set_defaults();}

/**
  *  Constructor with the following arguments:
  * - v:             Initial value for the number
  * - name:          The label of the JCAMP-DX parameter
  * - userParameter: Whether this is a user defined JCAMP-DX parameter
  * - mode:          Mode for ASCII representation of strings or arrays of strings
  * - parameter_mode: Mode for GUI accesibility of the parameter
  * - parx_equivalent: Equivalent parameter in PARX to which this parameter will be assigned
  * - parx_assign_factor: Scaling factor when assigning to the equivalent parameter in PARX
  * - parx_assign_offset: Offset when assigning to the equivalent parameter in PARX
  */
  JDXnumber(T v, const STD_string& name="",bool userParameter=true,
            compatMode mode=notBroken,parameterMode parameter_mode=edit,
            const STD_string& parx_equivalent="",
            float parx_assign_factor=1.0,float parx_assign_offset=0.0);


/**
  *  Copy constructor
  */
  JDXnumber(const JDXnumber<T>& bi) {JDXnumber<T>::operator = (bi);}

/**
  *  Assignment operator from a built-in number
  */
  JDXnumber<T>& operator = (T v) {val=v; return *this;}

/**
  *  Copy assignment
  */
  JDXnumber<T>& operator = (const JDXnumber<T>& bi);

  // code to make the number appear like a built-in number
  operator T () const {return val;}
  T operator += (T rhsval) {val+=rhsval; return *this;}
  T operator -= (T rhsval) {val-=rhsval; return *this;}
  T operator *= (T rhsval) {val*=rhsval; return *this;}
  T operator /= (T rhsval) {val/=rhsval; return *this;}
  T operator ++ ()    {val=val+(T)1; return val;}            // prefix
  T operator ++ (int) {T tmp=val; val=val+(T)1; return tmp;} // postfix
  T operator -- ()    {val=val-(T)1; return val;}            // prefix
  T operator -- (int) {T tmp=val; val=val-(T)1; return tmp;} // postfix

/**
  * Specifies the minimum and maximum allowed value for this number.
  * Only useful when editing the parameter in a GUI
  */
  JDXnumber<T>& set_minmaxval(double min,double max) {minval=min; maxval=max; return *this;}

  // overwriting virtual functions from JcampDxClass
  STD_string printvalstring() const;
  bool parsevalstring (const STD_string& parstring);
  STD_string get_parx_code(parxCodeType type, const ParxEquiv& equiv) const;
  ParxEquiv get_parx_equiv() const {return parx_equiv;}
  double get_minval() const {return minval;}
  double get_maxval() const {return maxval;}
  const char* get_typeInfo() const {return TypeTraits::type2label(val);}
  JcampDxClass* create_copy() const {return new JDXnumber<T>(*this);}
  T* cast(T*) {return &val;}

 private:

  void set_defaults();

  T val;
  mutable ParxEquiv parx_equiv;
  double minval,maxval;
};

/////////////////////////////////////////////////////////////////////////////
//
// Aliases:

/**
  * An integer number
  */
typedef JDXnumber<int>   JDXint;

/**
  * An single-precision floating point number
  */
typedef JDXnumber<float>   JDXfloat;

/**
  * An double-precision floating point number
  */
typedef JDXnumber<double>   JDXdouble;

/**
  * Complex number
  */
typedef JDXnumber<STD_complex>   JDXcomplex;


/** @}
  */

#endif

