// Copyright (C) 2003 Shai Ayal <shaiay@users.sourceforge.net>
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  

#include <iostream>
#include <stdio.h>
#include <vector>
#include <algorithm>
#include <FL/Fl.H>
#include "figure.h"
#include "figurewindow.h"
#include "property.h"
#include "root.h"
#include "../config.h"

#define OBJ_NAME "root"

Root::Root(ocpl::Handle Parent) : Object(Parent)
{
  //  const char* clipping[] = {"on","off"};
  char onoff[] = "|{on}|off|";
  char offon[] = "|on|{off}|";

  SET_TYPE;
  Properties["Children"]      = new HandleVect;
  Properties["CurrentFigure"] = new HandleScalar;

  ///////////////////////////////////////////
  // Defaults
  ///////////////////////////////////////////

  // Figure
  Properties["DefaultFigurePosition"] = Real2Matrix(10,10,400,300);
  Properties["DefaultFigureColor"] =  new Color(0.9,0.9,0.9,1.0);
  // default grayscale colormap
  const int cmap_size=64;
  double* gray = new double[3*cmap_size];
  for(int i=0;i<cmap_size;i++) {
    gray[i] = gray[i+cmap_size] = gray[i+2*cmap_size] = 
      static_cast<double>(i)/(cmap_size-1);
  }
  Properties["DefaultFigureColorMap"] = new Matrix(gray,cmap_size,3);

  
  // Axes
  Properties["DefaultAxesColor"]     = new ColorNone('w');
  Properties["DefaultAxesXColor"]    = new ColorNone('k');
  Properties["DefaultAxesYColor"]    = new ColorNone('k');
  Properties["DefaultAxesZColor"]    = new ColorNone('k');
  Properties["DefaultAxesPosition"]  = Real2Matrix(0.1,0.1,0.8,0.8);
  Properties["DefaultAxesLinewidth"] = new Scalar(1.0);
  Properties["DefaultAxesTickLength"]= Real2Matrix(5,5);
  Properties["DefaultAxesGridLineStyle"]  = new Radio("|:|-|--|-.|{none}|");
  char *as="|{linear}|log|";
  Properties["DefaultAxesXScale"] = new Radio(as);
  Properties["DefaultAxesYScale"] = new Radio(as);
  Properties["DefaultAxesZScale"] = new Radio(as);
  Properties["DefaultAxesXAxisLocation"] = new Radio("|top|{bottom}|");
  Properties["DefaultAxesYAxisLocation"] = new Radio("|right|{left}|");
  Properties["DefaultAxesBox"] = new Radio(onoff);
  const double color_order[] ={
    1,0,0,1,0,1,0,
    0,0,1,1,1,0,0,
    0,1,0,0,1,1,0,
    1,1,1,1,1,1,1
  };
  const int co_size = sizeof(color_order);
  double* co= new double[co_size];
  for(int i=0;i<co_size;i++) {
    co[i] = color_order[i];
  }
  Properties["DefaultAxesColorOrder"] = new Matrix(co,7,4);
  Properties["DefaultAxesFontSize"] = new Scalar(10);
  Properties["DefaultAxesTag"] = new String("");

  // line & marker for types with lines
  const char* linestyle = "|{-}|--|:|-.|none|";
  const char* marker = "|{none}|+|o|*|.|x|s|d|^|v|>||<||p||h|";

  // Text
  Properties["DefaultTextColor"] =  new Color('k');
  Properties["DefaultTextClipping"] = new Radio(offon);
  Properties["DefaultTextHorizontalAlignment"] = 
    new Radio("|{left}|center|right|");
  Properties["DefaultTextVerticalAlignment"] = 
    new Radio("|top|{middle}|bottom|");
  Properties["DefaultTextFontName"] = 
    new Radio("|{sans}|mono|serif|");
  Properties["DefaultTextFontSize"] = new Scalar(10);
  Properties["DefaultTextRotation"] = new Scalar(0);
  Properties["DefaultTextBackgroundColor"] = new ColorNone("none");
  Properties["DefaultTextEdgeColor"] = new ColorNone("none");
  Properties["DefaultTextLineWidth"] =  new Scalar(1.0);
  Properties["DefaultTextLineStyle"] = new Radio(linestyle);

  // Line
  Properties["DefaultLineColor"] =  new Color('r');
  Properties["DefaultLineMarkerEdgeColor"] =  new ColorNone('r');
  Properties["DefaultLineMarkerFaceColor"] =  new ColorNone('w');
  Properties["DefaultLineMarkerSize"] =  new Scalar(10);
  Properties["DefaultLineLineWidth"] =  new Scalar(1.0);
  Properties["DefaultLineClipping"] = new Radio(onoff);
  Properties["DefaultLineLineStyle"] = new Radio(linestyle);
  Properties["DefaultLineMarker"]    = new Radio(marker);
  Properties["DefaultLineArrowHead"] = new Radio("|{none}|<|>|<>|");
  Properties["DefaultLineArrowLength"] = new Scalar(0.1);
  Properties["DefaultLineArrowWidth"] = new Scalar(0.05);
  Properties["DefaultLineArrowRatio"] = new Scalar(0.5);

  // Patch
  Properties["DefaultPatchFaceAlpha"] = new Scalar(1.0);
  Properties["DefaultPatchFaceColor"] = new 
    ColorRadio('k',false,"|{flat}|interp|none|");
  Properties["DefaultPatchEdgeColor"] = new ColorNone('k');
  Properties["DefaultPatchLineStyle"] = new Radio(linestyle);
  Properties["DefaultPatchLineWidth"] =  new Scalar(1.0);
  Properties["DefaultPatchClipping"] = new Radio(onoff);
  Properties["DefaultPatchMarker"]    = new Radio(marker);
  Properties["DefaultPatchMarkerEdgeColor"] =  new ColorNone('r');
  Properties["DefaultPatchMarkerFaceColor"] =  new ColorNone('w');
  Properties["DefaultPatchMarkerSize"] =  new Scalar(10);

  // Surface
  Properties["DefaultSurfaceClipping"] = new Radio(onoff);
  Properties["DefaultSurfaceLineWidth"] = new Scalar(0.5); 
  Properties["DefaultSurfaceFaceColor"] = new ColorRadio('g',true,"|{flat}|interp|none|");
  Properties["DefaultSurfaceEdgeColor"] = new ColorRadio('k',false,"|{none}|");
  Properties["DefaultSurfaceLineStyle"] = new Radio(linestyle);
  Properties["DefaultSurfaceMarker"] = new Radio(marker);
  Properties["DefaultSurfaceMarkerFaceColor"] = new Color('w');
  Properties["DefaultSurfaceMarkerEdgeColor"] = new Color('r');
  Properties["DefaultSurfaceMarkerSize"] = new Scalar(10);
  Properties["DefaultSurfaceMeshStyle"] = new Radio("|{both}|column|row|");
  Properties["DefaultSurfaceFaceAlpha"] = new Scalar(1.0);

  // Legend
  Properties["DefaultLegendColor"]     = new Color('k');
  Properties["DefaultLegendFaceColor"] = new
    ColorRadio('w',false,"|{flat}|interp|none|");
  Properties["DefaultLegendEdgeColor"] = new ColorNone('k');
  Properties["DefaultLegendClipping"]  = new Radio(offon);
  Properties["DefaultLegendFontName"]  = 
    new Radio("|{sans}|mono|serif|");
  Properties["DefaultLegendFontSize"]  = new Scalar(10);
  Properties["DefaultLegendLocation"]  = 
    new Radio("|n|{ne}|e|se|s|sw|w|nw|north|northeast|east|southeast|south|southwest|west|northwest|");

  // FontDir
#ifndef _MSC_VER
  Properties["DefaultFontDir"]  = new String(FONTPATH);
#else
  int n = 1024;
  std::string fpath(n, '\0');

  while (true) {
    int status = GetModuleFileName(0, &fpath[0], n);

    if (status < n) {
      fpath.resize(status);
      break;
    }
    else {
      n *= 2;
      fpath.resize(n);
    }
  }

  if (!fpath.empty()) {
    int pos = fpath.rfind("\\oct\\");
    Properties["DefaultFontDir"]  = new String(fpath.substr(0, pos) + "\\fonts");
  }
#endif
}

//! Addes a new Figure object, taking care of the numbering etc...
// requested num is now silently ignored ....
void Root::AddFigure(int requested_num)
{
  char Caption[25];
  MAKE_REF(children,HandleVect);
  MAKE_REF(currentfigure,HandleScalar);
  MAKE_REF(defaultfigureposition,Matrix);

  // get a new figure number. This should be the lowest available
  int fignum = GetFignum(requested_num);
  if(fignum<0) return; // this figure exists
  
  sprintf(Caption,"octplot - Figure %d",fignum);

  FigureWindow *nf = 
    new FigureWindow(
      (int)defaultfigureposition[0],
      (int)defaultfigureposition[1],
      (int)defaultfigureposition[2],
      (int)defaultfigureposition[3],
      Caption,
      fignum,
      20);
  nf->size_range(40, 40);
  nf->AddChildren(GetHandle());
  Figure* Current = nf->fig;
  children.Add(Current->GetHandle());
//  Current->AddAxes();
  currentfigure.Clear();
  currentfigure.Add(Current->GetHandle());

  nf->show();
}

void Root::DelWindow(FigureWindow* fw)
{
  ocpl::Handle hnd = fw->fig->GetHandle();
  DeleteChild(hnd);
}

void Root::DeleteChild(ocpl::Handle Child)
{
  MAKE_REF(children,HandleVect);
  MAKE_REF(currentfigure,HandleScalar);

  Figure *fig = dynamic_cast<Figure*>(GetObjectD(Child)); 
  fig->win->hide();
  delete fig->win;

  children.Delete(Child);
  if(currentfigure()==Child)
  {
    currentfigure.Clear();
    ocpl::Handle last=0;
    children.First();
    while(!children.IsDone()) {
        last = children.CurrentHandle();
        children.Next();
    }
    if(last) {
      currentfigure.Add(last);
      Figure *fig = dynamic_cast<Figure*>(GetObjectD(last));
      Fl::focus(fig->win);
    }
  }
}

void Root::Parse(ocpl::command& command)
{
  MAKE_REF(currentfigure,HandleScalar);

  if(command.id()==ocpl::figure) {
    command.parsed(true);
    if(command.nargin()>1) {
      ret_error(command,"figure accepts 0 or 1 arguments");
      return;
    }
    
    if(command.nargin()==1) {
      int newfig = static_cast<int>(command.argin(0)->real_data()[0]);
      AddFigure(newfig);
    }
    else{
      AddFigure();
    }
  
    command.dirty(true);
    
    //return handle
    double* hnd = new double(static_cast<double>(currentfigure()));
    command.argout(0,ocpl::real,1,1,reinterpret_cast<char*>(hnd),true);
  }
  else { // All other commands should be passed down
    
    if(!currentfigure.Size()) {
      // redraw should not create a new figure, all others shpould
      if(command.id() == ocpl::redraw) {
        command.init_argout(0);
        return;
      }
      else {
        AddFigure();
      }
    }
      
    ::GetObjectD(currentfigure())->Parse(command);
    if(command.dirty()) dirty=true;
  }
}

bool Root::PreSet(ocpl::command& com)
{
  MAKE_REF(children,HandleVect)
  
  std::string prop(tolower(com.argin(1)->data));
  
  if(prop=="currentfigure") {
    ocpl::Handle new_ca = 
      static_cast<ocpl::Handle>(*(com.argin(2)->real_data()));
    if(!children.exist(new_ca)) {
      ocpl::ret_error(com,"Can't set CurrentFigure to non child");
      return false;
    }
  }

  return true;
}

void Root::PostSet(ocpl::command& com)
{
  MAKE_REF(children,HandleVect)
  MAKE_REF(currentfigure,HandleScalar)
  
  std::string prop(tolower(com.argin(1)->data));
  
  // raise current figure
  if(prop=="currentfigure") {
    Figure* fig = dynamic_cast<Figure*>(GetObjectD(currentfigure()));
    assert(fig);
    fig->win->show();
  }
}

int Root::GetFignum(int newnum) 
{
  int retnum;
  std::vector<int> nums;

  // populate nums vector
  MAKE_REF(children,HandleVect);
  for(children.First(); !children.IsDone() ; children.Next() )
    nums.push_back(dynamic_cast<Figure*>
		   ( (::GetObjectD(children.CurrentHandle())))->fignum());

  if(newnum == -1) {
    //find lowest free number

    // special casses
    if(nums.size()==0) {
      retnum=1;
    }
    else if(nums.size()==1) {
      if(nums[0] == 1) retnum = 2;
      else retnum = 1;
    }
    else { // general case
      std::sort(nums.begin(),nums.end());
      if(nums[0] > 1) retnum = 1;
      else {
	int i;
	for(i=1;i<nums.size();i++)
	  if(nums[i] - nums[i-1] > 1) break;

	retnum = nums[i-1]+1;
      }
    }
  }
  else { // number was requested -- see if it exists
    retnum = newnum;
    for(int i=0;i<nums.size();i++) {
      if(nums[i]==newnum) retnum = -1;
    }
  }
      
  return retnum;
}
