## Copyright (C) 2004  Dragan Tubic
## 
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2, or (at your option)
## any later version.
## 
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
## General Public License for more details. 
## 
## You should have received a copy of the GNU General Public License
## along with this file.  If not, write to the Free Software Foundation,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

## -*- texinfo -*-
## @deftypefn {Function File} {} vtk_plot3(@var{x}, @var{y}, @var{z}, [@var{fmt} | @var{prop},@var{val}])
## Plots a 3D point cloud.  The data point coordinates must be given as
## 3 equal length @var{x}, @var{y}, and @var{z} vectors.  The optional
## @var{fmt} takes the form of a "line specification"; e.g. "*r" would cause
## the points to be plotted as red spheres.  The optional
## @var{prop},@var{val} is a property,value pair argument.  Valid
## properties are MarkerColor, MarkerSize, and Opacity.
## @end deftypefn
## @seealso{vtk_arrows3,vtk_get_line_spec}

## Author: Dragan Tubic

function vtk_plot3(varargin)
  
  valid_props = ";MarkerColor;MarkerSize;Opacity;";
  [no_numerical_params, first_prop_index, line_spec_index] = vtk_parse_params(valid_props, all_va_args);
  
  if ( no_numerical_params < 3 )
    error("Syntax is vtk_plot3(x,y,z)");
  end
  
  x = nth (varargin,1);
  y = nth (varargin,2);
  z = nth (varargin,3);

  ## set default marker type and color
  marker_type = "*";
  color = [0 0 1];   # same as default in vtk_get_line_spec function

  ## use "line specs", if provided
  if ( line_spec_index > 0 )
    line_spec = nth (varargin,line_spec_index);
    [color, marker_type, line_style] = vtk_get_line_spec( line_spec );
  end
  ## use "property specs", if provided
  if ( first_prop_index > 0 )
    properties = struct(varargin{first_prop_index:length(varargin)});
    if ( isfield(properties,"MarkerColor") )
      color = properties.MarkerColor;
    end
  end
  properties.MarkerColor = color;

  
  vtk_init;   
  
  
  %% We'll create the building blocks of polydata including data attributes.
  polydata = vtkPolyData();
  points   = vtkPoints();
  polys    = vtkCellArray();
  scalars  = vtkFloatArray();
  
  
  %% Load the point data.
  coords = vtkFloatArray; 
  coords.SetNumberOfTuples( length(x) );
  coords.SetNumberOfComponents(3);
  pts = [x y z]';
  coords.SetArray( pts(:), 3*length(x), 0 );
  points.SetData(coords); 
  
  %% We now assign the pieces to the vtkPolyData.
  vtkPointSet(polydata,"SetPoints",points);
  
  if struct_contains(properties,"MarkerSize")
    src_size = properties.MarkerSize;
  else
    src_size = 0.05;
  end
  
  %% Create the "cloud" of particles, using a vtkGlyph3D object
  cloud = vtkGlyph3D();
  cloud.SetInput(polydata);
  
  if marker_type == "."
    point_source = vtkPointSource();
    point_source.SetRadius(0);
    point_source.SetNumberOfPoints(1);
    pd2 = point_source.GetOutput();
  elseif marker_type == "^"
    cube_source = vtkCubeSource();
    cube_source.SetXLength(src_size);
    cube_source.SetYLength(src_size);
    cube_source.SetZLength(src_size);
    pd2 = cube_source.GetOutput();
  elseif marker_type == ">"
    cone_source = vtkConeSource();
    cone_source.SetRadius(src_size);
    pd2 = cone_source.GetOutput();
  else
    sphere_source = vtkSphereSource();
    sphere_source.SetRadius(src_size);
    sphere_source.SetPhiResolution( 15 );
    sphere_source.SetThetaResolution( 30 );
    pd2 = sphere_source.GetOutput();
  end     
  
        
  cloud.SetSource(pd2);
  
  %% Now we'll look at it.
  cloudMapper = vtkPolyDataMapper();
  pd3 = cloud.GetOutput();
  cloudMapper.SetInput(pd3);
  
  cloud_actor = vtkActor();
  cloud_actor.SetMapper(cloudMapper);

  c = properties.MarkerColor;
  cloud_actor.GetProperty().SetDiffuseColor( c(1), c(2), c(3) );
  
  if struct_contains(properties,"Opacity")
    op = properties.Opacity;
    cloud_actor.GetProperty().SetOpacity( op );
  end
  
  f = vtk_figure(0);
  f.renderer.AddProp(cloud_actor);
  vtk_update(f);
  
endfunction
