(* ocamlgsl - OCaml interface to GSL                        *)
(* Copyright () 2002 - Olivier Andrieu                     *)
(* distributed under the terms of the GPL version 2         *)


open Gsl_fun

(* PLAIN algorithm *)
type plain_state

external alloc_plain : int -> plain_state
    = "ml_gsl_monte_plain_alloc"

external init_plain : plain_state -> unit
    = "ml_gsl_monte_plain_init"

external free_plain : plain_state -> unit
    = "ml_gsl_monte_plain_free"

external integrate_plain : monte_fun -> lo:float array -> up:float array ->
  int -> Gsl_rng.t -> plain_state -> Gsl_fun.result
    = "ml_gsl_monte_plain_integrate_bc" "ml_gsl_monte_plain_integrate"


(* MISER algorithm *)
type miser_state
type miser_params = {
    estimate_frac : float ;		(* 0.1 *)
    min_calls     : int ;		(* 16 * dim *)
    min_calls_per_bisection : int ;	(* 32 * min_calls *)
    miser_alpha   : float ;		(* 2. *)
    dither        : float ;             (* 0. *)
  } 

external alloc_miser : int -> miser_state
    = "ml_gsl_monte_miser_alloc"

external init_miser : miser_state -> unit
    = "ml_gsl_monte_miser_init"

external free_miser : miser_state -> unit
    = "ml_gsl_monte_miser_free"

external integrate_miser : monte_fun -> lo:float array -> up:float array ->
  int -> Gsl_rng.t -> miser_state -> Gsl_fun.result
    = "ml_gsl_monte_miser_integrate_bc" "ml_gsl_monte_miser_integrate"

external get_miser_params : miser_state -> miser_params
    = "ml_gsl_monte_miser_get_params"

external set_miser_params : miser_state -> miser_params -> unit
    = "ml_gsl_monte_miser_set_params"


(* VEGAS algorithm *)
type vegas_state
type vegas_info = {
    result : float ;
    sigma  : float ;
    chisq  : float ;
  } 
type vegas_mode = | STRATIFIED | IMPORTANCE_ONLY | IMPORTANCE 
type vegas_params = {
    vegas_alpha   : float ;		  (* 1.5 *)
    iterations    : int ;		  (* 5 *)
    stage         : int ;
    mode          : vegas_mode ;
    verbose       : int ;		  (* 0 *)
    ostream       : out_channel option ;  (* stdout *)
  } 

external alloc_vegas : int -> vegas_state
    = "ml_gsl_monte_vegas_alloc"

external init_vegas : vegas_state -> unit
    = "ml_gsl_monte_vegas_init"

external free_vegas : vegas_state -> unit
    = "ml_gsl_monte_vegas_free"

external integrate_vegas : monte_fun -> lo:float array -> up:float array ->
  int -> Gsl_rng.t -> vegas_state -> Gsl_fun.result
    = "ml_gsl_monte_vegas_integrate_bc" "ml_gsl_monte_vegas_integrate"

external get_vegas_info : vegas_state -> vegas_info
    = "ml_gsl_monte_vegas_get_info"

external get_vegas_params : vegas_state -> vegas_params
    = "ml_gsl_monte_vegas_get_params"

external set_vegas_params : vegas_state -> vegas_params -> unit
    = "ml_gsl_monte_vegas_set_params"



(* High-level version *)
type kind =
  | PLAIN
  | MISER
  | VEGAS

let integrate kind f ~lo ~up calls rng =
  let dim = Array.length lo in
  match kind with
  | PLAIN ->
      let state = alloc_plain dim in
      let res = integrate_plain f ~lo ~up calls rng state in
      free_plain state ;
      res
  | MISER ->
      let state = alloc_miser dim in
      let res = integrate_miser f ~lo ~up calls rng state in
      free_miser state ;
      res
  | VEGAS ->
      let state = alloc_vegas dim in
      let res = integrate_vegas f ~lo ~up calls rng state in
      free_vegas state ;
      res
