(**************************************************************************)
(*  Copyright (c) 2003 Mikhail Fedotov <mikhail@kittown.com>              *)
(*                                                                        *)
(*  Permission is hereby granted, free of charge, to any person           *)
(*  obtaining a copy of this software and associated documentation files  *)
(*  (the "Software"), to deal in the Software without restriction,        *)
(*  including without limitation the rights to use, copy, modify, merge,  *)
(*  publish, distribute, sublicense, and/or sell copies of the Software,  *)
(*  and to permit persons to whom the Software is furnished to do so,     *)
(*  subject to the following conditions:                                  *)
(*                                                                        *)
(*  The above copyright notice and this permission notice shall be        *)
(*  included in all copies or substantial portions of the Software.       *)
(*                                                                        *)
(*  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,       *)
(*  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES       *)
(*  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND              *)
(*  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS   *)
(*  BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN    *)
(*  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN     *)
(*  CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE      *)
(*  SOFTWARE.                                                             *)
(**************************************************************************)

(** API to work with Sqlite databases *)

(** {1 Exceptions} *)

(** Raised when an sqlite operation generates an error.  The error code
can be obtained with [db_rc] for functions that take a [db] as the first
argument, and with [vm_rc] for functions that take a [vm] as the first
argument.

When the error code is [RC_misuse], it usually means that a programming
error was made. An attempt to use an invalidated [db] handle (i.e. after
[Sqlite.close] was called on this handle) is an example. *)
exception Sqlite_error of string

(** Raised when an attempt is made to fetch a row from a query result
all of whose rows have been fetched. *)
exception Sqlite_done

(** Raised when a query is made to a busy database. The query may be
attempted at a later time. *)
exception Sqlite_busy

(** Raised by [Sqlite.step_simple] when a null value is found in the
result row. *)
exception Sqlite_null_value

(** {1 Types} *)

(** A database handle. A value of this type stores information about
an open database and the status code of the last operation which
took the database handle as a parameter. *)
type db

(** A virtual machine handle. A value of this type stores information
about a virtual machine created by the [compile] or [compile_simple]
function.

Column names and data types are stored in a [vm] handle as soon as
they become known.

A [vm] handle also stores the status code of the last operation
which took the virtual machine handle as a parameter. For example,
[Sqlite.compile] stores its status code in the [db] handle because
it takes the [db] handle as the first parameter, but [Sqlite.finalize]
stores its status code in the [vm] handle passed. *)
type vm

(** Possible status codes from an sqlite operation. Can be obtained
from a [db] handle or a [vm] handle depending on the function called.
Note: RC_misuse is the common error code when an invalid [db] or [vm]
is passed to a function.*)
type rc =
   RC_ok
 | RC_error
 | RC_internal
 | RC_perm
 | RC_abort
 | RC_busy
 | RC_locked
 | RC_nomem
 | RC_readonly
 | RC_interrupt
 | RC_ioerr
 | RC_corrupt
 | RC_notfound
 | RC_full
 | RC_cantopen
 | RC_protocol
 | RC_empty
 | RC_schema
 | RC_toobig
 | RC_constraint
 | RC_mismatch
 | RC_misuse 
 | RC_nofls
 | RC_auth
 | RC_format

(** {1 Database operations} *)

(** {2 General database operations} *)

(** [Sqlite.db_open f_name] opens the database file [f_name] and
returns a database handle. *)
external db_open  : string -> db = "caml_sqlite_open"

(** [Sqlite.db_close db_handle] closes the database file associated with
[db_handle] and invalidates the handle.  Raises an [Sqlite_error] exception
if an invalid handle is passed. *)
external db_close : db -> unit = "caml_sqlite_close"

(** [Sqlite.exec db_handle q_string] executes the query [q_string]. The
query result is ignored. *)
external exec  : db -> string -> unit = "caml_sqlite_exec"

(** [Sqlite.last_insert_rowid db_handle] returns the rowid of the last
inserted row. *)
external last_insert_rowid : db -> int = "caml_sqlite_last_insert_rowid"

(** [Sqlite.db_rc db_handle] returns the status code of the last
operation on [db_handle]. *)
external db_rc : db -> rc = "caml_sqlite_db_retcode"

(** {2 Database operations involving use of the Virtual Machine} *)

(** The following functions are used to compile a query to the sqlite
virtual machine instance. After compilation, result rows can be fetched.
Actual execution of the query begins when the first attempt to get a result
row is made. Column names and types can be obtained from the [vm] handle
after the first attempt. When there are no result rows left (or there
were no rows to fetch at all), an attempt to fetch a row raises an
[Sqlite_done] exception and immediately finalizes the virtual machine
involved. Column names and types can be obtained even after the [vm]
handle is finalized (and thus invalidated) if it was created with a
[compile] call with the third parameter set to [true]. *) 

(** NOTE: ALL VIRTUAL MACHINE HANDLES MUST BE FINALIZED BEFORE THE
DATABASE IS CLOSED! *)

(** [Sqlite.compile db_handle q_string skip keep_col_info] reads
a single query from the string [q_string], skipping [skip] initial
characters, and compiles it. It returns a [vm] handle, the position
in [q_string] of the still-unused part of the string, and a boolean
value telling if the end of the string has been reached.

Passing character positions is necessary for strings containing
more than one statement, since a [vm] is associated with a single
statement.

If [keep_col_info] is true, the [vm] handle will retain information
about column names and data types after it is finalized (e.g. with
[sqlite.finalize]), but not yet garbage-collected. Of course, the
information is saved only if it is known at the time of finalization.
*)
external compile : db -> string -> int -> bool -> (vm * int * bool) = "caml_sqlite_compile"

(** [Sqlite.compile_simple db_handle q_string] compiles a single query,
read from the beginning of the string [q_string]. It returns a [vm] handle.
Any unused remainder of the [q_string] is ignored. Information about
column names and types is not available after the [vm] is finalized. *)
external compile_simple: db -> string -> vm = "caml_sqlite_compile_simple"

(** [Sqlite.step vm default_value] executes [vm] to return the next
row of the result as a string array. Null values are replaced with
[default_value]. Even if the query compiled to [vm] is not expected
to return any rows, the function should be called at least once for
the query to be executed. The use of [exec] is preferred for such
queries, however.*)
external step        : vm -> string -> string array          = "caml_sqlite_step"

(** [Sqlite.step_simple] is the same as [Sqlite.step], except that it
raises a [Sqlite_null_value] exception if the result row contains a
null value, instead of replacing it with a default value. *)
external step_simple : vm -> string array                    = "caml_sqlite_step_simple"

(** [Sqlite.step_opt vm] executes [vm] and returns the next row
of the result as a string option array. Null values are returned as
[None] array elements, and non-null values as [Some(value)] elements.*)
external step_opt    : vm -> string option array             = "caml_sqlite_step_opt"

(** [Sqlite.finalize vm] finalizes vm explicitly, thus invalidating the
handle. The vm is finalized automatically when an error occurs or when
fetching is attempted on a query all of whose rows have been fetched.
[Sqlite.finalize] may be used to terminate query execution at will,
whenever needed. *)
external finalize    : vm -> unit                            = "caml_sqlite_vm_finalize"

(** [Sqlite.vm_rc vm] returns the status code of the last operation on vm. *)
external vm_rc   : vm -> rc = "caml_sqlite_vm_retcode"

(** [Sqlite.column_names vm] returns the column names of the query result.
If [Sqlite.compile] was called with a true third parameter, the column
names are available even after vm is finalized. *)
external column_names : vm -> string array = "caml_sqlite_column_names"

(** [Sqlite.column_types vm] returns the column types of query result.
If [Sqlite.compile] was called with a true third parameter, the column
types are available even after vm is finalized. *)
external column_types : vm -> string array = "caml_sqlite_column_types"

(* vim: set tw=72 : *)

