/* NVTV nv_tv -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This is open software protected by the GPL. See LICENSE for details.
 *
 * Header: Structures and defines for the BT and CH chip. This part could
 * eventually become a part of the XFree NV-Driver.
 * 
 */

#ifndef _NV_TVCHIP_H
#define _NV_TVCHIP_H

#include "xfree.h"

typedef enum {
  TV_SYSTEM_NONE    = -1,
  TV_SYSTEM_NTSC    = 0,
  TV_SYSTEM_NTSC_J  = 1, /* same as NTSC_60 */
  TV_SYSTEM_PAL     = 2, 
  TV_SYSTEM_PAL_60  = 3,
  TV_SYSTEM_PAL_N   = 4,
  TV_SYSTEM_PAL_NC  = 5,
  TV_SYSTEM_PAL_M   = 6,
  TV_SYSTEM_PAL_M60 = 7,
  TV_SYSTEM_PAL_X   = 8, /* Fake PAL System to correct color carriers,
                            useful at least in Sweden PAL-B */
} NVSystem;

typedef enum {
  NV_NO_CHIP   = 0,
  NV_CHRONTEL  = 1,
  NV_BROOKTREE = 2,
  NV_CONEXANT  = 3,
  NV_PHILIPS   = 4,
} NVTvChip;

/* -------- Brooktree -------- */

#define BT_FLAG1_NI_OUT		(1 << 0)
#define BT_FLAG1_SETUP		(1 << 1)
#define BT_FLAG1_625LINE	(1 << 2)
#define BT_FLAG1_VSYNC_DUR	(1 << 3)
#define BT_FLAG1_DIS_SCRESET	(1 << 4)
#define BT_FLAG1_PAL_MD		(1 << 5)
#define BT_FLAG1_ECLIP		(1 << 6)

#define BT_FLAG1_NTSC		(BT_FLAG1_VSYNC_DUR | BT_FLAG1_SETUP)
#define BT_FLAG1_NTSC_JAPAN	(BT_FLAG1_VSYNC_DUR)
#define BT_FLAG1_PAL_BDGHI	(BT_FLAG1_PAL_MD | BT_FLAG1_625LINE)
#define BT_FLAG1_PAL_N		(BT_FLAG1_VSYNC_DUR | BT_FLAG1_SETUP | \
				 BT_FLAG1_PAL_MD | BT_FLAG1_625LINE)
#define BT_FLAG1_PAL_M		(BT_FLAG1_VSYNC_DUR | BT_FLAG1_SETUP | \
				 BT_FLAG1_PAL_MD)
#define BT_FLAG1_PAL_BDGHI	(BT_FLAG1_PAL_MD | BT_FLAG1_625LINE)
#define BT_FLAG1_PAL_60		(BT_FLAG1_PAL_MD | BT_FLAG1_VSYNC_DUR)

#define BT_FLAG2_DIS_FFILT	(1 << 0)
#define BT_FLAG2_DIS_YFLPF	(1 << 1)
#define BT_FLAG2_DIS_GMSHY	(1 << 2)
#define BT_FLAG2_DIS_GMUSHY	(1 << 3)
#define BT_FLAG2_DIS_GMSHC	(1 << 4)
#define BT_FLAG2_DIS_GMUSHC	(1 << 5)
#define BT_FLAG2_DIS_CHROMA	(1 << 6)

#define BT_FLAG2_DIS_GM 	(BT_FLAG2_DIS_GMSHY | BT_FLAG2_DIS_GMUSHY \
			       | BT_FLAG2_DIS_GMSHC | BT_FLAG2_DIS_GMUSHC)

#define BT_FLAG4_DACDISA	(1 << 0)
#define BT_FLAG4_DACDISB	(1 << 1)
#define BT_FLAG4_DACDISC	(1 << 2)
#define BT_FLAG4_DAC		(BT_FLAG4_DACDISA | BT_FLAG4_DACDISB \
			       | BT_FLAG4_DACDISC)

#define BT_FLAG4_FBAS		(BT_FLAG4_DACDISB | BT_FLAG4_DACDISC)
#define BT_FLAG4_SVHS		(BT_FLAG4_DACDISA)
#define BT_FLAG4_CONVERT	(BT_FLAG4_DACDISA | BT_FLAG4_DACDISB)
#define BT_FLAG4_BOTH		0

typedef struct {
  int hsynoffset;
  int vsynoffset;
  int hsynwidth; /* don't confuse with hsync_width ! */
  int vsynwidth;
  int h_clko;
  int h_active;
  int hsync_width;
  int hburst_begin;
  int hburst_end;
  int h_blanko;
  int v_blanko;
  int v_activeo;
  int h_fract;
  int h_clki;
  int h_blanki;
  int v_blank_dly;
  int v_linesi;
  int v_blanki;
  int v_activei;
  int v_scale;
  int pll_fract;
  int pll_int;
  int sync_amp;
  int bst_amp;
  int mcr;
  int mcb;
  int my;
  unsigned long msc;
  int flags1;
  int flags2;
  int flags4; /* FIXME */
  int f_selc;
  int f_sely;
  int ycoring;
  int ccoring;
  int yattenuate;
  int cattenuate;
  int ylpf;
  int clpf;
  int out_muxa;
  int out_muxb;
  int out_muxc;
  int macro;
} NVBtRegs;

/* -------- Chrontel -------- */

#define CH_FLAG_DAC		1:0
#define CH_FLAG_DAC_MASK	3
#define CH_FLAG_FBAS		2
#define CH_FLAG_SVHS		0
#define CH_FLAG_BOTH		3

typedef struct {
  int dmr_ir;
  int dmr_vs;
  int dmr_sr;
  int ffr_fc;
  int ffr_fy;
  int ffr_ft;
  int vbw_flff;  /* flag */
  int vbw_cvbw;  /* flag */
  int vbw_cbw;
  int vbw_ypeak; /* flag */
  int vbw_ysv;
  int vbw_ycv;   /* flag */
  int dacg;      /* flag */
  int aciv;      /* flag */
  int civh;      
  int sav;
  int blr;
  int hpr;
  int vpr;
  int ce;
  int pll_m;
  int pll_n;
  int pllcap;    /* flag */
  unsigned long fsci;
#if 0 /* Test register; no documentation */
  int ylm; /* y multiplier ? */
  int clm; /* c multiplier ? */
#endif
  int flags;
  int mode;      /* for macrovision table */
  int macro;
} NVChRegs;

/* -------- CRT -------- */

#ifndef V_INTERLACE
#define V_INTERLACE 0x0010
#endif

#ifndef V_DBLSCAN
#define V_DBLSCAN 0x0020
#endif

#ifndef V_HSKEW
#define V_HSKEW 0x0200 
#endif

/* Flags that describe the mode (in NVCrtRegs.PrivFlags) */

#define NV_MODE_TVMODE		(1 << 0)  /* Is TV Mode */
#define NV_MODE_DUALVIEW	(1 << 1)  /* Has dualview */
#define NV_MODE_MACROVIS_OFF	(1 << 2)  /* Has macrovision disable */
#define NV_MODE_MACROVIS_ON	(1 << 3)  /* Has macrovision enable */

#define NV_MODE_MACROVIS	(NV_MODE_MACROVIS_OFF | NV_MODE_MACROVIS_ON)

/* Flags that affect the mode (in DisplayModeRec.PrivFlags) */

#define NV_PRIV_TVMODE		(1 << 0)  /* TV or monitor mode? */
#define NV_PRIV_DUALVIEW	(1 << 1)  /* Set single/dualview */
#define	NV_PRIV_BYPASS		(1 << 4)  /* Bypass settings */

/* We use our own crt structure in the frontend and database, and convert 
 * to DisplayModeRec, XF86VidModeModeLine, or XF86VidModeModeInfo as
 * required. The direct backend uses DisplayModeRec. The 'private'
 * structure is the same in all cases, and defined in nv_type.h. 
 */

typedef struct {
  int HDisplay;	/* horizontal timing */
  int HSyncStart;
  int HSyncEnd;
  int HTotal;
  int VDisplay;	/* vertical timing */
  int VSyncStart;
  int VSyncEnd;
  int VTotal;
  int Flags;
  int PrivFlags;
} NVCrtRegs;

/* -------- Common -------- */

typedef union {
  NVBtRegs bt;
  NVChRegs ch;
} NVTvRegs;

typedef enum {
  NV_TV_UNKNOWN = 0,
  NV_TV_OFF     = 1,
  NV_TV_BARS    = 2,
  NV_TV_ON      = 3
} NVTvState;

typedef enum {
  CONNECT_AUTO    = -2,
  CONNECT_NONE    = -1,
  CONNECT_FBAS    = 0,
  CONNECT_SVHS    = 1,
  CONNECT_BOTH    = 2,
  CONNECT_CONVERT = 3, /* FBAS on S-VHS Y-line, for converter */
} NVConnect;

typedef struct {
  int tv_hoffset, tv_voffset;
  int mon_hoffset, mon_voffset;
  int brightness; /* 0-200 % */
  int color;      /* 0-200 % */
  int contrast;   /* 0-200 % */
  int saturation; /* 0-200 % */
  int sharpness;  /* 0-100 % */
  int flicker;    /* 0-100 % */
  int bandwidth;  /* 0-200 % */
  int monochrome;  /* flag */
  int macrovision; /* flag */
  NVConnect connector; 
  /* FIXME: Treat flags somewhere else? */
} NVSettings;

typedef struct {
  char magic[4];
  NVTvChip chip;
  NVTvRegs tv;
} NVPrivate;

typedef struct {
  NVSystem	system;
  int		res_x;
  int		res_y;
  char		*size;
  double	hoc;
  double	voc;
  NVCrtRegs	*crt;   
  NVTvRegs	*tv; 
} NVMode;

#endif
