/* NVTV Brooktree chip data -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Data tables and routines for the Brooktree (and Conexant) chips
 *
 */

#include <stddef.h>
#include "data_bt.h"

/* -------- BT -------- NTSC -------- */

/* -------- 640x480 -------- */

NVBtRegs bt_ntsc_small_a = { /* over comp 13.79 13.58 % (std 0) */
  hsynoffset   : 0, 
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1792,
  h_active     : 640,
  hsync_width  : 132,
  hburst_begin : 150,
  hburst_end   : 96,
  h_blanko     : 381,
  v_blanko     : 34,
  v_activeo    : 212,
  h_fract      : 0,
  h_clki       : 784,
  h_blanki     : 126,
  v_blank_dly  : 0,
  v_linesi     : 600,
  v_blanki     : 75,
  v_activei    : 480,
  v_scale      : 5266,
  pll_fract    : 34830,
  pll_int      : 12,
  sync_amp     : 229,
  bst_amp      : 118,
  mcr          : 121,
  mcb          : 68,
  my           : 133,
  msc          : 545259520,
  flags1       : BT_FLAG1_NTSC,
};

NVCrtRegs bt_crt_ntsc_small_a = { 
  HDisplay    : 640,
  HSyncStart  : 664-8,
  HSyncEnd    : 696-8,
  HTotal      : 744,
  VDisplay    : 480,
  VSyncStart  : 524,
  VSyncEnd    : 526,
  VTotal      : 600,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};

NVBtRegs bt_ntsc_normal_a = { /* over comp 10.59 08.23 % (status voc=08.64) */
  hsynoffset   : 12, /* HSkew: 12/2 */
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1728,
  h_active     : 640,
  hsync_width  : 128,
  hburst_begin : 144,
  hburst_end   : 88,
  h_blanko     : 345,
  v_blanko     : 28,
  v_activeo    : 224,
  h_fract      : 0,
  h_clki       : 800,
  h_blanki     : 140,
  v_blank_dly  : 0,
  v_linesi     : 567,
  v_blanki     : 58,
  v_activei    : 480,
  v_scale      : 4751,
  pll_fract    : 5499,
  pll_int      : 12,
  sync_amp     : 229,
  bst_amp      : 118,
  mcr          : 121,
  mcb          : 68,
  my           : 133,
  msc          : 565454317,
  flags1       : BT_FLAG1_NTSC,
};

NVCrtRegs bt_crt_ntsc_normal_a = { 
  HDisplay    : 640,
  HSyncStart  : 680-8,
  HSyncEnd    : 736-8,
  HTotal      : 752,
  VDisplay    : 480,
  VSyncStart  : 506,
  VSyncEnd    : 508,
  VTotal      : 562,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};

NVBtRegs bt_ntsc_huge_a = { /* over comp 02.46 01.23 % */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1584,
  h_active     : 640,
  hsync_width  : 118,
  hburst_begin : 132,
  hburst_end   : 70,
  h_blanko     : 263,
  v_blanko     : 19,
  v_activeo    : 242,
  h_fract      : 0,
  h_clki       : 792,
  h_blanki     : 133,
  v_blank_dly  : 0,
  v_linesi     : 525,
  v_blanki     : 36,
  v_activei    : 480,
  v_scale      : 4096,
  pll_fract    : 5041,
  pll_int      : 11,
  sync_amp     : 229,
  bst_amp      : 118,
  mcr          : 122,
  mcb          : 68,
  my           : 133,
  msc          : 616859255,
  flags1       : BT_FLAG1_NTSC,
};

NVCrtRegs bt_crt_ntsc_huge_a = { 
  HDisplay    : 640,
  HSyncStart  : 672-8,
  HSyncEnd    : 696-8,
  HTotal      : 760,
  VDisplay    : 480,
  VSyncStart  : 486,
  VSyncEnd    : 488,
  VTotal      : 521,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};

/* -------- 800x600 -------- */

NVBtRegs bt_ntsc_small_b = { /* over comp 21.62 11.52 % (std 2) */
  hsynoffset   : -8, /* HSkew: -8/2 */
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2464,
  h_active     : 800,
  hsync_width  : 182,
  hburst_begin : 206,
  hburst_end   : 180,
  h_blanko     : 597,
  v_blanko     : 32,
  v_activeo    : 216,
  h_fract      : 0,
  h_clki       : 880,
  h_blanki     : 66,
  v_blank_dly  : 0,
  v_linesi     : 735,
  v_blanki     : 86,
  v_activei    : 600,
  v_scale      : 7373,
  pll_fract    : 15124,
  pll_int      : 17,
  sync_amp     : 229,
  bst_amp      : 116,
  mcr          : 119,
  mcb          : 67,
  my           : 133,
  msc          : 396552378,
  flags1       : BT_FLAG1_NTSC,
};

NVCrtRegs bt_crt_ntsc_small_b = { 
  HDisplay    : 800,
  HSyncStart  : 808-8,
  HSyncEnd    : 832-8,
  HTotal      : 848,
  VDisplay    : 600,
  VSyncStart  : 630,
  VSyncEnd    : 632,
  VTotal      : 718,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};

NVBtRegs bt_ntsc_normal_b = { /* over comp 11.90 05.35 % */
  hsynoffset   : -62, /* HSkew: -62/2 */
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2192,
  h_active     : 800,
  hsync_width  : 162,
  hburst_begin : 182,
  hburst_end   : 146,
  h_blanko     : 445, /* tool: 480 */
  v_blanko     : 24,
  v_activeo    : 232,
  h_fract      : 0,
  h_clki       : 840,
  h_blanki     : 26,
  v_blank_dly  : 0,
  v_linesi     : 685,
  v_blanki     : 60,
  v_activei    : 600,
  v_scale      : 6593,
  pll_fract    : 21540,
  pll_int      : 15,
  sync_amp     : 229,
  bst_amp      : 117,
  mcr          : 120,
  mcb          : 67,
  my           : 133,
  msc          : 445759608,
  flags1       : BT_FLAG1_NTSC,
};

NVCrtRegs bt_crt_ntsc_normal_b = { /* FIXME: SyncStart ok ?? */
  HDisplay    : 800,
  HSyncStart  : 800-8,
  HSyncEnd    : 824-8,
  HTotal      : 840,
  VDisplay    : 600,
  VSyncStart  : 630,
  VSyncEnd    : 632,
  VTotal      : 690,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};

NVBtRegs bt_ntsc_huge_b = { /* over comp 07.15 00.004 % */
  hsynoffset   : -60, /* HSkew: -60/2, */
  vsynoffset   : 0,
  hsynwidth    : 1,
  vsynwidth    : 2,
  h_clko       : 2080,
  h_active     : 800,
  hsync_width  : 154,
  hburst_begin : 174,
  hburst_end   : 132,
  h_blanko     : 383, /* calc: 383, tool: 400 */
  v_blanko     : 18,
  v_activeo    : 244,
  h_fract      : 0,
  h_clki       : 840,
  h_blanki     : 26,
  v_blank_dly  : 0,
  v_linesi     : 650,
  v_blanki     : 42,
  v_activei    : 600,
  v_scale      : 6046,
  pll_fract    : 35747,
  pll_int      : 14,
  sync_amp     : 229,
  bst_amp      : 117,
  mcr          : 120,
  mcb          : 67,
  my           : 133,
  msc          : 469762048,
  flags1       : BT_FLAG1_NTSC,
};

NVCrtRegs bt_crt_ntsc_huge_b = { 
  HDisplay    : 800,
  HSyncStart  : 808-8,
  HSyncEnd    : 824-8,
  HTotal      : 848,
  VDisplay    : 600,
  VSyncStart  : 608,
  VSyncEnd    : 610,
  VTotal      : 650,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};

/* -------- 720x480 -------- */

NVBtRegs bt_ntsc_normal_c = { /* Overscan 08.762 x 18.107 */
  hsynoffset   : -4,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2048,
  h_active     : 720,
  hsync_width  : 152,
  hburst_begin : 170,
  hburst_end   : 128,
  h_blanko     : 445,
  v_blanko     : 34,
  v_activeo    : 212,
  h_fract      : 0,
  h_clki       : 896,
  h_blanki     : 154,
  v_blank_dly  : 0,
  v_linesi     : 600,
  v_blanki     : 75,
  v_activei    : 480,
  v_scale      : 5266,
  pll_fract    : 21081,
  pll_int      : 14,
  sync_amp     : 229,
  bst_amp      : 117,
  mcr          : 120,
  mcb          : 68,
  my           : 133,
  msc          : 477102080,
  flags1       : BT_FLAG1_NTSC,
};

NVCrtRegs bt_crt_ntsc_normal_c = { 
  HDisplay    : 720,
  HSyncStart  : 796-8,
  HSyncEnd    : 936-8,
  HTotal      : 896,
  VDisplay    : 480,
  VSyncStart  : 527,
  VSyncEnd    : 557,
  VTotal      : 600,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};

/* -------- 720x480 DVD -------- */

/* -------- BT -------- PAL -------- */

/* -------- 640x480 -------- */

NVBtRegs bt_pal_def_a = {
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1888,
  h_active     : 640,
  hsync_width  : 138,
  hburst_begin : 166,
  hburst_end   : 104,
  h_blanko     : 449,
  v_blanko     : 46,
  v_activeo    : 242,
  h_fract      : 0,
  h_clki       : 944,
  h_blanki     : 266,
  v_blank_dly  : 0,
  v_linesi     : 625,
  v_blanki     : 90,
  v_activei    : 480,
  v_scale      : 4096,
  pll_fract    : 7282,
  pll_int      : 13,
  sync_amp     : 240,
  bst_amp      : 88,
  mcr          : 129,
  mcb          : 73,
  my           : 140,
  msc          : 645499916,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

NVBtRegs bt_pal_small_a = { /* over comp 16.56 16.67 % (std 1) */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1888,
  h_active     : 640,
  hsync_width  : 138,
  hburst_begin : 166,
  hburst_end   : 104,
  h_blanko     : 449,
  v_blanko     : 46,
  v_activeo    : 242,
  h_fract      : 0,
  h_clki       : 944,
  h_blanki     : 266,
  v_blank_dly  : 0,
  v_linesi     : 625,
  v_blanki     : 90,
  v_activei    : 480,
  v_scale      : 4096,
  pll_fract    : 7282,
  pll_int      : 13,
  sync_amp     : 240,
  bst_amp      : 88,
  mcr          : 129,
  mcb          : 73,
  my           : 140,
  msc          : 645499916,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

NVCrtRegs bt_crt_pal_small_a = { 
  HDisplay    : 640,
  HSyncStart  : 744-8,
  HSyncEnd    : 888-8,
  HTotal      : 944,
  VDisplay    : 480,
  VSyncStart  : 532,
  VSyncEnd    : 534,
  VTotal      : 625,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};

NVBtRegs bt_pal_normal_a = { /* over comp 12.87 07.64 % */
  hsynoffset   : 0,  /* 208 */
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 1808,
  h_active     : 640,
  hsync_width  : 132,
  hburst_begin : 158,
  hburst_end   : 94,
  h_blanko     : 403, /* 416 */
  v_blanko     : 33,
  v_activeo    : 268,
  h_fract      : 0,
  h_clki       : 1000,
  h_blanki     : 315,
  v_blank_dly  : 0,
  v_linesi     : 565,
  v_blanki     : 57,
  v_activei    : 480,
  v_scale      : 3310,
  pll_fract    : 36409,
  pll_int      : 12,
  sync_amp     : 240,
  bst_amp      : 88,
  mcr          : 130,
  mcb          : 73,
  my           : 140,
  msc          : 674061860,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

NVCrtRegs bt_crt_pal_normal_a = { /* no mon sync !! */
  HDisplay    : 640,
  HSyncStart  : 768-8,
  HSyncEnd    : 880-8,
  HTotal      : 1000,
  VDisplay    : 480,
  VSyncStart  : 520,
  VSyncEnd    : 522,
  VTotal      : 565,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_MACROVIS_OFF, /* no DUALVIEW */
  Flags       : 0,
};

NVBtRegs bt_pal_huge_a = { /* over comp 06.22 00.69 % */
  hsynoffset   : 0, /* 208, */
  vsynoffset   : 0,
  hsynwidth    : 1,
  vsynwidth    : 1,
  h_clko       : 1680,
  h_active     : 640,
  hsync_width  : 124,
  hburst_begin : 146, /* 148, */
  hburst_end   : 78,
  h_blanko     : 331, /* 344, */
  v_blanko     : 23,
  v_activeo    : 288,
  h_fract      : 0,
  h_clki       : 1000,
  h_blanki     : 299, /* 312, */
  v_blank_dly  : 0,
  v_linesi     : 525,
  v_blanki     : 36,
  v_activei    : 480,
  v_scale      : 2785,
  pll_fract    : 43691,
  pll_int      : 11,
  sync_amp     : 240,
  bst_amp      : 89,
  mcr          : 131,
  mcb          : 73,
  my           : 140,
  msc          : 725418954,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

NVCrtRegs bt_crt_pal_huge_a = { /* no mon sync !! */
  HDisplay    : 640,
  HSyncStart  : 736-8,
  HSyncEnd    : 872-8,
  HTotal      : 976,
  VDisplay    : 480,
  VSyncStart  : 483,
  VSyncEnd    : 485,
  VTotal      : 520,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_MACROVIS_OFF, /* no NV_MODE_DUALVIEW */
  Flags       : 0,
};

/* -------- 800x600 -------- */

NVBtRegs bt_pal_def_b = {
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2304,
  h_active     : 800,
  hsync_width  : 170,
  hburst_begin : 202,
  hburst_end   : 154,
  h_blanko     : 525,
  v_blanko     : 41,
  v_activeo    : 252,
  h_fract      : 0,
  h_clki       : 960,
  h_blanki     : 140,
  v_blank_dly  : 0,
  v_linesi     : 750,
  v_blanki     : 95,
  v_activei    : 600,
  v_scale      : 5734,
  pll_fract    : 0,
  pll_int      : 16,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 528951320,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

NVBtRegs bt_pal_small_b = {  /* over comp 14.53 13.19 % (std 3) */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2304,
  h_active     : 800,
  hsync_width  : 170,
  hburst_begin : 202,
  hburst_end   : 154,
  h_blanko     : 525, /* 530 */
  v_blanko     : 41,
  v_activeo    : 252,
  h_fract      : 0,
  h_clki       : 960,
  h_blanki     : 140, /* 141 */
  v_blank_dly  : 0,
  v_linesi     : 750,
  v_blanki     : 95,
  v_activei    : 600,
  v_scale      : 5734, 
  pll_fract    : 0,
  pll_int      : 16,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 528951320,  
  flags1       : BT_FLAG1_PAL_BDGHI,
};

NVCrtRegs bt_crt_pal_small_b = {
  HDisplay    : 800,
  HSyncStart  : 824-8, /* 880, */
  HSyncEnd    : 880-8, /* 912, */
  HTotal      : 912,
  VDisplay    : 600,
  VSyncStart  : 658,
  VSyncEnd    : 660,
  VTotal      : 750,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};

NVBtRegs bt_pal_normal_b = { /* over comp 10.81 05.56 % */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2208, 
  h_active     : 800,
  hsync_width  : 162,  
  hburst_begin : 194,  
  hburst_end   : 142, 
  h_blanko     : 471,  /* 478, */
  v_blanko     : 30,   
  v_activeo    : 274,  
  h_fract      : 0,
  h_clki       : 1000, 
  h_blanki     : 175,  /* 100, 140 */
  v_blank_dly  : 0,
  v_linesi     : 690,  
  v_blanki     : 64,   
  v_activei    : 600,
  v_scale      : 4948, 
  pll_fract    : 21845, 
  pll_int      : 15,    
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 551949204, 
  flags1       : BT_FLAG1_PAL_BDGHI,
};

NVCrtRegs bt_crt_pal_normal_b = {
  HDisplay    : 800,
  HSyncStart  : 840-8,
  HSyncEnd    : 864-8,
  HTotal      : 952,
  VDisplay    : 600,
  VSyncStart  : 625,
  VSyncEnd    : 627,
  VTotal      : 687,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};

NVBtRegs bt_pal_large_b = { /* over comp 07.461 x 02.083 */
  hsynoffset   : -8,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2128,
  h_active     : 800,
  hsync_width  : 156,
  hburst_begin : 186,
  hburst_end   : 134,
  h_blanko     : 425,
  v_blanko     : 25,
  v_activeo    : 284,
  h_fract      : 0,
  h_clki       : 1000,
  h_blanki     : 190,
  v_blank_dly  : 0,
  v_linesi     : 665,
  v_blanki     : 51,
  v_activei    : 600,
  v_scale      : 4620,
  pll_fract    : 50972,
  pll_int      : 14,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 572699174,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

NVCrtRegs bt_crt_pal_large_b = {
  HDisplay    : 800,
  HSyncStart  : 848-8,
  HSyncEnd    : 928-8,
  HTotal      : 1000,
  VDisplay    : 600,
  VSyncStart  : 615,
  VSyncEnd    : 617,
  VTotal      : 665,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};


NVBtRegs bt_pal_huge_b = { /* over comp 00.039 00.00 % (no calc'd mode) */
  hsynoffset   : 14, /* HSkew: 14/2, */
  vsynoffset   : 0,
  hsynwidth    : 1,
  vsynwidth    : 1,
  h_clko       : 1970,  
  h_active     : 800,	
  hsync_width  : 144,   
  hburst_begin : 172,   
  hburst_end   : 114,   
  h_blanko     : 303,   
  v_blanko     : 22,    
  v_activeo    : 309,   
  h_fract      : 0,  	
  h_clki       : 945,   
  h_blanki     : 101,   
  v_blank_dly  : 0,	
  v_linesi     : 652,   
  v_blanki     : 133,   
  v_activei    : 600,	
  v_scale      : 4438,  
  pll_fract    : 44478, 
  pll_int      : 13,    
  sync_amp     : 240,	
  bst_amp      : 88,    
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 618715949, 
  flags1       : BT_FLAG1_PAL_BDGHI,
};

NVCrtRegs bt_crt_pal_huge_b = {
  HDisplay    : 800,
  HSyncStart  : 944-8,
  HSyncEnd    : 968-8,
  HTotal      : 1000,
  VDisplay    : 600,
  VSyncStart  : 653,
  VSyncEnd    : 657,
  VTotal      : 696,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_MACROVIS_OFF, /* no NV_MODE_DUALVIEW */
  Flags       : 0,
};


/* -------- 768x576 -------- */

/* 768 x 576 (720x576 scaled by xine) */

NVBtRegs bt_pal_small_c = { /* hoc = 13.122 voc = 07.986 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2176,
  h_active     : 768,
  hsync_width  : 160,
  hburst_begin : 190,
  hburst_end   : 138,
  h_blanko     : 485,
  v_blanko     : 34,
  v_activeo    : 267,
  h_fract      : 0,
  h_clki       : 1000,
  h_blanki     : 203,
  v_blank_dly  : 0,
  v_linesi     : 680,
  v_blanki     : 71,
  v_activei    : 576,
  v_scale      : 4817,
  pll_fract    : 7282,
  pll_int      : 15,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 560066112,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

NVCrtRegs bt_crt_pal_small_c = { 
  HDisplay    : 768,
  HSyncStart  : 824-8,
  HSyncEnd    : 896-8,
  HTotal      : 976,
  VDisplay    : 576,
  VSyncStart  : 610,
  VSyncEnd    : 621,
  VTotal      : 680,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};


NVBtRegs bt_pal_normal_c = { /* hoc =  09.806 voc = 04.514 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2096,
  h_active     : 768,
  hsync_width  : 154,
  hburst_begin : 184,
  hburst_end   : 130,
  h_blanko     : 439,
  v_blanko     : 29,
  v_activeo    : 277,
  h_fract      : 0,
  h_clki       : 1000,
  h_blanki     : 203,
  v_blank_dly  : 0,
  v_linesi     : 655,
  v_blanki     : 58,
  v_activei    : 576,
  v_scale      : 4489,
  pll_fract    : 36409,
  pll_int      : 14,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 581442673,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

NVCrtRegs bt_crt_pal_normal_c = { 
  HDisplay    : 768,
  HSyncStart  : 824-8,
  HSyncEnd    : 896-8,
  HTotal      : 968,
  VDisplay    : 576,
  VSyncStart  : 599,
  VSyncEnd    : 601,
  VTotal      : 655,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};

NVBtRegs bt_pal_normal1_c = { /* hoc = 11.163 voc = 05.903 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2128,
  h_active     : 768,
  hsync_width  : 156,
  hburst_begin : 186,
  hburst_end   : 134,
  h_blanko     : 457,
  v_blanko     : 31,
  v_activeo    : 273,
  h_fract      : 0,
  h_clki       : 1000,
  h_blanki     : 203,
  v_blank_dly  : 0,
  v_linesi     : 665,
  v_blanki     : 63,
  v_activei    : 576,
  v_scale      : 4620,
  pll_fract    : 50972,
  pll_int      : 14,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 572699174,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

NVCrtRegs bt_crt_pal_normal1_c = { 
  HDisplay    : 768,
  HSyncStart  : 824-8,
  HSyncEnd    : 896-8,
  HTotal      : 968,
  VDisplay    : 576,
  VSyncStart  : 600,
  VSyncEnd    : 602,
  VTotal      : 665,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};

/* -------- 800x576 -------- */

/* 800 x 576 (for 1024x576 widescreen scaled by xine) */

NVBtRegs bt_pal_normal_d = { /* hoc = 08.83 voc = 07.29 */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 2,
  vsynwidth    : 1,
  h_clko       : 2160,
  h_active     : 800,
  hsync_width  : 158,
  hburst_begin : 188,
  hburst_end   : 136,
  h_blanko     : 443,
  v_blanko     : 33,
  v_activeo    : 269,
  h_fract      : 0,
  h_clki       : 1000,
  h_blanki     : 175,
  v_blank_dly  : 0,
  v_linesi     : 675,
  v_blanki     : 69,
  v_activei    : 576,
  v_scale      : 4751,
  pll_fract    : 0,
  pll_int      : 15,
  sync_amp     : 240,
  bst_amp      : 87,
  mcr          : 128,
  mcb          : 72,
  my           : 140,
  msc          : 564214742,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

NVCrtRegs bt_crt_pal_normal_d = { 
  HDisplay    : 800,
  HSyncStart  : 840-8,
  HSyncEnd    : 888-8,
  HTotal      : 960,
  VDisplay    : 576,
  VSyncStart  : 608,
  VSyncEnd    : 610,
  VTotal      : 675,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};

/* -------- 720x576 DVD -------- */

/* Note: Locks up the system, even with Beos CRT values.
   FIXME: Check how Beos converts crtc vals to registers */

NVBtRegs bt_pal_dvd_a = { /* NV/Beos DVD */
  hsynoffset   : 0,
  vsynoffset   : 0,
  hsynwidth    : 0,
  vsynwidth    : 0,
  h_clko       : 1776,
  h_active     : 720,
  hsync_width  : 130,
  hburst_begin : 156,
  hburst_end   : 90,
  h_blanko     : 305,
  v_blanko     : 22,
  v_activeo    : 290,
  h_fract      : 0,
  h_clki       : 888,
  h_blanki     : 147,
  v_blank_dly  : 0,
  v_linesi     : 625,
  v_blanki     : 42,
  v_activei    : 576,
  v_scale      : 4096,
  pll_fract    : 21845,
  pll_int      : 12,
  sync_amp     : 240,
  bst_amp      : 89,
  mcr          : 130,
  mcb          : 73,
  my           : 140,
  msc          : 686207118,
  flags1       : BT_FLAG1_PAL_BDGHI,
};

NVCrtRegs bt_crt_pal_dvd_a = { 
  HDisplay    : 720,
  HSyncStart  : 744-8, /* Beos: 744 */
  HSyncEnd    : 880-8, /* Beos: 744+140 -- not a multiple of 8 */
  HTotal      : 888,
  VDisplay    : 576,
  VSyncStart  : 579, /* Beos: 579 */
  VSyncEnd    : 621, /* Beos: 579+42=621 */
  VTotal      : 625,
  PrivFlags   : NV_MODE_TVMODE | NV_MODE_DUALVIEW | NV_MODE_MACROVIS_OFF,
  Flags       : 0,
};

/* -------- All modes -------- */

/*

Modes:

NTSC 640x480 S tv ok,  mon ok, oc ok, calcable (std 0)
NTSC 640x480 N tv ok,  mon ok, oc ok, calcable
NTSC 640x480 H tv ok,  mon ok, oc ok, calcable

NTSC 800x600 S tv ok,  mon ok, oc ok, calcable (std 2)
NTSC 800x600 N tv ok,  mon ok, oc ok, calcable 
NTSC 800x600 H tv ok*, mon ok, oc ok, calcable

PAL  640x480 S tv ok, mon ok, oc ok, calcable (std 1)
PAL  640x480 N tv ok, mon NO, oc ok, calcable
PAL  640x480 H tv  ?, mon  ?, oc ok, calcable

PAL  800x600 S tv ok, mon ok, oc ok, calcable (std 3)
PAL  800x600 N tv ok, mon ok, oc ok, calcable
PAL  800x600 H tv ok, mon NO, oc ?, not calcable

PAL  768x576 S tv ok, mon ok, oc ok, calcable
PAL  768x576 N tv ok, mon ok, oc ok, calcable

* needs several attempts, monitor takes some time syncing.

(Huge modes check with HWcursor = False)

Bad modes:

PAL  800x600 H distorted dualview
PAL  640x480 N no dualview
PAL  640x480 H no dualview
NTSC 800x600 N tv left border wrong (sync?)

*/

NVMode bt_modes [] = {
  {TV_SYSTEM_NTSC, 640, 480, "Small", 13.79, 13.58, 
   &bt_crt_ntsc_small_a, (NVTvRegs *) &bt_ntsc_small_a},
  {TV_SYSTEM_NTSC, 640, 480, "Normal", 10.59, 08.23,
   &bt_crt_ntsc_normal_a, (NVTvRegs *) &bt_ntsc_normal_a},
  {TV_SYSTEM_NTSC, 640, 480, "Huge", 02.46, 01.23, 
   &bt_crt_ntsc_huge_a, (NVTvRegs *) &bt_ntsc_huge_a},

  {TV_SYSTEM_NTSC, 800, 600, "Small", 21.62, 11.52, 
   &bt_crt_ntsc_small_b, (NVTvRegs *) &bt_ntsc_small_b},
  {TV_SYSTEM_NTSC, 800, 600, "Normal", 11.90, 05.35, 
   &bt_crt_ntsc_normal_b, (NVTvRegs *) &bt_ntsc_normal_b},
  {TV_SYSTEM_NTSC, 800, 600, "Huge", 07.15, 00.004,
   &bt_crt_ntsc_huge_b, (NVTvRegs *) &bt_ntsc_huge_b},

  {TV_SYSTEM_NTSC, 720, 480, "Normal",  08.762, 18.107,
   &bt_crt_ntsc_normal_c, (NVTvRegs *) &bt_ntsc_normal_c},

  {TV_SYSTEM_PAL, 640, 480, "Small", 16.56, 16.67,
   &bt_crt_pal_small_a, (NVTvRegs *) &bt_pal_small_a},
  {TV_SYSTEM_PAL, 640, 480, "Normal", 12.87, 07.64, 
   &bt_crt_pal_normal_a, (NVTvRegs *) &bt_pal_normal_a},
  {TV_SYSTEM_PAL, 640, 480, "Huge", 06.22, 00.69,
   &bt_crt_pal_huge_a, (NVTvRegs *) &bt_pal_huge_a},

  {TV_SYSTEM_PAL, 800, 600, "Small", 14.53, 13.19,
   &bt_crt_pal_small_b, (NVTvRegs *) &bt_pal_small_b},
  {TV_SYSTEM_PAL, 800, 600, "Normal", 10.81, 05.56,
   &bt_crt_pal_normal_b, (NVTvRegs *) &bt_pal_normal_b},
  {TV_SYSTEM_PAL, 800, 600, "Large", 07.461, 02.083,
   &bt_crt_pal_large_b, (NVTvRegs *) &bt_pal_large_b},
  {TV_SYSTEM_PAL, 800, 600, "Huge", 00.039, 00.00,
   &bt_crt_pal_huge_b, (NVTvRegs *) &bt_pal_huge_b},

  {TV_SYSTEM_PAL, 768, 576, "Small", 13.122, 07.986,
   &bt_crt_pal_small_c, (NVTvRegs *) &bt_pal_small_c},
  {TV_SYSTEM_PAL, 768, 576, "Normal",  09.806, 04.514,
   &bt_crt_pal_normal_c, (NVTvRegs *) &bt_pal_normal_c},

  {TV_SYSTEM_PAL, 800, 576, "Normal", 08.83, 07.29,
   &bt_crt_pal_normal_d, (NVTvRegs *) &bt_pal_normal_d},

#if 0 /* non-overlaced DVD lock system up */
  {TV_SYSTEM_PAL, 720, 576, "DVD", 00.000, 00.000,
   &bt_crt_pal_dvd_a, (NVTvRegs *) &bt_pal_dvd_a},
#endif

  {TV_SYSTEM_NONE, 0, 0, NULL, 0.0, 0.0, NULL, NULL}
};

/* -------- -------- */

static int clamp_attenuate[8] = {0, 10, 25, 50, 70, 80, 90, 100};
static int clamp_flicker[5]   = {0, 25, 50, 75, 100};
static int clamp_bandwidth[6] = {0, 50, 75, 100, 150, 200};

void data_init_bt (NVSystem system, NVTvRegs *r)
{
  r->bt.ylpf = 3;
  r->bt.clpf = 0;
  r->bt.flags2 = BT_FLAG2_DIS_GM;
  r->bt.flags4 = 0;
  r->bt.f_selc = 2;
  r->bt.f_sely = 2;
  r->bt.out_muxa = 0;
  r->bt.out_muxb = 2;
  r->bt.out_muxc = 1;
  switch (system)
  {
    case TV_SYSTEM_NONE:
    case TV_SYSTEM_NTSC:
    case TV_SYSTEM_NTSC_J: 
      r->bt.macro = 0; break;
    case TV_SYSTEM_PAL:
    case TV_SYSTEM_PAL_X:
    case TV_SYSTEM_PAL_60: 
    case TV_SYSTEM_PAL_N:  
    case TV_SYSTEM_PAL_NC: 
    case TV_SYSTEM_PAL_M:  
    case TV_SYSTEM_PAL_M60:  
      r->bt.macro = 4; break;
  }
}

void data_default_bt (NVSettings *s)
{
  s->tv_hoffset = s->mon_hoffset = 0;
  s->tv_voffset = s->mon_voffset = 0;
  s->brightness = 100;
  s->color = 100;
  s->contrast = 100;
  s->saturation = 100;
  s->flicker = 75;
  s->sharpness = 50;
  s->bandwidth = 100;
  s->macrovision = 0;
  s->monochrome = 0;
}

void data_clamp_bt (NVSettings *s)
{
  if (s->tv_hoffset <= -100) s->tv_hoffset = -100;
  if (s->tv_hoffset >=  100) s->tv_hoffset =  100;
  if (s->tv_voffset <= -100) s->tv_voffset = -100;
  if (s->tv_voffset >=  100) s->tv_voffset =  100;
  if (s->mon_hoffset <= -100) s->mon_hoffset = -100;
  if (s->mon_hoffset >=  100) s->mon_hoffset =  100;
  if (s->mon_voffset <= -100) s->mon_voffset = -100;
  if (s->mon_voffset >=  100) s->mon_voffset =  100;
  s->tv_voffset = 0; /* FIXME. Clamp to 0 for now, util 'VTotal magic' */
  if (s->brightness <   0) s->brightness = 0;
  if (s->brightness > 200) s->brightness = 200;
  if (s->color <   0) s->color = 0;
  if (s->color > 200) s->color = 200;
  s->contrast   = data_clamp (s->contrast, 8, clamp_attenuate); 
  s->saturation = data_clamp (s->saturation, 8, clamp_attenuate); 
  s->sharpness = 50;
  s->flicker = data_clamp (s->flicker, 5, clamp_flicker); 
  s->bandwidth = data_clamp (s->bandwidth, 6, clamp_bandwidth); 
  /* FIXME: Operate on mode flags */
}

void data_setup_bt (NVSettings *s, NVCrtRegs *c, NVTvRegs *r)
{
  float f;

  /* FIXME: Pure monitor movement, and tv vert movement */
  /* If HTotal = h_clki - 3*8, the monitor position is the same as at hclk_i.
     However, the image moves, so compensate with hsyncoffset. Try to keep
     hsynoffset always positive */
  /* Can't make HTotal too small, or monitor image will disappear... 
     So sometimes you have to use negative hsynoffset ... */
#if 0 /* TODO */
  if (c->HTotal > r->bt.h_clki - 3*8 && c->HTotal < r->bt.h_clki) {
    /* compensate for bad HTotal value */
    r->bt.hsynoffset += c->HTotal - (r->bt.h_clki - 3*8);
    c->HTotal = r->bt.h_clki - 3*8;
  }
#endif
  c->VSyncStart -= s->mon_voffset;
  c->VSyncEnd   -= s->mon_voffset;
  c->HSyncStart -= s->mon_hoffset * 8;
  c->HSyncEnd   -= s->mon_hoffset * 8;
  r->bt.hsynoffset += s->tv_hoffset * 2;
  f = 0.01 * s->brightness * r->bt.my;
  if (f < 0.0) f = 0.0;
  if (f > 255.0) f = 255.0;
  r->bt.my = (int) f;
  f = 0.01 * s->color * r->bt.mcr;
  if (f < 0.0) f = 0.0;
  if (f > 255.0) f = 255.0;
  r->bt.mcr = (int) f;
  f = 0.01 * s->color * r->bt.mcb;
  if (f < 0.0) f = 0.0;
  if (f > 255.0) f = 255.0;
  r->bt.mcb = (int) f;
  r->bt.yattenuate = 7 - data_pick (s->contrast, 8, clamp_attenuate); 
  r->bt.cattenuate = 7 - data_pick (s->saturation, 8, clamp_attenuate); 
  r->bt.flags2 &= ~BT_FLAG2_DIS_FFILT;
  switch (data_pick (s->flicker, 5, clamp_flicker)) {
    case 0: r->bt.flags2 |= BT_FLAG2_DIS_FFILT; break;
    case 1: r->bt.f_sely = 1; r->bt.f_selc = 1; break;
    case 2: r->bt.f_sely = 2; r->bt.f_selc = 1; break;
    case 3: r->bt.f_sely = 3; r->bt.f_selc = 2; break;
    case 4: r->bt.f_sely = 0; r->bt.f_selc = 4; break;
  }
  /* There is a special case here: For v_scale values less then 4096,
     f_sely = 3 gives bad results, so set it to 2 in this case */
  if (r->bt.v_scale < 4096 && r->bt.f_sely == 3) {
    r->bt.f_sely = 2;
  }
  switch (data_pick (s->bandwidth, 6, clamp_bandwidth)) {
    case 0: r->bt.ylpf = 2; r->bt.clpf = 2; break;
    case 1: r->bt.ylpf = 2; r->bt.clpf = 3; break;
    case 2: r->bt.ylpf = 2; r->bt.clpf = 0; break;
    case 3: r->bt.ylpf = 3; r->bt.clpf = 0; break;
    case 4: r->bt.ylpf = 1; r->bt.clpf = 0; break;
    case 5: r->bt.ylpf = 0; r->bt.clpf = 0; break;
  }
  r->bt.macro = (r->bt.macro & ~0x3) | ((s->macrovision) ? 1 : 0);
  if (s->monochrome) {
    r->bt.flags2 &= ~BT_FLAG2_DIS_CHROMA;
  } else {
    r->bt.flags2 |= BT_FLAG2_DIS_CHROMA;
  }
  /* FIXME: Do Chroma in nv_tv */
  r->bt.out_muxa = 0;
  r->bt.out_muxb = 2;
  r->bt.out_muxc = 1;
  r->bt.flags4 &= ~BT_FLAG4_DAC;
  switch (s->connector) 
  {
    case CONNECT_FBAS: 
      r->bt.flags4 |= BT_FLAG4_FBAS; 
      break;
    case CONNECT_SVHS: 
      r->bt.flags4 |= BT_FLAG4_SVHS;
      break;
    case CONNECT_CONVERT:
      r->bt.flags4 |= BT_FLAG4_CONVERT;
      r->bt.out_muxa = 2;
      r->bt.out_muxc = 0;
      break;
    case CONNECT_BOTH: 
    default:
      r->bt.flags4 |= BT_FLAG4_BOTH;
      break;
  }
  /* FIXME dualview;     */
}

#if 0 /* FIXME */
NVPtr gui_bt_setup_nv (NVPtr pNv, GuiSettings *set) 
{
  int th, tv, mh, mv;
  float f;

  th = set->tv_hoffset;
  tv = set->tv_voffset;
  mh = set->mon_hoffset;
  mv = set->mon_voffset;
  pNv->CrtRegs.VSyncStart -= mv;
  pNv->CrtRegs.VSyncEnd   -= mv;
  pNv->CrtRegs.VBlankEnd  -= mv - tv;
  pNv->CrtRegs.VTotal     -= mv - tv;
  pNv->CrtRegs.HSyncStart -= mh * 8;
  pNv->CrtRegs.HSyncEnd   -= mh * 8;
  pNv->CrtRegs.HBlankEnd  -= mh * 8;
  pNv->CrtRegs.HTotal     -= mh * 8;
  pNv->TvRegs.bt.hsynoffset += pNv->CrtRegs.HSkew;
  pNv->TvRegs.bt.hsynoffset += th;
}
#endif

DataFunc bt_func = {
  modes: bt_modes,
  init: data_init_bt, 
  defaults: data_default_bt,
  setup: data_setup_bt, 
  clamp: data_clamp_bt
};

/*
Setup:

Brightness Y_OFF in Bt871; use MY in Bt869 (which is really Contrast)
Contrast   Y_ATTENUATE (8 values)
Saturation C_ATTENUATE
Sharpness  PKFIL_SEL (4 values) in Bt871; none in Bt869
Flicker    5 values
Bandwidth  low pass filters: YLPF CLPF

Monochrome Disable chroma DCHROMA
           Coring = clamp value below threshold


Brightness 0-200 %
Color      0-200 %
Contrast   0-100 % (8)
Saturation 0-100 % (8)
Sharpness  0-100 % (4)
Flicker    0-100 % (5)
Bandwidth  0-200 % (6)

Dualview    flag
Macrovision flag
Monochrome  flag

X/Y Pos horiz instead of vertical (smaller field)

Macrovision mode CH regs/BT Flags
Interlace, doublescan -> CRT Regs

*/
