/* upscommon.h - prototypes for upscommon.c

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <sys/termios.h>	/* for speed_t */
#include "config.h"
#include "shared.h"

#ifdef HAVE_MMAP
#include <sys/types.h>
#include <sys/mman.h>
#endif

/* called by alarm signal when the port open fails */
void openfail(int sig);

/* lock the serial port using flock or uu_lock */
void lockport (int upsfd, const char *port);

/* try to open the port at a given speed */
void open_serial_simple(const char *port, speed_t speed, int flags);
void open_serial(const char *port, speed_t speed);

/* put a notice in the syslog */
void notice (const char *msg);

/* function for erasing "timeout"-conditions */
void nolongertimeout(void);

/* alarm signal handler for when serial reads time out */
void timeout(int sig);

/* receive up to buflen bytes from the ups into buf until endchar is read */
/* any characters received that match members of ignchars are discarded */
int upsrecv (char *buf, int buflen, char endchar, const char *ignchars);

/* read buflen chars and store in buf */
int upsrecvchars (char *buf, int buflen);

/* send a byte to the ups */
int upssendchar (char data);
/* send a string to the ups */
int upssend(const char *fmt, ...);

/* flush any pending input from the ups, with reporting */
void upsflushin (int expected, int debugit, const char *ignchars);

/* install the data that the ups returns for 'reqchar' in the info
   array position that has type infotype - endchar/ignchars pass to upsrecv */
void installinfo (int infotype, char reqchar, char endchar, const char *ignchars);

/* store data into the array */
void setinfo(int infotype, const char *fmt, ...);

/* set the flags on an existing member of the array */
void setflag (int infotype, int newflags);

/* return the data in the info array with type infotype */
char *getdata (int infotype);

/* write data to state file - effectively a no-op for shared memory mode */
void writeinfo(void);

/* create info array of size numinfo from shared memory or normal */
/* shmok == 1 to allow shared memory, 0 to deny */
void create_info(int numinfo, int shmok);

/* call this after calling create_info and then doing all your add/setinfos */
void info_ready(void);

/* create SysV IPC message queue if possible */
void createmsgq(void);

/* get a message from the queue and parse it - returns 1 if successful */
int getupsmsg(int wait);

/* place a message into the queue */
void sendupsmsg (const msgtype *buf);

/* lightweight way to send just an OK/ERR response to a command */
void msgreply (int reptype);

/* add another type to the info array */
void addinfo (int type, const char *value, int flags, int auxdata);

/* add a new ENUM info entry and do other related housekeeping */
void addenum (int basetype, const char *value);

/* close the state fd - used by the exiting parent */
void close_statefd(void);

/* structure for funcs that get called by msg parse routine */
struct ups_handler
{
	void	(*setvar)(int, int, char *);
	void	(*instcmd)(int, int, char *);
};

extern itype *info;
extern int infomax;
extern struct ups_handler upsh;
extern int upsfd;
extern int flag_timeoutfailure;
extern unsigned int upssend_delay;

extern int experimental_driver;

#define SIZE_OF_STATEFN 512
extern char statefn[SIZE_OF_STATEFN];

void rtrim(char *in, char sep);
void unlockport(int upsfd, const char *port);

#ifdef HAVE_SHMAT

/*
 * The use of SHM_R and SHM_W is not portable. Neither Posix nor
 * SUSv2 specify it. Solaris and Linux have them (as well as newer
 * versions of NetBSD) but either do not document them (Solaris) or
 * explicitly depreceate their use (NetBSD).
 */
#ifdef SHM_R
#define IPC_MODE IPC_CREAT|SHM_R|SHM_W|S_IRGRP|S_IWGRP
#else
#define IPC_MODE IPC_CREAT|0600|S_IRGRP|S_IWGRP
#endif

#endif	/* HAVE_SHMAT */

/* Linux doesn't have MAP_NOSYNC, but it's the default, so... */

#ifndef MAP_NOSYNC
#define MAP_NOSYNC 0
#endif

/* Linux libc5 doesn't have MAP_FAILED */

#ifndef MAP_FAILED
#define MAP_FAILED (void *) -1 
#endif

