/* upsset - CGI program to manage read/write variables

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include "common.h"

#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>

#include "upsfetch.h"
#include "cgilib.h"
#include "parseconf.h"

	int	fd;
	char	*ups, *host, *username, *password, *function, *upscommand;

	/* set once the MAGIC_ENABLE_STRING is found in the upsset.conf */
	int	magic_string_set = 0;

/* see the stock upsset.conf for the whole rant on what this is */
#define MAGIC_ENABLE_STRING "I_HAVE_SECURED_MY_CGI_DIRECTORY"

#define HARD_UPSVAR_LIMIT_NUM	64
#define HARD_UPSVAR_LIMIT_LEN	256

typedef struct {
	char	*var;
	char	*value;
	void	*next;
}	uvtype;

	uvtype	*firstuv = NULL;

void parsearg(char *var, char *value)
{
	char	*ptr;
	uvtype	*last, *tmp = NULL;
	static	int upsvc = 0;

	/* store variables from a SET command for the later commit */
	if (!strncmp(var, "UPSVAR_", 7)) {

		/* if someone bombs us with variables, stop at some point */
		if (upsvc > HARD_UPSVAR_LIMIT_NUM)
			return;

		/* same idea: throw out anything that's much too long */
		if (strlen(value) > HARD_UPSVAR_LIMIT_LEN)
			return;

		ptr = strchr(var, '_');

		if (!ptr)		/* sanity check */
			return;

		ptr++;

		tmp = last = firstuv;
		while (tmp) {
			last = tmp;
			tmp = tmp->next;
		}

		tmp = xmalloc(sizeof(uvtype));
		tmp->var = xstrdup(ptr);
		tmp->value = xstrdup(value);
		tmp->next = NULL;

		if (last)
			last->next = tmp;
		else
			firstuv = tmp;

		upsvc++;

		return;
	}

	if (!strcmp(var, "username")) {
		if (username)
			free(username);

		username = xstrdup(value);
	}

	if (!strcmp(var, "password")) {
		if (password)
			free(password);

		password = xstrdup(value);
	}

	if (!strcmp(var, "function")) {
		if (function)
			free(function);

		function = xstrdup(value);
	}

	if (!strcmp(var, "ups")) {
		if (ups)
			free(ups);

		ups = xstrdup(value);
	}

	if (!strcmp(var, "upscommand")) {
		if (upscommand)
			free(upscommand);

		upscommand = xstrdup(value);
	}
}

void do_enum(char *varname)
{
	char	out[SMALLBUF], temp[SMALLBUF], *val, *sel, *ptr;

	snprintf(out, sizeof(out), "ENUM %s\n", varname);	
	
	if (upssendraw(fd, out) < 0) {
		fprintf(stderr, "ENUM %s failed: %s\n", varname,
			upsstrerror(upserror));
		printf("Unable to get ENUM list\n");
		return;
	}

	if (upsreadraw(fd, temp, sizeof(temp)) < 0) {
		fprintf(stderr, "Read ENUM %s failed: %s\n", varname,
			upsstrerror(upserror));
		printf("Unable to get ENUM list\n");
		return;
	}

	if (strncmp(temp, "ENUM", 4) != 0) {
		printf("Bogus reply from server for %s\n", varname);
		return;
	}

	printf("<SELECT NAME=\"UPSVAR_%s\">\n", varname);

	/* 'OPTION "<val>" [SELECTED]' || 'END' */
	if (upsreadraw(fd, temp, sizeof(temp)) < 0) {
		printf("<!-- Error reading OPTION list -->\n");

		fprintf(stderr, "Can't read OPTION list: %s\n",
			upsstrerror(upserror));
		printf("</SELECT>\n");
		return;
	}

	while (strcmp(temp, "END") != 0) {

		/* split into value and selected */
		val = strchr(temp, '"');

		/* sanity check - no " means a protocol problem */
		if (!val) {
			printf("<!-- Error: no quote detected in ENUM response -->\n");
			printf("</SELECT>\n");
			return;
		}

		/* move off the " */
		val++;

		sel = strchr(val, ' ');
		if (sel)
			sel++;
		ptr = strchr(val, '"');

		/* same idea */
		if (!ptr) {
			printf("<!-- Error: no quote detected in ENUM response -->\n");
			printf("</SELECT>\n");
			return;
		}

		val [ptr - val] = '\0';
			
		printf("<OPTION VALUE=\"%s\" ", val);
			
		if ((sel != NULL) && (!strcmp(sel, "SELECTED")))
			printf("SELECTED");

		printf(">%s</OPTION>\n", val);	

		if (upsreadraw(fd, temp, sizeof(temp)) < 0) {
			printf("<!-- Error reading OPTION list -->\n");

			fprintf(stderr, "Can't read OPTION list: %s\n",
				upsstrerror(upserror));
			printf("</SELECT>\n");
			return;
		}
	}

	printf("</SELECT>\n");
}

void do_string(char *upsname, char *varname, int typelen)
{
	char	temp[SMALLBUF];

	getupsvarfd(fd, upsname, varname, temp, sizeof(temp));
	printf("<INPUT TYPE=\"TEXT\" NAME=\"UPSVAR_%s\" VALUE=\"%s\" SIZE=\"%i\">\n",
	       varname, temp, typelen);
}

void do_header(char *title)
{
	printf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\"\n");
	printf("	\"http://www.w3.org/TR/REC-html40/loose.dtd\">\n");
	printf("<HTML>\n");
	printf("<HEAD><TITLE>upsset: %s</TITLE></HEAD>\n", title);

	printf("<BODY BGCOLOR=\"#FFFFFF\" TEXT=\"#000000\" LINK=\"#0000EE\" VLINK=\"#551A8B\">\n"); 

	printf("<TABLE BGCOLOR=\"#50A0A0\" ALIGN=\"CENTER\">\n");
	printf("<TR><TD>\n");
}

void start_table()
{
	printf("<TABLE CELLPADDING=\"5\" CELLSPACING=\"0\" ALIGN=\"CENTER\" WIDTH=\"100%%\">\n");
	printf("<TR><TH COLSPAN=2 BGCOLOR=\"#60B0B0\">\n");
	printf("<FONT SIZE=\"+2\">Network UPS Tools upsset %s</FONT>\n", 
	        UPS_VERSION);
	printf("</TH></TR>\n");
}

/* propagate login details across pages - no cookies here! */
void do_hidden(char *next)
{
	printf("<INPUT TYPE=\"HIDDEN\" NAME=\"username\" VALUE=\"%s\">\n",
	       username);
	printf("<INPUT TYPE=\"HIDDEN\" NAME=\"password\" VALUE=\"%s\">\n",
	       password);

	if (next)
		printf("<INPUT TYPE=\"HIDDEN\" NAME=\"function\" VALUE=\"%s\">\n", 
		       next);
}

/* generate SELECT chooser from hosts.conf entries */
void upslist_arg(int numargs, char **arg)
{
	if (numargs < 3)
		return;

	/* MONITOR <ups> <description> */
	if (!strcmp(arg[0], "MONITOR")) {
		printf("<OPTION VALUE=\"%s\"", arg[1]);

		if (ups)
			if (!strcmp(ups, arg[1]))
				printf("SELECTED");

		printf(">%s</OPTION>\n", arg[2]);
	}
}

void upslist_err(int linenum, char *errtext)
{
	if (linenum == 0)
		fprintf(stderr, "hosts.conf: %s\n", errtext);
	else
		fprintf(stderr, "hosts.conf line %d: %s\n", linenum, errtext);
}

/* this defaults to wherever we are now, ups and function-wise */
void do_pickups(char *currfunc)
{
	char	hostfn[SMALLBUF];
	int	ret;

	snprintf(hostfn, sizeof(hostfn), "%s/hosts.conf", CONFPATH);

	printf("<FORM METHOD=\"POST\" ACTION=\"upsset.cgi\">\n");

	printf("Select UPS and function:\n<BR>\n");
	printf("<SELECT NAME=\"ups\">\n");

	ret = parseconf(hostfn, upslist_arg, upslist_err);

	printf("</SELECT>\n");

	if (ret != 0) {
		printf("Error: hosts.conf unavailable\n");
		printf("</FORM>\n");
		return;
	}

	printf("<SELECT NAME=\"function\">\n");

	/* FUTURE */
	/*	printf("<OPTION VALUE=\"showstatus\">Status</OPTION>\n");  */

	/* TODO: clean this up */

	if (!strcmp(currfunc, "showsettings"))
		printf("<OPTION VALUE=\"showsettings\" SELECTED>Settings</OPTION>\n");
	else
		printf("<OPTION VALUE=\"showsettings\">Settings</OPTION>\n");

	if (!strcmp(currfunc, "showcmds"))
		printf("<OPTION VALUE=\"showcmds\" SELECTED>Commands</OPTION>\n");
	else
		printf("<OPTION VALUE=\"showcmds\">Commands</OPTION>\n");

	printf("</SELECT>\n");
	do_hidden(NULL);

	printf("<INPUT TYPE=\"SUBMIT\" VALUE=\"View\">\n");
	printf("</FORM>\n");
}

void errorpage(char *title, const char *msg, char *func)
{
	do_header(title);
	start_table();
	printf("<TR><TH COLSPAN=2 BGCOLOR=\"#60B0B0\">\n");
	printf("Error: %s\n", msg);
	printf("</TH></TR>\n");

	printf("<TR><TD ALIGN=\"CENTER\" COLSPAN=2>\n");
	do_pickups(func);
	printf("</TD></TR>\n");

	printf("</TABLE>\n");
	printf("</TD></TR></TABLE>\n");
	printf("</BODY></HTML>\n");
	exit(0);
}

void loginscreen(void)	/* HTML 4.0 transitional checked */
{
	do_header("Login");
	printf("<FORM METHOD=\"POST\" ACTION=\"upsset.cgi\">\n");
	start_table();

	printf("<TR BGCOLOR=\"#60B0B0\">\n");
	printf("<TH>Username</TH>\n");
	printf("<TD><INPUT TYPE=\"TEXT\" NAME=\"username\" VALUE=\"\"></TD>\n");
	printf("</TR>\n");

	printf("<TR BGCOLOR=\"#60B0B0\">\n");
	printf("<TH>Password</TH>\n");
	printf("<TD><INPUT TYPE=\"PASSWORD\" NAME=\"password\" VALUE=\"\"></TD>\n");
	printf("</TR>\n");

	printf("<TR><TD COLSPAN=2 ALIGN=\"CENTER\">\n");
	printf("<INPUT TYPE=\"HIDDEN\" NAME=\"function\" VALUE=\"pickups\">\n");
	printf("<INPUT TYPE=\"SUBMIT\" VALUE=\"Login\">\n");
	printf("<INPUT TYPE=\"RESET\" VALUE=\"Reset fields\">\n");
	printf("</TD></TR></TABLE>\n");
	printf("</FORM>\n");
	printf("</TD></TR></TABLE>\n");
	printf("</BODY></HTML>\n");
	exit(0);
}

/* try to connect to upsd - generate an error page if it fails */
void upsd_connect(void)
{
	fd = upsconnect(host);

	if (fd < 0) {
		char	ebuf[SMALLBUF];

		snprintf(ebuf, sizeof(ebuf), 
		         "Unable to connect to %s - %s", host,
		         upsstrerror(upserror));
		errorpage("Connect failure", ebuf, "showsettings");
	}
}

/* generate a list of instant commands */
void showcmds(void)
{
	int	ret;
	char	*desc, *upstmp, *ptr, *upsname, cmds[SMALLBUF], 
		temp[SMALLBUF], *tmp;

	if (!checkhost(ups, &desc))
		errorpage("Access denied",
		          "Access to that host is not authorized",
		          "showsettings");

	upstmp = xstrdup(ups);
	ptr = strchr(upstmp, '@');
	if (ptr != NULL) {
		ptr[0] = 0;
		upsname = upstmp;
		host = ptr + 1;
	}
	else {
		upsname = NULL;
		host = upstmp;
	}

	upsd_connect();

	if (upsname != NULL) {
		snprintf(temp, sizeof(temp), "LISTINSTCMD %s\n", upsname);
		ret = upssendraw(fd, temp);
	}
	else
		ret = upssendraw(fd, "LISTINSTCMD\n");

	if (ret < 0) {
		printf("Can't list instant commands\n");

		fprintf(stderr, "LISTINSTCMD failed: %s\n", 
			upsstrerror(upserror));
		return;
	}
	
	if (upsreadraw(fd, cmds, sizeof(cmds)) < 0) {
		char	ebuf[SMALLBUF];

		snprintf(ebuf, sizeof(ebuf),
		         "Unable to get list of commands - %s\n",
		         upsstrerror(upserror));
		errorpage("Protocol failure", ebuf, "showcmds");
	}

	if (strncmp("INSTCMDS", cmds, 7))
		errorpage("Invalid response from server",
		          "Server did not return INSTCMDS reply", "showcmds");

	/* "INSTCMDS" or "INSTCMDS @ups" --> no instant commands available */

	ptr = strchr(cmds, ' ');
	if (!ptr)
		errorpage("No instant commands supported",
		          "This UPS doesn't support any instant commands.",
		          "showcmds");

	ptr++;

	/* possibly skip '@ups' */
	if (ptr[0] == '@') {
		ptr = strchr(ptr, ' ');
		if (!ptr)
			errorpage("No instant commands supported",
			          "This UPS doesn't support any instant commands.",
			          "showcmds");
		ptr++;
	}

	do_header("Instant commands");
	printf("<FORM ACTION=\"upsset.cgi\" METHOD=\"POST\">\n");
	start_table();

	/* include the description from checkhost() if present */
	if (desc)
		printf("<TR><TH BGCOLOR=\"#60B0B0\"COLSPAN=2>%s</TH></TR>\n",
		       desc);

	printf("<TR BGCOLOR=\"#60B0B0\" ALIGN=\"CENTER\">\n");
	printf("<TD>Instant commands</TD>\n");

	printf("<TD>\n");
	printf("<SELECT NAME=\"upscommand\">\n");

	/* provide a dummy do-nothing default choice */
	printf("<OPTION VALUE=\"\" SELECTED></OPTION>\n");

	while (ptr) {
		char	buf[SMALLBUF], *q, *desc;

		tmp = ptr;
		ptr = strchr(tmp, ' ');
		if (ptr) {
			*ptr = '\0';
			ptr++;
		}

		/* get a description for this command from the server */
		snprintf(buf, sizeof(buf), "INSTCMDDESC %s\n", tmp);

		/* if description retrieval fails, go with the cmd name */
		if (upssendraw(fd, buf) < 0) {
			printf("<OPTION VALUE=\"%s\">%s</OPTION>\n", tmp, tmp);
			continue;
		}

		if (upsreadraw(fd, buf, sizeof(buf)) < 0) {
			printf("<OPTION VALUE=\"%s\">%s</OPTION>\n", tmp, tmp);
			continue;
		}

		q = strchr(buf, '"');

		if (!q)	{	/* invalid/missing description, use cmd name */
			printf("<OPTION VALUE=\"%s\">%s</OPTION>\n", tmp, tmp);
			continue;
		}

		desc = ++q;
		
		q = strchr(desc, '"');
		if (!q)	{	/* missing second quote - don't use it */
			printf("<OPTION VALUE=\"%s\">%s</OPTION>\n", tmp, tmp);
			continue;
		}

		*q = '\0';
		printf("<OPTION VALUE=\"%s\">%s</OPTION>\n", tmp, desc);
	}

	printf("</SELECT>\n");
	printf("</TD></TR>\n");

	printf("<TR BGCOLOR=\"#60B0B0\">\n");
	printf("<TD COLSPAN=\"2\" ALIGN=\"CENTER\">\n");
	do_hidden("docmd");
	printf("<INPUT TYPE=\"HIDDEN\" NAME=\"ups\" VALUE=\"%s\">\n", ups);
	printf("<INPUT TYPE=\"SUBMIT\" VALUE=\"Issue command\">\n");
	printf("<INPUT TYPE=\"RESET\" VALUE=\"Reset\">\n");
	printf("</TD></TR>\n");
	printf("</TABLE>\n");
	printf("</FORM>\n");

	printf("<TR><TD ALIGN=\"CENTER\">\n");
	do_pickups("showcmds");
	printf("</TD></TR>\n");

	printf("</TABLE>\n");
	printf("</BODY></HTML>\n");

	exit(0);
}

/* take a command from the client and send it to upsd */
void docmd(void)
{
	char	*desc, *upstmp, *ptr, *upsname, buf[SMALLBUF];

	if (!checkhost(ups, &desc))
		errorpage("Access denied",
		          "Access to that host is not authorized",
		          "showsettings");

	/* the user is messing with us */
	if (!upscommand)	
		errorpage("Form error", "No instant command selected", "showcmds");

	/* (l)user took the default blank option */
	if (strlen(upscommand) == 0)
		errorpage("Form error", "No instant command selected", "showcmds");

	upstmp = xstrdup(ups);
	ptr = strchr(upstmp, '@');
	if (ptr != NULL) {
		ptr[0] = 0;
		upsname = upstmp;
		host = ptr + 1;
	}
	else {
		upsname = NULL;
		host = upstmp;
	}

	upsd_connect();

	/* log into server - it makes the authentication decision now */
	snprintf(buf, sizeof(buf), "USERNAME %s\n", username);

	if (upssendraw(fd, buf) < 0) {
		printf("Can't set username\n");

		fprintf(stderr, "Can't set username: %s\n",
			upsstrerror(upserror));
		return;
	}

	if (upsreadraw(fd, buf, sizeof(buf)) < 0) {
		/* test for old upsd that doesn't do USERNAME */
		if (upserror == UPSF_UNKNOWNCMD) {
			errorpage("Protocol mismatch",
			          "upsd version too old - USERNAME not supported",
				  "showcmds");
		}

		errorpage("Can't set username", upsstrerror(upserror),
			   "showcmds");
	}

	snprintf(buf, sizeof(buf), "PASSWORD %s\n", password);

	if (upssendraw(fd, buf) < 0)
		errorpage("Password set failed", upsstrerror(upserror),
		          "showcmds");

	if (upsreadraw(fd, buf, sizeof(buf)) < 0)
		errorpage("Password set failed", upsstrerror(upserror),
		          "showcmds");

	/* send command, get response, show it */
	if (upsname)
		snprintf(buf, sizeof(buf), "INSTCMD %s@%s\n", upscommand,
		         upsname);
	else
		snprintf(buf, sizeof(buf), "INSTCMD %s\n", upscommand);

	if (upssendraw(fd, buf) < 0) {
		do_header("Error while issuing command");

		start_table();    

		printf("<TR><TD>Error sending command: %s\n</TD></TR>",
			upsstrerror(upserror));   

		printf("<TR><TD ALIGN=\"CENTER\" COLSPAN=2>\n");
		do_pickups("showcmds");
		printf("</TD></TR>\n");

		printf("</TABLE>\n");

		printf("</TD></TR></TABLE>\n");
		printf("</BODY></HTML>\n");
		exit(0);
	}

	if (upsreadraw(fd, buf, sizeof(buf)) < 0) {
		do_header("Error while reading command response");

		start_table();    

		printf("<TR><TD>Error reading command response: %s\n</TD></TR>",
			upsstrerror(upserror));   

		printf("<TR><TD ALIGN=\"CENTER\" COLSPAN=2>\n");
		do_pickups("showcmds");
		printf("</TD></TR>\n");

		printf("</TABLE>\n");

		printf("</TD></TR></TABLE>\n");
		printf("</BODY></HTML>\n");
		exit(0);
	}

	do_header("Issuing commands");
	start_table();

	printf("<TR><TD><PRE>\n");
	printf("Sending command: %s\n", upscommand);
	printf("Response: %s\n", buf);
	printf("</PRE></TD></TR>\n");

	printf("<TR><TD ALIGN=\"CENTER\" COLSPAN=2>\n");
	do_pickups("showcmds");
	printf("</TD></TR>\n");

	printf("</TABLE>\n");
	printf("</TD></TR></TABLE>\n");
	printf("</BODY></HTML>\n");

	exit(0);
}

/* dump a table of all read/write variables and their current values */
void showsettings(void)	/* HTML 4.0/transitional checked (main path) */
{
	char	vars[SMALLBUF], temp[SMALLBUF], out[SMALLBUF], *v, 
		*upsname, *upstmp, *type, *ptr, *desc, *vptr;
	int	typelen, ret;

	if (!checkhost(ups, &desc))
		errorpage("Access denied",
		          "Access to that host is not authorized",
		          "showsettings");

	upstmp = xstrdup(ups);
	ptr = strchr(upstmp, '@');
	if (ptr != NULL) {
		ptr[0] = 0;
		upsname = upstmp;
		host = ptr + 1;
	}
	else {
		upsname = NULL;
		host = upstmp;
	}

	upsd_connect();

	if (upsname != NULL) {
		snprintf(temp, sizeof(temp), "LISTRW %s\n", upsname);
		ret = upssendraw(fd, temp);
	}
	else
		ret = upssendraw(fd, "LISTRW\n");

	if (ret < 0) {
		printf("Can't get RW variable list\n");

		fprintf(stderr, "LISTRW failed: %s\n",
			upsstrerror(upserror));
		return;
	}
	
	if (upsreadraw(fd, vars, sizeof(vars)) < 0) {
		char	ebuf[SMALLBUF];
		snprintf(ebuf, sizeof(ebuf), 
		         "Unable to get variable list - %s\n",
		         upsstrerror(upserror));
		errorpage("Server protocol error", ebuf, "showsettings");
	}

	/* "RW" by itself means no RW variables exist */
	if ((strlen(vars) == 2) &&  (!strncmp(vars, "RW", 2))) {
		errorpage("No read/write variables", 
		          "This UPS has no read/write variables.",
		          "showsettings");
	}

	ptr = strchr(vars, ' ');

	/* no spaces = seriously broken string */
	if (!ptr) {		
		errorpage("Server protocol error", 
		          "Unable to get variable list - broken string", 
		          "showsettings");
	}

	/* skip over 'RW' */
	*ptr++ = '\0';
	v = ptr;

	/* no spaces found this time = no variables for default UPS */
	if (!v) {	
		errorpage("No read/write variables", 
		          "This UPS has no read/write variables.",
		          "showsettings");
	}

	/* v may be pointing at '@upsname' right now, if so, skip it */
	if (v[0] == '@') {
		ptr = strchr(v, ' ');
		if (ptr)
			*ptr++ = '\0';
		v = ptr;
	}

	/* one more time: deal with the no r/w variables supported case */
	if (!ptr) {	
		errorpage("No read/write variables", 
		          "This UPS has no read/write variables.",
		          "showsettings");
	}

	do_header("Current settings");
	printf("<FORM ACTION=\"upsset.cgi\" METHOD=\"POST\">\n");
	start_table();

	/* include the description from checkhost() if present */
	if (desc)
		printf("<TR><TH BGCOLOR=\"#60B0B0\"COLSPAN=2>%s</TH></TR>\n",
		        desc);

	printf("<TR BGCOLOR=\"#60B0B0\">\n");
	printf("<TH>Setting</TH>\n");
	printf("<TH>Value</TH></TR>\n");

	while (v != NULL) {
		vptr = strchr(v, ' ');
		if (vptr)
			*vptr++ = '\0';

		/* get description */
		snprintf(out, sizeof(out), "VARDESC %s\n", v);
		if (upssendraw(fd, out) < 0) {
			printf("Can't get VARDESC\n");

			fprintf(stderr, "Can't get VARDESC of %s: %s\n",
				v, upsstrerror(upserror));
			return;
		}

		if (upsreadraw(fd, temp, sizeof(temp)) < 0) {
			printf("Can't get VARDESC\n");

			fprintf(stderr, "Can't get VARDESC of %s: %s\n",
				v, upsstrerror(upserror));
			return;
		}

		printf("<TR BGCOLOR=\"#60B0B0\" ALIGN=\"CENTER\">\n");

		/* strip off leading/trailing quotes */
		ptr = strchr(temp, '"');
		if (ptr != NULL) {
			char	*ptr2;

			ptr++;
			ptr2 = strchr(ptr, '"');

			if (ptr2)
				*ptr2 = '\0';
			else {
				printf("Unbalanced quotes from server\n");
				exit(0);
			}
		}

		printf("<TD>%s</TD>\n", ptr);

		/* now get variable type */
		snprintf(out, sizeof(out), "VARTYPE %s\n", v);

		if (upssendraw(fd, out) < 0) {
			printf("Can't get VARTYPE\n");

			fprintf(stderr, "Can't get VARTYPE %s: %s\n",
				v, upsstrerror(upserror));
			return;
		}

		if (upsreadraw(fd, temp, sizeof(temp)) < 0) {
			printf("Can't get VARTYPE\n");

			fprintf(stderr, "Can't get VARTYPE %s: %s\n",
				v, upsstrerror(upserror));
			return;
		}

		type = NULL;
		typelen = 0;

		ptr = strchr(temp, ' ');
		if (ptr) {
			*ptr++ = '\0';
			type = ptr;
			ptr = strchr(type, ' ');
			if (ptr) {
				*ptr++ = '\0';
				typelen = strtol(ptr, (char **) NULL, 10);
			}
		}

		if (type) {
			printf("<TD>\n");

			if (!strcmp(type, "ENUM"))
				do_enum(v);

			if (!strcmp(type, "STRING"))
				do_string(upsname, v, typelen);
		}
		else
			printf("Broken type received from server\n");

		printf("</TD></TR>\n");

		v = vptr;		
	}

	printf("<TR BGCOLOR=\"#60B0B0\">\n");
	printf("<TD COLSPAN=\"2\" ALIGN=\"CENTER\">\n");
	do_hidden("savesettings");
	printf("<INPUT TYPE=\"HIDDEN\" NAME=\"ups\" VALUE=\"%s\">\n", ups);
	printf("<INPUT TYPE=\"SUBMIT\" VALUE=\"Save changes\">\n");
	printf("<INPUT TYPE=\"RESET\" VALUE=\"Reset\">\n");
	printf("</TD></TR>\n");
	printf("</TABLE>\n");
	printf("</FORM>\n");

	printf("<TR><TD ALIGN=\"CENTER\">\n");
	do_pickups("showsettings");
	printf("</TD></TR>\n");

	printf("</TABLE>\n");
	printf("</BODY></HTML>\n");

	exit(0);
}

/* turn a form submission of settings into SET commands for upsd */
void savesettings(void)	/* HTML 4.0/transitional checked (main loop) */
{
	char	*desc, *upstmp, *ptr, *upsname, buf[SMALLBUF];
	uvtype	*upsvar;

	if (!checkhost(ups, &desc)) 
		errorpage("Access denied",
		          "Access to that host is not authorized",
		          "showsettings");

	upstmp = xstrdup(ups);
	ptr = strchr(upstmp, '@');
	if (ptr != NULL) {
		ptr[0] = 0;
		upsname = upstmp;
		host = ptr + 1;
	}
	else {
		upsname = NULL;
		host = upstmp;
	}

	upsd_connect();

	upsvar = firstuv;

	snprintf(buf, sizeof(buf), "USERNAME %s\n", username);

	if (upssendraw(fd, buf) < 0) {
		printf("Send username failed\n");

		fprintf(stderr, "Can't set username: %s\n",
			upsstrerror(upserror));
		return;
	}

	if (upsreadraw(fd, buf, sizeof(buf)) < 0) {
		/* test for old upsd that doesn't do USERNAME */
		if (upserror == UPSF_UNKNOWNCMD) {
			errorpage("Protocol mismatch",
			          "upsd version too old - USERNAME not supported",
				  "showsettings");
		}

		errorpage("Can't set username", upsstrerror(upserror),
			   "showcmds");
		closeupsfd(fd);
		exit(0);
	}

	snprintf(buf, sizeof(buf), "PASSWORD %s\n", password);
	
	if (upssendraw(fd, buf) < 0) 
		errorpage("Password set failed", upsstrerror(upserror),
		          "showcmds");

	if (upsreadraw(fd, buf, sizeof(buf)) < 0)
		errorpage("Password set failed", upsstrerror(upserror),
		          "showcmds");

	do_header("Saving settings");
	start_table();

	printf("<TR><TD><PRE>\n");

	while (upsvar) {
		getupsvarfd(fd, upsname, upsvar->var, buf, sizeof(buf));

		if (strcmp(upsvar->value, buf)) {
			printf("set %s to %s (was %s)\n", upsvar->var, 
			        upsvar->value, buf);

			if (upsname)
				snprintf(buf, sizeof(buf), "SET %s@%s %s\n", 
				         upsvar->var, upsname, upsvar->value);
			else
				snprintf(buf, sizeof(buf), "SET %s %s\n", 
				         upsvar->var, upsvar->value);

			if (upssendraw(fd, buf) < 0) {
				printf("Error: SET failed: %s\n",
				       upsstrerror(upserror));
			} else {
				if (upsreadraw(fd, buf, sizeof(buf)) < 0) {
					printf("Error: SET failed: %s\n",
					       upsstrerror(upserror));
				} else {
					printf("response: %s\n", buf);
				}
			}
		}

		upsvar = upsvar->next;
	}

	printf("</PRE></TD></TR>\n");

	printf("<TR><TD ALIGN=\"CENTER\" COLSPAN=2>\n");
	do_pickups("showsettings");
	printf("</TD></TR>\n");

	printf("</TABLE>\n");
	printf("</TD></TR></TABLE>\n");
	printf("</BODY></HTML>\n");

	exit(0);
}

void initial_pickups (void)
{
	do_header("Select a UPS");
	start_table();

	printf("<TR><TD ALIGN=\"CENTER\" COLSPAN=2>\n");
	do_pickups("");
	printf("</TD></TR>\n");

	printf("</TABLE>\n");
	printf("</TD></TR></TABLE>\n");
	printf("</BODY></HTML>\n");
	exit(0);
}

void upsset_conf_arg(int numargs, char **arg)
{
	if (numargs < 1)
		return;

	if (!strcmp(arg[0], MAGIC_ENABLE_STRING))
		magic_string_set = 1;
}

void upsset_conf_err(int linenum, char *errtext)
{
	if (linenum == 0)
		fprintf(stderr, "upsset.conf: %s\n", errtext);
	else
		fprintf(stderr, "upsset.conf line %d: %s\n", linenum, errtext);
} 

/* see if the user has confirmed their cgi directory's secure state */
static void check_conf(void)
{
	char	fn[SMALLBUF];
	int	ret;

	snprintf(fn, sizeof(fn), "%s/upsset.conf", CONFPATH);

	ret = parseconf(fn, upsset_conf_arg, upsset_conf_err);

	if (ret != 0) {
		printf("<PRE>\n");
		printf("Error: Can't open upsset.conf to verify security settings.\n");
		printf("Refusing to start until this is fixed.\n");
		printf("</PRE>\n");

		/* leave something in the httpd log for the admin */
		fprintf(stderr, "upsset.conf does not exist to permit execution\n");
		exit(0);
	}

	/* if we've been enabled, jump out of here and go to work */
	if (magic_string_set == 1)
		return;

	printf("<PRE>\n");
	printf("Error: Secure mode has not been enabled in upsset.conf.\n");
	printf("Refusing to start until this is fixed.\n");
	printf("</PRE>\n");

	/* leave something in the httpd log for the admin */
	fprintf(stderr, "upsset.conf does not permit execution\n");

	exit(0);
}	

int main(int argc, char **argv)
{
	username = password = function = ups = NULL;

	printf("Content-type: text/html\n\n");

	/* see if the magic string is present in the config file */
	check_conf();

	extractpostargs();

	if ((!username) || (!password) || (!function))
		loginscreen();

	if (!strcmp(function, "pickups"))
		initial_pickups();

	if (!strcmp(function, "showsettings"))
		showsettings();

	if (!strcmp(function, "savesettings"))
		savesettings();

#if 0		/* FUTURE */
	if (!strcmp(function, "showstatus"))
		showstatus();
#endif

	if (!strcmp(function, "showcmds"))
		showcmds();

	if (!strcmp(function, "docmd"))
		docmd();

	printf("Error: Unhandled function name [%s]\n", function);
	
	return 0;
}
