/* upslog - log ups values to a file for later collection and analysis

   Copyright (C) 1998  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "common.h"
#include "upsfetch.h"

#include "config.h"
#include "timehead.h"

static	FILE	*logfile;
static	const char *logfn;
static	sigset_t sigmask;

void sig_hup(int signal)
{
	if (logfile == stdout) {
		upslogx(LOG_INFO, "SIGHUP: logging to stdout");
		return;
	}

	upslogx(LOG_INFO, "SIGHUP: Reopening log file");

	fclose (logfile);
	logfile = fopen (logfn, "a");
	if (logfile == NULL)
		fatal("could not reopen logfile %s", logfn);

	return;
}

/* install handler for reloading config on SIGHUP */
void instsighandler (void)
{
	struct	sigaction	sa;

	sigemptyset (&sigmask);
	sigaddset (&sigmask, SIGHUP);
	sa.sa_mask = sigmask;
	sa.sa_handler = sig_hup;
	sa.sa_flags = 0;
	if (sigaction (SIGHUP, &sa, NULL) < 0)
		fatal("Can't install SIGHUP handler");
}

int main(int argc, char *argv[])
{
	const char *monhost;
	int	interval, i;
	char logformat[LARGEBUF];
	
	strlcpy(logformat, "%TIME @Y@m@d @H@M@S% %VAR battpct% %VAR utility% %VAR loadpct% [%VAR status%] %VAR upstemp% %VAR acfreq%", sizeof(logformat));

	if (argc < 4) {
		printf ("usage: %s <host> <logfile> <interval> [format string]\n"
			, argv[0]);
		printf ("logfile may be - to log to stdout\n");
		printf ("Valid format string escapes:\n");
		printf ("\t%%%% insert a single %%\n");
		printf ("\t%%TIME format%% insert the time with strftime formatting\n");
		printf ("\t%%HOST%% insert the local hostname\n");
		printf ("\t%%UPSHOST%% insert the host of the ups being monitored\n");
		printf ("\t%%PID%% insert the pid of upslog\n");
		printf ("\t%%VAR varname%% insert the value of ups variable varname\n");
		printf ("format string defaults to:\n");
		printf ("%s", logformat);
		printf ("See README in source tree for more information\n");
		exit (0);
	}	

	monhost = argv[1];
	logfn = argv[2];
	interval = atoi (argv[3]);

	if (argc != 4) {
		/* read out the remaining argv entries to the format string */
		logformat[0] = logformat[sizeof(logformat)-1] = 0;
		for (i = 4; i < argc; i++) {
			strlcat(logformat, argv[i], sizeof(logformat));
			strlcat(logformat, " ", sizeof(logformat));
		};
	};

	printf ("Network UPS Tools upslog %s\n", UPS_VERSION);
	printf ("logging status of %s to %s (%is intervals)\n", 
		monhost, logfn, interval);

	if (strcmp(logfn, "-") == 0)
		logfile = stdout;
	else
		logfile = fopen (logfn, "a");

	if (logfile == NULL)
		fatal("could not open logfile %s", logfn);

	openlog ("upslog", LOG_PID, LOG_FACILITY); 

	if (logfile != stdout) {
		background();
		instsighandler();
	}

	writepid("upslog");

	for (;;) {
		/* variables local to the for loop */
		char	logbuffer[LARGEBUF];
		int pctpos, logbufremain;
		time_t	tod;
		
		time (&tod);
		pctpos = 0; /* to make -Wall happy */

		/* parse the format string */
		memset(logbuffer, 0, sizeof(logbuffer));
		logbufremain = sizeof(logbuffer) - 1;
		for (i = 0; (i < strlen(logformat)) && (logbufremain > 0); i++) {
			if (logformat[i] != '%') {
				strncat(logbuffer, logformat+i, 1);
				logbufremain--;
			}
			else {
				char *cmpstr = logformat+i+1;
				char *cpctpos;
				/* we have hit a format string, parse it */
				/* find the pos of the ending % */
				cpctpos = strchr(cmpstr,'%');
				if (cpctpos == NULL) {
					strlcat(logbuffer, "INVALID", sizeof(logbuffer));
					logbufremain -= 7;
					i = strlen(logformat);
					break; /* skip to the next iteration of the for loop */
				};
				pctpos = cpctpos - logformat;
				/* now, what is between the % signs? */
				if (pctpos == i + 1) {
					/* nothing */
					strlcat(logbuffer, "%", sizeof(logbuffer));
					logbufremain--;
				}
				else if (strncasecmp(cmpstr, "HOST", 4) == 0) {
					char hnbuffer[LARGEBUF];
					/* print the hostname */
					gethostname(hnbuffer, sizeof(hnbuffer));
					strlcat(logbuffer, hnbuffer, sizeof(logbuffer));
					logbufremain -= strlen(hnbuffer);
				}
				else if (strncasecmp(cmpstr, "UPSHOST", 7) == 0) {
					strlcat(logbuffer, monhost, sizeof(logbuffer));
					logbufremain -= strlen(monhost);
				}
				else if (strncasecmp(cmpstr, "PID", 3) == 0) {
					pid_t mypid;
					char tmpstr[11];
					mypid = getpid();
					snprintf(tmpstr, 11, "%d", mypid);
					tmpstr[10] = 0;
					strlcat(logbuffer, tmpstr, sizeof(logbuffer));
					logbufremain -= strlen(tmpstr);
				}
				else if (strncasecmp(cmpstr, "TIME", 4) == 0) {
					/* this one requires a little processing */
					int j;
					char timeformat[64], timestr[64];
					if (pctpos - i - 6 <= 0) {
						strlcat(logbuffer, "INVALID", sizeof(logbuffer));
						logbufremain -= 7;
					}
					else {
						memset(timeformat, 0, 64);
						strncpy(timeformat, cmpstr+5, pctpos - i - 6);
						/* build the strftime format string; to avoid problems,
						* @ is passed on the command line instead of % */
						for (j = 0; j < sizeof(timeformat); j++)
							if (timeformat[j] == '@')
								timeformat[j] = '%';
						timestr[0] = 0;
						strftime(timestr, sizeof(timestr), timeformat, localtime(&tod));
						strlcat(logbuffer, timestr, sizeof(logbuffer));
						logbufremain -= strlen(timestr);
					};
				}
				else if (strncasecmp(cmpstr, "VAR", 3) == 0) {
					char varname[64], varval[16];
					int res;
					if (pctpos - i - 5 <= 0) {
						strlcat(logbuffer, "INVALID", sizeof(logbuffer));
						logbufremain -= 7;
					}
					else {
						memset(varname, 0, sizeof(varname));
						strncpy(varname, cmpstr+4, pctpos - i - 5);
						res = getupsvar (monhost, varname, varval, sizeof(varval));
						if (res < 0)
							strcpy(varval, "NA");
						strlcat(logbuffer, varval, sizeof(logbuffer));
						logbufremain -= strlen(varval);
					};
				}
				else if (strncasecmp(cmpstr, "ETIME", 5) == 0) {
					char	timebuf[16];

					snprintf (timebuf, sizeof(timebuf), "%ld", (unsigned long)tod);
					strlcat(logbuffer, timebuf, sizeof(logbuffer));
					logbufremain -= strlen(timebuf);
				}
				else {
					strlcat(logbuffer, "INVALID", sizeof(logbuffer));
					logbufremain -= 7;
				}; /* what escape sequence */
				i = pctpos;
			}; /* if (logformat[i] != '%') */
		}; /* for (i = 0; (i < strlen(logformat)) && (logbufremain > 0); i++) */
		
		fprintf (logfile, "%s\n", logbuffer);
		fflush (logfile);
		sleep (interval);
	}

	/* notreached */

	return (1);
}
