/**
 * @file user_chunk.c
 *
 * Functions the work with the user pointer
 */
#include <lib/portability.h>
#include <lib/allocator.h>
#include <lib/callstack_pool.h>
#include <lib/entry_pool.h>
#include <lib/heap_entry.h>
#include <lib/user_chunk.h>
#include <lib/output.h>
#include <lib/util.h>
#include <lib/block.h>

static struct nj_heap_entry *prot_none_chunk_get_entry(nj_addr_t, struct nj_entry_pool *, nj_entry_index_t *);
static struct nj_heap_entry *prot_over_chunk_get_entry(nj_addr_t, struct nj_entry_pool *, nj_entry_index_t *);
static struct nj_heap_entry *prot_under_chunk_get_entry(nj_addr_t, struct nj_entry_pool *, nj_entry_index_t *);
static struct nj_heap_entry *prot_sunder_chunk_get_entry(nj_addr_t, struct nj_entry_pool *, nj_entry_index_t *);

/** 
 * Sets the fencepost
 *
 * @param user_chunk The user pointer
 * @param user_len The user's length
 * @param prefs The user's prefs
 */
void __nj_user_chunk_set_fencepost(nj_addr_t user_chunk, size_t user_len,
			struct nj_dynamic_prefs prefs)
{
	switch(prefs.alloc_type)
	{
		case NJ_PROT_NONE:
			NJ_PROT_NONE_GET_FENCEPOST(user_chunk, user_len) = NJ_FENCEPOST;
			break;
		case NJ_PROT_OVER:
			NJ_PROT_OVER_GET_FENCEPOST(user_chunk, user_len) = NJ_FENCEPOST;
			break;
		case NJ_PROT_UNDER:
			NJ_PROT_UNDER_GET_FENCEPOST(user_chunk, user_len) = NJ_FENCEPOST;
			break;
		case NJ_PROT_SUNDER:
			NJ_PROT_SUNDER_GET_FENCEPOST(user_chunk, user_len) = NJ_FENCEPOST;
			break;
	}
}

/**
 * Find the heap entry for the user chunk
 *
 * This function is used when the user says they wish to change their allocation
 * type on the fly. It uses the fact that different allocation mechanisms
 * store the indice in predictable places based on the user chunk.
 * 
 * @param user_chunk The chunk the search
 * @param entry_pool The entry pool to use
 * @param idx Where to store the index if we find it
 * @returns The entry if we find it, otherwise we exit with fatal error
 */
struct nj_heap_entry *__nj_user_chunk_find_entry(nj_addr_t user_chunk, 
		struct nj_entry_pool *entry_pool,
		nj_entry_index_t *idx)
{
	struct nj_heap_entry *entry;
	nj_addr_t page_boundary = user_chunk & NJ_PAGE_MASK;
	/**
	 * The segments rank thuswise in their alignments:
     * 
	 * PROT_OVER has a fencepost right before its non-page aligned buffer
	 *  (Though it may be page aligned if len == 4096)
	 *  and an entry index on the page boundary before that (if it fit, 
	 *  otherwise 1 more)
	 * PROT_UNDER has a entry index right before its non-page aligned buffer
	 * PROT_NONE has an entry index right before its non-page or page aligned
	 *  buffer
	 * PROT_SUNDER has an entry index 1 page below 
	 *  
	 * This allows for 6 cases. Draw them out on paper.
	 */

	/* If the user ptr is not page aligned */
	if(page_boundary != user_chunk)
	{
		/* Cases: most over, most none, all under  */
		if(*(u_int *)NJ_ALIGN_DOWN(user_chunk - sizeof(NJ_FENCEPOST), NJ_INT_ALIGNMENT) == NJ_FENCEPOST)
		{
			/* We have overflow checking for sure at this point */
			if((entry = prot_over_chunk_get_entry(user_chunk, entry_pool, idx)) != NULL)
				return entry;
			else
				/* index itself was corrupt: small underflow or bad ptr */
				/** @FIXME Make a code and class for this */
				__nj_output_fatal_user_error(0, 0, user_chunk, NULL, "Small underflow or bad pointer");
		}
		else
		{
			/* If we didn't find a fencepost, check for prot_under.
			 * if that fails, try prot_none.
			 * If that fails, we have a corrupt index, or a bad user ptr */
			if((entry = prot_under_chunk_get_entry(user_chunk, entry_pool, idx)) != NULL)
				return entry;
			else if((entry = prot_none_chunk_get_entry(user_chunk, entry_pool, idx)) != NULL)
				return entry;
			else /* index itself corrupt: small underflow, bad ptr */
				__nj_output_fatal_user_error(0, 0, user_chunk, NULL, "Small underflow or bad pointer");
		}
	}
	else
	{
		/* Cases: all sunder, some none, and some over segments 
		 *   (those with length == 4096) are page aligned */

		/* First try sunder, since it will kill us. 
		 * Sunder will also catch over segments that are page aligned..
		 * Draw it and then realize the entry pointer tells fencepost_ok
		 *   what type it really is, and you'll see...
		 * Isn't that neat? ;) 
		 * */
		if((entry = prot_sunder_chunk_get_entry(user_chunk, entry_pool, idx)) != NULL)
			return entry;
		else if((entry = prot_none_chunk_get_entry(user_chunk, entry_pool, idx)) != NULL)
			return entry;
		else /* Bad index: underflow, or bad ptr */
			__nj_output_fatal_user_error(0, 0, user_chunk, NULL, "Small underflow or bad pointer");
	}

	/* not reached */
	__nj_eprintf(__FUNCTION__" INTERNAL ERROR.. Should not have reached here\n");
	return NULL;
}


/**
 * Get an entry given an alloc type
 *
 * Much simpler than find entry, this is used when the allocation type will 
 * not change during the execution of the program
 * 
 * @param user_chunk THe user's pointer
 * @param entry_pool The entry pool to search
 * @param alloc_type The allocation type of the segment
 * @param idx The place we store the index
 * @returns the heap entry we found, or we exit
 */
struct nj_heap_entry *__nj_user_chunk_get_entry(nj_addr_t user_chunk, 
		struct nj_entry_pool *entry_pool,
		int alloc_type,
		nj_entry_index_t *idx)
{
	struct nj_heap_entry *entry;

	switch(alloc_type)
	{
		case NJ_PROT_NONE:
			if((entry = prot_none_chunk_get_entry(user_chunk, entry_pool, idx)) != NULL)
				return entry;
			else
				/* index itself corrupt: small underflow, or bad ptr */
				__nj_output_fatal_user_error(0, 0, user_chunk, NULL, "Small underflow or bad pointer");
			break;

		case NJ_PROT_OVER:
			if((entry = prot_over_chunk_get_entry(user_chunk, entry_pool, idx)) != NULL)
				return entry;
			else
				/* index itself corrupt: small underflow, or bad ptr */
				__nj_output_fatal_user_error(0, 0, user_chunk, NULL, "Small underflow or bad pointer");
			break;

		case NJ_PROT_UNDER:
			if((entry = prot_under_chunk_get_entry(user_chunk, entry_pool, idx)) != NULL)
				return entry;
			else
				/* index itself corrupt: small underflow, or bad ptr */
				__nj_output_fatal_user_error(0, 0, user_chunk, NULL, "Small underflow or bad pointer");
			break;

		case NJ_PROT_SUNDER:
			if((entry = prot_sunder_chunk_get_entry(user_chunk, entry_pool, idx)) != NULL)
				return entry;
			else
				/* index itself corrupt: small underflow, or bad ptr */
				__nj_output_fatal_user_error(0, 0, user_chunk, NULL, "Small underflow or bad pointer");
			break;
	}

	__nj_output_fatal_internal_error(0,0,__FUNCTION__, __LINE__, " INTERNAL ERROR.. Should not have reached here\n");
	return NULL;
}
/**@{ @name Entry retrival functions
 *
 * These functions will find the entry index and get the entry for each alloc type
 */	
static struct nj_heap_entry *prot_none_chunk_get_entry(nj_addr_t user_chunk, 
		struct nj_entry_pool *entry_pool,
		nj_entry_index_t *idx)
{
	struct nj_heap_entry *entry;
	*idx = *((nj_entry_index_t *)NJ_ALIGN_DOWN(user_chunk-sizeof(nj_entry_index_t), NJ_INT_ALIGNMENT));

	entry = __nj_entry_pool_get_valid_entry(entry_pool, *idx, NJ_ALLOCATOR_BLOCK_UNKNOWN, user_chunk);

	if(NJ_PROT_NONE_GET_FENCEPOST(user_chunk, entry->user_len) == NJ_FENCEPOST)
		return entry;
	else
		__nj_output_fatal_user_error(0, 0,
			user_chunk, NULL, "Fencepost Corrupt by Overflow");

	return NULL;
}

static struct nj_heap_entry *prot_over_chunk_get_entry(nj_addr_t user_chunk,
		struct nj_entry_pool *entry_pool,
		nj_entry_index_t *idx)
{
	struct nj_heap_entry *entry;
	nj_addr_t block = 
		((nj_entry_index_t *)(user_chunk - (user_chunk & NJ_PAGE_MASK) < (sizeof(nj_entry_index_t) + sizeof(NJ_FENCEPOST))) ?
		 ((user_chunk & NJ_PAGE_MASK) - NJ_PAGE_SIZE) :
		 (user_chunk & NJ_PAGE_MASK));

	*idx = *(nj_entry_index_t *)block;

	entry = __nj_entry_pool_get_valid_entry(entry_pool, *idx, block, user_chunk);

	if(NJ_PROT_OVER_GET_FENCEPOST(user_chunk, entry->user_len) == NJ_FENCEPOST)
		return entry;
	else
		__nj_output_fatal_user_error(0, 0,
			user_chunk, NULL, "Fencepost Corrupt by Underflow");

	return NULL;
}

static struct nj_heap_entry *prot_under_chunk_get_entry(nj_addr_t user_chunk,
		struct nj_entry_pool *entry_pool,
		nj_entry_index_t *idx)
{
	struct nj_heap_entry *entry;
	nj_addr_t block = (user_chunk & NJ_PAGE_MASK) - NJ_PAGE_SIZE;

	*idx = *((nj_entry_index_t *)(user_chunk & NJ_PAGE_MASK));

	entry = __nj_entry_pool_get_valid_entry(entry_pool, *idx, block, user_chunk);

	if(NJ_PROT_UNDER_GET_FENCEPOST(user_chunk, entry->user_len) == NJ_FENCEPOST)
		return entry;
	else
		__nj_output_fatal_user_error(0, 0,
			user_chunk, NULL, "Fencepost Corrupt by Overflow");

	/* not reached */
	return NULL;

}

static struct nj_heap_entry *prot_sunder_chunk_get_entry(nj_addr_t user_chunk,
		struct nj_entry_pool *entry_pool,
		nj_entry_index_t *idx)
{
	struct nj_heap_entry *entry;
	nj_addr_t block = (user_chunk - NJ_PAGE_SIZE);

	/* allow us access to that first page where the idx is */
	mprotect((void *)block, NJ_PAGE_SIZE, PROT_READ|PROT_WRITE|PROT_EXEC);
	
	*idx = *(nj_entry_index_t *)block;

	/* Alignment doesn't matter for sunder.. it's all page aligned */
	entry = __nj_entry_pool_get_valid_entry(entry_pool, *idx, block, 
			user_chunk);

	if(NJ_PROT_SUNDER_GET_FENCEPOST(user_chunk, entry->user_len) == NJ_FENCEPOST)
		return entry;
	else
		__nj_output_fatal_user_error(0, 0,
			user_chunk, NULL, "Fencepost Corrupt by Overflow");

	/* not reached */
	return NULL;
}
/*@}*/


// vim:ts=4
