#include <fe/njamd.h>
#include <fe/fe.h>
#include <fe/util/exec.h>
#include <fe/util/libcwrap.h>

static void _njConsSigChld(int signo);
static void _njConsSigCleanup(int signo);
static void _njCleanup(void);
static void _njKeypress(int ch);
static void _njGDBKeypress(int ch);
static void _njHelp(void);
static void _njSet(const char* buf);
static void _njStart(void);
static void _njStats(void);

static pid_t gdb = -1;
static pid_t child = -1;
static WINDOW* njamdwin;
static WINDOW* gdbwin;
static WINDOW* currwin;
static int gdbstdin, gdbstdout, gdbstderr;
static NJOPTIONS* options;
static int listensock = -1;
static int tcpsock = -1;
static struct sockaddr_in svrAddr;
static size_t svrAddrLen = sizeof(svrAddr);
static struct sockaddr_in clientAddr;
static size_t clientAddrLen = sizeof(clientAddr);

static int alignment = 1;
static int isPost = 1;
static int segvOnDoubleFree = 0;
static int statsOnExit = 1;
static int unmapOnFree = 0;
static int allowRead = 1;

const int whiteOnBlack = 1;
const int redOnBlack = 2;
const int greenOnBlack = 3;
const int blueOnBlack = 4;
const int yellowOnBlack = 5;
const int cyanOnBlack = 6;

void njConsStart(NJOPTIONS* opts)
{
	int     i;
	char**  args;
	int     wantToExit = 0;
	
	options = opts;

	atexit(_njCleanup);

	njSignal(SIGCHLD, _njConsSigChld);
	njSignal(SIGINT, _njConsSigCleanup);
	njSignal(SIGTERM, _njConsSigCleanup);

	listensock = njSocket(AF_INET, SOCK_STREAM, 0);
	svrAddr.sin_family = AF_INET;
	svrAddr.sin_port = htons(0);
	svrAddr.sin_addr.s_addr = htons(INADDR_ANY);
	njBind(listensock, (struct sockaddr*) &svrAddr, svrAddrLen);
	njGetSockName(listensock, (struct sockaddr*) &svrAddr, &svrAddrLen);
	njListen(listensock, 1);

	initscr();
	start_color();
	cbreak();
	noecho();
	nl();
	keypad(stdscr, TRUE);

	init_pair(whiteOnBlack, COLOR_WHITE, COLOR_BLACK);
	init_pair(redOnBlack, COLOR_RED, COLOR_BLACK);
	init_pair(greenOnBlack, COLOR_GREEN, COLOR_BLACK);
	init_pair(blueOnBlack, COLOR_BLUE, COLOR_BLACK);
	init_pair(yellowOnBlack, COLOR_YELLOW, COLOR_BLACK);
	init_pair(cyanOnBlack, COLOR_CYAN, COLOR_BLACK);

	wmove(stdscr, 0, 0);
	wattron(stdscr, COLOR_PAIR(cyanOnBlack));
	wprintw(stdscr, "NJAMD - Not Just Another Malloc Debugger");
	wattroff(stdscr, COLOR_PAIR(cyanOnBlack));
	wmove(stdscr, 1, 0);
	wattron(stdscr, COLOR_PAIR(blueOnBlack));
	for (i = 0; i < COLS; i++)
		waddch(stdscr, '-');
	wattroff(stdscr, COLOR_PAIR(blueOnBlack));
	njamdwin = subwin(stdscr, 5, COLS, 2, 0);
	wmove(stdscr, 7, 0);
	wattron(stdscr, COLOR_PAIR(blueOnBlack));
	for (i = 0; i < COLS; i++)
		waddch(stdscr, '-');
	wattroff(stdscr, COLOR_PAIR(blueOnBlack));
	gdbwin = subwin(stdscr, LINES - 8, COLS, 8, 0);
	touchwin(stdscr);

	scrollok(njamdwin, TRUE);
	scrollok(gdbwin, TRUE);

	nodelay(stdscr, TRUE);
	nodelay(njamdwin, TRUE);
	nodelay(gdbwin, TRUE);

	wrefresh(stdscr);

	wattron(njamdwin, COLOR_PAIR(greenOnBlack));
	wprintw(njamdwin, "njamd> ");
	wattroff(njamdwin, COLOR_PAIR(greenOnBlack));
	wrefresh(njamdwin);

	args = njMalloc(sizeof(char*) * 2);

	args[0] = njStrdup("gdb");
	args[1] = njStrdup(opts->argv0);

	currwin = njamdwin;

	gdb = njExecuteProgram(args, &gdbstdin, &gdbstdout, &gdbstderr, NULL);

	while (!wantToExit) {
		int    maxfd = -1;
		int    ret,x,y;
		fd_set fdset;
		char   buf[80];

		getyx(currwin, y, x);
		wmove(currwin, y, x);
		wrefresh(currwin);

		FD_ZERO(&fdset);
		FD_SET(STDIN_FILENO, &fdset);
		FD_SET(gdbstdout, &fdset);
		FD_SET(gdbstderr, &fdset);
		if (listensock != -1)
			FD_SET(listensock, &fdset);
		if (tcpsock != -1)
			FD_SET(tcpsock, &fdset);

		maxfd = NJ_MAX(maxfd, STDIN_FILENO);
		maxfd = NJ_MAX(maxfd, gdbstdout);
		maxfd = NJ_MAX(maxfd, gdbstderr);
		if (listensock != -1)
			maxfd = NJ_MAX(maxfd, listensock);
		if (tcpsock != -1)
			maxfd = NJ_MAX(maxfd, tcpsock);

		njSelect(maxfd + 1, &fdset, NULL, NULL, NULL);

		if (listensock != -1 && FD_ISSET(listensock, &fdset)) {
			tcpsock = njAccept(listensock, (struct sockaddr*) &clientAddr,
					&clientAddrLen);

			ret = njRead(tcpsock, buf, 80);
			buf[ret] = '\0';

			child = atoi(buf);

			njClose(listensock);
			listensock = -1;
		}

		if (tcpsock != -1 && FD_ISSET(tcpsock, &fdset)) {
			ret = njRead(tcpsock, buf, 80);

			for (i = 0; i < ret; i++) {
				wattron(gdbwin, A_BOLD | COLOR_PAIR(yellowOnBlack));
				waddch(gdbwin, buf[i]);
				wattroff(gdbwin, A_BOLD | COLOR_PAIR(yellowOnBlack));
				wrefresh(gdbwin);
			}
		}

		if (FD_ISSET(STDIN_FILENO, &fdset)) {
			int ch = getch();

			switch (ch) {
			case KEY_F(1):
				currwin = njamdwin;
				break;
			case KEY_F(2):
				currwin = gdbwin;
				break;
			default:
				if (currwin == gdbwin)
					_njGDBKeypress(ch);
				else
					_njKeypress(ch);
				break;
			}
		}

		if (FD_ISSET(gdbstdout, &fdset)) {
			ret = njRead(gdbstdout, buf, 80);

			for (i = 0; i < ret; i++) {
				waddch(gdbwin, buf[i]);
				wrefresh(gdbwin);
			}
		}

		if (FD_ISSET(gdbstderr, &fdset)) {
			ret = njRead(gdbstderr, buf, 80);
			
			for (i = 0; i < ret; i++) {
				waddch(gdbwin, buf[i]);
				wrefresh(gdbwin);
			}
		}
	}

	endwin();
}

/* Clean up by waiting for all children... */
static void _njConsSigChld(int signo)
{
	pid_t pid;
	int   stat;

	while ((pid = waitpid(-1, &stat, WNOHANG)) > 0) {
		if (WIFEXITED(stat)) {
			if (pid == gdb) {
				exit(EXIT_SUCCESS);
			}
		}

		fprintf(stderr, "Process %i exited with unknown status: %i\n", pid,
				stat);
		exit(EXIT_FAILURE);
	}
}

static void _njConsSigCleanup(int signo)
{
	signal(signo, SIG_DFL);

	_njCleanup();

	if (signo == SIGINT)
		exit(EXIT_SUCCESS);
}

static void _njCleanup(void)
{
	endwin();
}

static void _njKeypress(int ch)
{
	static char buf[80];
	static int  currIndx = 0;
	int y,x;

	switch (ch) {
	case KEY_BACKSPACE:
	case KEY_DC:
	case 0x7f:
		if (currIndx > 0) {

			getyx(njamdwin, y, x);
			wmove(njamdwin, y, x - 1);
			wdelch(njamdwin);
			wrefresh(njamdwin);
			currIndx--;
		}
		break;
	case '\n':
		if (currIndx == 0)
			return;
		buf[currIndx] = '\0';
		currIndx = 0;

		waddch(njamdwin, '\n');
		wrefresh(njamdwin);

		if (!strcasecmp(buf, "help")) _njHelp();
		else if (!strcasecmp(buf, "quit")) exit(EXIT_SUCCESS);
		else if (!strncasecmp(buf, "set", 3)) _njSet(buf);
		else if (!strcasecmp(buf, "start")) _njStart();
		else if (!strcasecmp(buf, "stats")) _njStats();
		else
			wprintw(njamdwin, "Unrecognized command: %s\n", buf);

		wattron(njamdwin, COLOR_PAIR(greenOnBlack));
		wprintw(njamdwin, "njamd> ");
		wattroff(njamdwin, COLOR_PAIR(greenOnBlack));
		wrefresh(njamdwin);
		break;
	case 0x15:	/* ^U */
		wdeleteln(njamdwin);
		getyx(njamdwin, y, x);
		wmove(njamdwin, y, 0);
		wattron(njamdwin, COLOR_PAIR(greenOnBlack));
		wprintw(njamdwin, "njamd> ");
		wattroff(njamdwin, COLOR_PAIR(greenOnBlack));
		wrefresh(njamdwin);
		currIndx=0;
		break;
	default:
		buf[currIndx++] = (char) ch;
		waddch(njamdwin, ch);
		wrefresh(njamdwin);
		break;
	}
}

static void _njGDBKeypress(int ch)
{
	static char buf[80];
	static int  currIndx = 0;

	switch (ch) {
	case KEY_BACKSPACE:
	case KEY_DC:
	case 0x7f:
		if (currIndx > 0) {
			int y,x;

			getyx(gdbwin, y, x);
			wmove(gdbwin, y, x - 1);
			wdelch(gdbwin);
			wrefresh(gdbwin);
			currIndx--;
		}
		break;
	case '\n':
		if (currIndx == 0)
			return;
		buf[currIndx++] = '\n';
		buf[currIndx] = '\0';
		currIndx = 0;

		waddch(gdbwin, '\n');
		wrefresh(gdbwin);

		njWrite(gdbstdin, buf, strlen(buf));
		break;
	default:
		buf[currIndx++] = (char) ch;
		waddch(gdbwin, ch);
		wrefresh(gdbwin);
		break;
	}
}

static void _njHelp(void)
{
	wprintw(njamdwin, "NJAMD Commands:\n");
	wprintw(njamdwin, "help quit set start stats\n");
	wrefresh(njamdwin);
}

static void _njSet(const char* buf)
{
	char* str;
	char* assign;
	char* varname;
	char* value;

	str = njStrdup(buf);
	(void) strtok(str, "\t ");
	assign = strtok(NULL, "\t ");

	if (assign == NULL) {
		wprintw(njamdwin, "alignment = %i ", alignment);
		wprintw(njamdwin, "is_post = %i ", isPost);
		wprintw(njamdwin, "segv_on_dbl_free = %i\n", segvOnDoubleFree);
		wprintw(njamdwin, "stats_on_exit = %i ", statsOnExit);
		wprintw(njamdwin, "unmap_on_free = %i ", unmapOnFree);
		wprintw(njamdwin, "allow_read = %i\n", allowRead);
	} else {
		varname = strtok(assign, "=");
		value = strtok(NULL, "\n");

		if (value == NULL) {
			/* print out value of variable varname */
			if (!strcasecmp(varname, "alignment"))
				wprintw(njamdwin, "alignment = %i\n", alignment);
			else if (!strcasecmp(varname, "is_post"))
				wprintw(njamdwin, "is_post = %i\n", isPost);
			else if (!strcasecmp(varname, "segv_on_dbl_free"))
				wprintw(njamdwin, "segv_on_dbl_free = %i\n", segvOnDoubleFree);
			else if (!strcasecmp(varname, "stats_on_exit"))
				wprintw(njamdwin, "stats_on_exit = %i\n", statsOnExit);
			else if (!strcasecmp(varname, "unmap_on_free"))
				wprintw(njamdwin, "unmap_on_free = %i\n", unmapOnFree);
			else if (!strcasecmp(varname, "allow_read"))
				wprintw(njamdwin, "allow_read = %i\n", allowRead);
			else
				wprintw(njamdwin, "Unrecognized variable: %s\n", varname);
			wrefresh(njamdwin);
		} else {
			/* set varname equal to value */
			if (!strcasecmp(varname, "alignment"))
				alignment = atoi(value);
			else if (!strcasecmp(varname, "is_post"))
				isPost = atoi(value);
			else if (!strcasecmp(varname, "segv_on_dbl_free"))
				segvOnDoubleFree = atoi(value);
			else if (!strcasecmp(varname, "stats_on_exit"))
				statsOnExit = atoi(value);
			else if (!strcasecmp(varname, "unmap_on_free"))
				unmapOnFree = atoi(value);
			else if (!strcasecmp(varname, "allow_read"))
				allowRead = atoi(value);
			else
				wprintw(njamdwin, "Unrecognized variable: %s\n", varname);
			wrefresh(njamdwin);
		}
	}
}

static void _njStart(void)
{
	char buf[80];
	int ret,i;

	snprintf(buf, 80, "set env LD_PRELOAD=libnjamd.so\n"); 
	njWrite(gdbstdin, buf, strlen(buf));
	wprintw(gdbwin, "%s", buf);

	ret = njRead(gdbstdout, buf, 80);
	for (i = 0; i < ret; i++) {
		waddch(gdbwin, buf[i]);
		wrefresh(gdbwin);
	}

	snprintf(buf, 80, "set env NJAMD_PROT=%s\n", 
			(isPost == 1 ? "over" : "under"));
	njWrite(gdbstdin, buf, strlen(buf));
	wprintw(gdbwin, "%s", buf);

	ret = njRead(gdbstdout, buf, 80);
	for (i = 0; i < ret; i++) {
		waddch(gdbwin, buf[i]);
		wrefresh(gdbwin);
	}

	snprintf(buf, 80, "set env NJAMD_ALIGN=%i\n", alignment);
	njWrite(gdbstdin, buf, strlen(buf));
	wprintw(gdbwin, "%s", buf);

	ret = njRead(gdbstdout, buf, 80);
	for (i = 0; i < ret; i++) {
		waddch(gdbwin, buf[i]);
		wrefresh(gdbwin);
	}

	if (statsOnExit) {
		snprintf(buf, 80, "set env NJAMD_DUMP_LEAKS_ON_EXIT=1\n");
		njWrite(gdbstdin, buf, strlen(buf));
		wprintw(gdbwin, "%s", buf);
	
		ret = njRead(gdbstdout, buf, 80);
		for (i = 0; i < ret; i++) {
			waddch(gdbwin, buf[i]);
			wrefresh(gdbwin);
		}
	}
	
	if (segvOnDoubleFree) {
		snprintf(buf, 80, "set env NJAMD_CHK_FREE=segv\n");
		njWrite(gdbstdin, buf, strlen(buf));
		wprintw(gdbwin, "%s", buf);

		ret = njRead(gdbstdout, buf, 80);
		for (i = 0; i < ret; i++) {
			waddch(gdbwin, buf[i]);
			wrefresh(gdbwin);
		}
	}

	if (unmapOnFree) {
		snprintf(buf, 80, "set env NJAMD_CHK_FREE=none\n");
		njWrite(gdbstdin, buf, strlen(buf));
		wprintw(gdbwin, "%s", buf);
	
		ret = njRead(gdbstdout, buf, 80);
		for (i = 0; i < ret; i++) {
			waddch(gdbwin, buf[i]);
			wrefresh(gdbwin);
		}
	}

	if (allowRead) {
		snprintf(buf, 80, "set env NJAMD_ALLOW_READ=1\n");
		njWrite(gdbstdin, buf, strlen(buf));
		wprintw(gdbwin, "%s", buf);

		ret = njRead(gdbstdout, buf, 80);
		for (i = 0; i < ret; i++) {
			waddch(gdbwin, buf[i]);
			wrefresh(gdbwin);
		}
	}

	snprintf(buf, 80, "set env NJAMD_FE_PORT=%i\n", ntohs(svrAddr.sin_port));
	njWrite(gdbstdin, buf, strlen(buf));
	wprintw(gdbwin, "%s", buf);

	ret = njRead(gdbstdout, buf, 80);
	for (i = 0; i < ret; i++) {
		waddch(gdbwin, buf[i]);
		wrefresh(gdbwin);
	}

	njWrite(gdbstdin, "run ", 4);
	wprintw(gdbwin, "run ");

	if (options->args != NULL) {
		char** ptr = options->args;

		while (*ptr != NULL) {
			njWrite(gdbstdin, *ptr, strlen(*ptr));
			wprintw(gdbwin, "%s", *ptr);
			ptr++;
		}
	}

	njWrite(gdbstdin, "\n", 1);
	wprintw(gdbwin, "\n");
	wrefresh(gdbwin);
}

static void _njStats(void)
{
	if (child != -1) {
		char buf[80];

		snprintf(buf, 80, "signal SIGUSR1\n");
		njWrite(gdbstdin, buf, strlen(buf));
		wprintw(gdbwin, "%s", buf);
	}
}
