# -*- coding: utf-8 -*-
#
#  Copyright (C) 2004-2013 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

# TODO:
# - きのことサーフェスの間に他のウインドウが入ることができてしまうのを直す.
# - NAYUKI/2.0
# - 透明度の設定

import os
import logging

import gtk
import glib
import cairo

import ninix.config
import ninix.seriko
import ninix.pix


class Menu(object):

    def __init__(self, accelgroup):
        self.request_parent = lambda *a: None # dummy
        ui_info = '''
        <ui>
          <popup name='popup'>
            <menuitem action='Settings'/>
            <menu action='Skin'>
            </menu>
            <separator/>
            <menuitem action='Exit'/>
          </popup>
        </ui>
        '''
        self.__menu_list = {
            'settings': [('Settings', None, ('Settings...(_O)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'edit_preferences')),
                         '/ui/popup/Settings'],
            'skin':     [('Skin', None, _('Skin(_K)'), None),
                         None, '/ui/popup/Skin'],
            'exit':     [('Exit', None, _('Exit(_Q)'), None,
                          '', lambda *a: self.request_parent('NOTIFY', 'close')),
                         '/ui/popup/Exit'],
            }
        self.__skin_list = None
        actions = gtk.ActionGroup('Actions')
        entry = [value[0] for value in self.__menu_list.values()]
        actions.add_actions(tuple(entry))
        ui_manager = gtk.UIManager()
        ui_manager.insert_action_group(actions, 0)
        ui_manager.add_ui_from_string(ui_info)
        self.__popup_menu = ui_manager.get_widget('/ui/popup')
        for key in self.__menu_list:
            path = self.__menu_list[key][-1]
            self.__menu_list[key][1] = ui_manager.get_widget(path)

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def popup(self, button):
        skin_list = self.request_parent('GET', 'get_skin_list')
        self.__set_skin_menu(skin_list)
        self.__popup_menu.popup(
            None, None, None, button, gtk.get_current_event_time())

    def __set_skin_menu(self, list): ## FIXME
        key = 'skin'
        if list:
            menu = gtk.Menu()
            for skin in list:
                item = gtk.MenuItem(skin['title'])
                item.connect(
                    'activate',
                    lambda a, k: self.request_parent('NOTIFY', 'select_skin', k),
                    (skin))
                menu.add(item)
                item.show()
            self.__menu_list[key][1].set_submenu(menu)
            menu.show()
            self.__menu_list[key][1].show()
        else:
            self.__menu_list[key][1].hide()


class Nayuki(object):

    def __init__(self):
        pass


class Kinoko(object):

    def __init__(self, skin_list):
        self.skin_list = skin_list
        self.skin = None

    def edit_preferences(self):
        pass

    def finalize(self):
        self.__running = 0
        self.target.detach_observer(self)
        if self.skin is not None:
            self.skin.destroy()

    def observer_update(self, event, args):
        if self.skin is None:
            return
        if event in ['set position', 'set surface']:
            self.skin.set_position()
            self.skin.show()
        elif event == 'set scale':
            scale = self.target.get_surface_scale()
            self.skin.set_scale(scale)
        elif event == 'hide':
            side = args
            if side == 0: # sakura side
                self.skin.hide()
        elif event == 'iconified':
            self.skin.hide()
        elif event == 'deiconified':
            self.skin.show()
        elif event == 'finalize':
            self.finalize()
        elif event == 'move surface':
            side, xoffset, yoffset = args
            if side == 0: # sakura side
                self.skin.set_position(xoffset, yoffset)
        elif event == 'raise':
            side = args
            if side == 0: # sakura side
                self.skin.set_position() ## FIXME
        else:
            ##logging.debug('OBSERVER(kinoko): ignore - {0}'.format(event))
            pass

    def load_skin(self):
        scale = self.target.get_surface_scale()
        self.skin = Skin(self.accelgroup)
        self.skin.set_responsible(self.handle_request)
        self.skin.load(self.data, scale)

    def handle_request(self, event_type, event, *arglist, **argdict):
        assert event_type in ['GET', 'NOTIFY']
        handlers = {
            'get_target_window': lambda *a: self.target.surface.window[0].window, # XXX
            'get_kinoko_position': self.target.get_kinoko_position,
            }
        handler = handlers.get(event,
                               getattr(self, event,
                                       lambda *a: None)) ## FIXME
        result = handler(*arglist, **argdict)
        if event_type == 'GET':
            return result

    def load(self, data, target):
        self.data = data
        self.target = target
        self.target.attach_observer(self)
        self.accelgroup = gtk.AccelGroup()
        self.load_skin()
        if self.skin is None:
            return 0
        else:
            self.send_event('OnKinokoObjectCreate')
        self.__running = 1
        glib.timeout_add(10, self.do_idle_tasks) # 10[ms]
        return 1

    def do_idle_tasks(self):
        return True if self.__running else False

    def close(self):
        self.finalize()
        self.send_event('OnKinokoObjectDestroy')

    def send_event(self, event):
        if event not in ['OnKinokoObjectCreate', 'OnKinokoObjectDestroy',
                         'OnKinokoObjectChanging', 'OnKinokoObjectChanged',
                         'OnKinokoObjectInstalled']:
                         ## 'OnBatteryLow', 'OnBatteryCritical',
                         ## 'OnSysResourceLow', 'OnSysResourceCritical'
            return
        args = (self.data['title'],
                self.data['ghost'],
                self.data['category'])
        self.target.notify_event(event, *args)

    def get_skin_list(self):
        return self.skin_list

    def select_skin(self, args):
        self.send_event('OnKinokoObjectChanging')
        self.skin.destroy()
        self.data = args
        self.load_skin()
        if self.skin is None:
            return 0
        else:
            self.send_event('OnKinokoObjectChanged')
        return 1


class Skin(object):

    def __init__(self, accelgroup):
        self.frame_buffer = []
        self.accelgroup = accelgroup
        self.request_parent = lambda *a: None # dummy
        self.__menu = Menu(self.accelgroup)
        self.__menu.set_responsible(self.handle_request)

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def handle_request(self, event_type, event, *arglist, **argdict):
        assert event_type in ['GET', 'NOTIFY']
        handlers = {
            }
        handler = handlers.get(event, getattr(self, event, None))
        if handler is None:
            result = self.request_parent(
                event_type, event, *arglist, **argdict)
        else:
            result = handler(*arglist, **argdict)
        if event_type == 'GET':
            return result

    def load(self, data, scale):
        self.data = data
        self.__scale = scale
        self.__shown = False
        self.surface_id = 0 # dummy
        self.window = ninix.pix.TransparentWindow()
        ##self.window.set_title(''.join(('surface.', name)))
        self.window.set_skip_taskbar_hint(True)
        self.window.connect('delete_event', self.delete)
        self.window.realize()
        self.window.window.set_back_pixmap(None, False)
        self.window.add_accel_group(self.accelgroup) ## FIXME
        if self.data['animation'] is not None:
            path = os.path.join(self.data['dir'], self.data['animation'])
            actors = {'': ninix.seriko.get_actors(ninix.config.create_from_file(path))}
        else:
            base, ext = os.path.splitext(self.data['base'])
            path = os.path.join(self.data['dir'], ''.join((base, 'a.txt')))
            if os.path.exists(path):
                actors = {'': ninix.seriko.get_actors(ninix.config.create_from_file(path))}
            else:
                actors = {'': []}
        self.seriko = ninix.seriko.Controler(
            actors)
        self.seriko.set_responsible(self.handle_request)
        path = os.path.join(self.data['dir'], self.data['base'])
        try:
            self.image_surface = ninix.pix.create_surface_from_file(path)
            w = max(8, int(self.image_surface.get_width() * self.__scale / 100))
            h = max(8, int(self.image_surface.get_height() * self.__scale / 100))
        except: ## FIXME
            self.request_parent('NOTIFY', 'close')
            return
        self.path = path
        self.w, self.h = w, h
        self.darea = self.window.get_child()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK|
                               gtk.gdk.BUTTON_PRESS_MASK|
                               gtk.gdk.BUTTON_RELEASE_MASK|
                               gtk.gdk.POINTER_MOTION_MASK|
                               gtk.gdk.LEAVE_NOTIFY_MASK)
        self.darea.connect('button_press_event', self.button_press)
        self.darea.connect('button_release_event', self.button_release)
        self.darea.connect('motion_notify_event', self.motion_notify)
        self.darea.connect('leave_notify_event', self.leave_notify)
        self.darea.connect('expose_event', self.redraw)
        self.window.update_size(self.w, self.h)
        self.set_position()
        target_window = self.request_parent('GET', 'get_target_window')
        if self.data['ontop']:
            self.window.set_transient_for(target_window)
        else:
            target_window.set_transient_for(self.window)
        self.show()
        self.seriko.reset(self, '') # XXX
        self.seriko.start(self)
        self.seriko.invoke_kinoko(self)

    def get_preference(self, name): # dummy
        if name == 'animation_quality':
            return 1.0
        else:
            return None

    def show(self):
        if not self.__shown:
            self.window.show()
            self.__shown = True

    def hide(self):
        if self.__shown:
            self.window.hide()
            self.__shown = False

    def append_actor(self, frame, actor):
        self.seriko.append_actor(frame, actor)

    def set_position(self, xoffset=0, yoffset=0):
        base_x, base_y = self.request_parent(
            'GET', 'get_kinoko_position', self.data['baseposition'])
        a, b = [(0.5, 1), (0.5, 0), (0, 0.5), (1, 0.5), (0, 1),
                (1, 1), (0, 0), (1, 0), (0.5, 0.5)][self.data['baseadjust']]
        offsetx = int(self.data['offsetx'] * self.__scale / 100)
        offsety = int(self.data['offsety'] * self.__scale / 100)
        self.x = base_x - int(self.w * a) + offsetx + xoffset
        self.y = base_y - int(self.h * b) + offsety + yoffset
        self.window.move(self.x, self.y)

    def set_scale(self, scale):
        self.__scale = scale
        self.reset_surface()
        self.set_position()

    def get_surface(self): ## FIXME
        return None

    def redraw(self, widget, cr):
        scale = self.__scale
        cr.scale(scale / 100.0, scale / 100.0)
        cr.set_source_surface(self.image_surface, 0, 0)
        cr.set_operator(cairo.OPERATOR_SOURCE)
        cr.paint()
        self.window.set_shape_surface(self.image_surface, 0, 0)

    def get_image_surface(self, surface_id):
        path = os.path.join(self.data['dir'],
                            ''.join(('surface', str(surface_id), '.png')))
        if os.path.exists(path):
            surface = ninix.pix.create_surface_from_file(path)
        else:
            surface = None
        return surface

    def create_image_surface(self, surface_id):
        if surface_id is not None and surface_id != '':
            surface = self.get_image_surface(surface_id)
        else:
            surface = ninix.pix.create_surface_from_file(self.path)
        return surface

    def update_frame_buffer(self):
        new_surface = self.create_image_surface(self.seriko.base_id)
        assert new_surface is not None
        # draw overlays
        for surface_id, x, y, method in self.seriko.iter_overlays():
            try:
                overlay_surface = self.get_image_surface(surface_id)
            except:
                continue
            # overlay surface
            cr = cairo.Context(new_surface)
            cr.set_source_surface(overlay_surface, x, y)
            cr.mask_surface(overlay_surface, x, y)
            del cr
        #self.darea.queue_draw_area(0, 0, w, h)
        self.image_surface = new_surface
        self.darea.queue_draw()

    def terminate(self):
        self.seriko.terminate(self)

    def add_overlay(self, actor, surface_id, x, y, method):
        self.seriko.add_overlay(self, actor, surface_id, x, y, method)

    def remove_overlay(self, actor):
        self.seriko.remove_overlay(actor)

    def move_surface(self, xoffset, yoffset):
        self.window.move(self.x + xoffset, self.y + yoffset)

    def reset_surface(self): ## FIXME
        self.seriko.reset(self, '') # XXX
        path = os.path.join(self.data['dir'], self.data['base'])
        w, h = ninix.pix.get_png_size(path)
        w = max(8, int(w * self.__scale / 100))
        h = max(8, int(h * self.__scale / 100))
        self.w, self.h = w, h # XXX
        self.window.update_size(w, h)
        self.window.queue_resize()
        self.seriko.start(self)
        self.seriko.invoke_kinoko(self)

    def set_surface(self, surface_id, restart=1): ## FIXME
        path = os.path.join(self.data['dir'],
                            ''.join(('surface', str(surface_id), '.png')))
        if os.path.exists(path):
            self.path = path
        else:
            #self.path = None
            self.path = os.path.join(self.data['dir'], self.data['base'])

    def invoke(self, actor_id, update=0):
        self.seriko.invoke(self, actor_id, update)

    def delete(self, widget, event):
        self.request_parent('NOTIFY', 'close')

    def destroy(self):
        self.seriko.destroy()
        self.window.destroy()

    def button_press(self, widget, event): ## FIXME
        self.x_root = event.x_root
        self.y_root = event.y_root
        if event.type == gtk.gdk.BUTTON_PRESS:
            click = 1
        else:
            click = 2
        button = event.button
        if button == 3 and click == 1:
            self.__menu.popup(button)
        return True

    def button_release(self,  widget, event): ## FIXME
        pass

    def motion_notify(self,  widget, event): ## FIXME
        pass

    def leave_notify(self,  widget, event): ## FIXME
        pass
