/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/

// RaStore.h: interface for the RaStore class.
//
//////////////////////////////////////////////////////////////////////

#ifndef RASTORE_H
#define RASTORE_H

#include "NewPKIStore.h"
#include <ReadersWriter.h>
#include <PEM_DER.h>
#include <HashCorrelation.h>
#include "AsynchJobs.h"
#include "AutoSynchLDAP.h"
#include "MailInfo.h"
#include "RaStore_ASN1.h"

/*!
	This class is the store for a RA
*/

class RaStore : public AutoSynchLDAPEvent, public NewPKIStore
{
public:
	/*! \brief This is the constructor.
	 *  \param EntityName [IN] The name of the entity.
	 *  \param e [IN] The ENGINE, can be NULL.
	 */
	RaStore(const mString & EntityName, ENGINE * e);

	/*! \brief This is the destructor.
	 */
	virtual ~RaStore();

	bool CreateTables(const SQL_Connection * DbConn);
	
	/*! \brief This function sets the groups info.
	 *  \param groups [IN] The groups info.
	 *  \return true on success, false on failure.
	 */
	bool SetGroups(const mVector<UsersGroup> & groups);
	
	/*! \brief This function sets the validator.
	 *  \param AclValidator [IN] The validator.
	 */
	void SetAclValidator(const X509_ACL_Validator * AclValidator);
	
	/*! \brief This function sets the minimum PKCS#12 password length.
	 *  \param MinPasswdLen [IN] The minimum PKCS#12 password length.
	 */
	void SetMinPasswdLen(unsigned long MinPasswdLen);

	/*! \brief This function sets the DN policies.
	 *  \param Policies [IN] The policies.
	 */
	void set_Policies(const HashTable_String & Policies);

	bool OnNewProfile(const NewpkiProfile & newProfile);

	bool GetKnownUIDs(mVector<mString> & KnownUIDs);

	/*! \brief This function inserts a new profile.
	 *  \param UserCert [IN] The certificate of the creator.
	 *  \param Profile [IN] The new profile.
	 *  \param ProfileId [OUT] The new profile's internal UID.
	 *  \return true on success, false on failure.
	 */
	bool InsertProfile(const PKI_CERT &UserCert, const NewpkiProfile & Profile, long & ProfileId);

	/*! \brief This function inserts a new profile from an EE.
	 *  \param dn [IN] The DN.
	 *  \param ee_id [IN] The EE id.
	 *  \param transactionId [IN] The transaction ID.
	 *  \param admin_validation [IN] The profile should be validated by an admin.
	 *  \param ownergroup [IN] The owner group.
	 *  \param ProfileId [OUT] The new profile's internal UID.
	 *  \return true on success, false on failure.
	 */
	bool InsertProfileFromEE(const X509_NAME * dn, const mString & ee_id, const Asn1OctetString & transactionId, bool admin_validation, unsigned long ownergroup, long & ProfileId);

	/*! \brief This function gets the list of profiles.
	 *  \param UserCert [IN] The certificate of the operator requesting the operation.
	 *  \param Profiles [OUT] The profiles list.
	 *  \param index [IN] The index for the enumeration.
	 *  \param num [IN] The maximum number of entries to return.
	 *  \param state [IN] The profiles state.
	 *  \param filter [IN] The a string to search for in the profiles' dn.
	 *  \param AllProfiles [IN] Should all profiles be enumerated by ignoring the owner.
	 *  \return true on success, false on failure.
	 */
	bool EnumProfiles(const PKI_CERT &UserCert, mVector<NewpkiProfileDatas> & Profiles, long index, long num, long state, const mString & filter, bool AllProfiles = false);

	/*! \brief This function deletes a profile.
	 *  \param UserCert [IN] The optional certificate of the operator requesting the operation.
	 *  \param ProfileId [IN] The profile's internal UID.
	 *  \param CheckOwnership [IN] Should the ownership be checked.
	 *  \return true on success, false on failure.
	 */
	bool DeleteProfile(const PKI_CERT &UserCert, long ProfileId, bool CheckOwnership);

	/*! \brief This function deletes a request.
	 *  \param UserCert [IN] The certificate of the operator requesting the operation.
	 *  \param RequestId [IN] The profile's internal UID.
	 *  \return true on success, false on failure.
	 */
	bool DeleteRequest(const PKI_CERT &UserCert, long RequestId);

	/*! \brief This function gets a profile.
	 *  \param UserCert [IN] The certificate of the operator requesting the operation.
	 *  \param ProfileId [IN] The profile's internal UID.
	 *  \param Profile [OUT] The profile.
	 *  \return true on success, false on failure.
	 */
	bool GetProfile(const PKI_CERT & UserCert, unsigned long ProfileId, NewpkiProfileDatas & Profile);

	/*! \brief This function gets the owner profile of a certificate.
	 *  \param CertId [IN] The certificate ID.
	 *  \param Profile [OUT] The profile.
	 *  \return true on success, false on failure.
	 */
	bool GetProfileByCertId(unsigned long CertId, NewpkiProfileDatas & Profile);

	/*! \brief This function gets an EE profile.
	 *  \param ee_name [IN] The name of the EE.
	 *  \param ProfileId [IN] The profile's internal UID.
	 *  \param Profile [OUT] The profile.
	 *  \return true on success, false on failure.
	 */
	bool GetEeProfile(const mString & ee_name, unsigned long ProfileId, NewpkiProfileDatas & Profile);

	/*! \brief This function returns the transaction ID associated with an EE profile.
	 *  \param UserCert [IN] The certificate of the operator requesting the operation.
	 *  \param ProfileId [IN] The profile's internal UID.
	 *  \param transactionId [OUT] The transaction ID.
	 *  \return true on success, false on failure.
	 */
	bool GetProfileTransactionId(const PKI_CERT & UserCert, unsigned long ProfileId, Asn1OctetString & transactionId);

	/*! \brief This function returns the transaction ID associated with an EE profile's certificate request.
	 *  \param CertReqId [IN] The certificate request's internal UID.
	 *  \param transactionId [OUT] The transaction ID.
	 *  \return true on success, false on failure.
	 */
	bool GetCertReqTransactionId(unsigned long CertReqId, Asn1OctetString & transactionId);

	/*! \brief This function sets the transaction ID associated with an EE profile's certificate request.
	 *  \param CertReqId [IN] The certificate request's internal UID.
	 *  \param transactionId [IN] The transaction ID.
	 *  \return true on success, false on failure.
	 */
	bool SetCertReqTransactionId(unsigned long CertReqId, const Asn1OctetString & transactionId);

	/*! \brief This function deletes the PKCS12 associated with a profile.
	 *  \param UserCert [IN] The certificate of the operator requesting the operation.
	 *  \param id [IN] The profile's internal UID.
	 *  \return true on success, false on failure.
	 */
	bool DeletePKCS12(const PKI_CERT &UserCert, unsigned long id);
	
	/*! \brief This function gets a certificate request.
	 *  \param ra_id [IN] The profile ID.
	 *  \param Cert [OUT] The certificate request.
	 *  \return true on success, false on failure.
	 */
	bool GetCertReq(unsigned long ra_id, NewpkiProfileDatasCert & Cert);

	/*! \brief This function deletes a certificate request.
	 *  \param CertReqId [IN] The certificate request's internal UID.
	 *  \return true on success, false on failure.
	 */
	bool DeleteCertReq(unsigned long CertReqId);

	/*! \brief This function marks a certificate for revocation.
	 *  \param UserCert [IN] The certificate of the operator requesting the operation.
	 *  \param CheckOwner [IN] Should the ownership of the certificate be verified.
	 *  \param id [IN] The certificate's internal UID.
	 *  \param serial [OUT] The serial of the certificate.
	 *  \param ca_name [OUT] The name of the CA that generated the certificate.
	 *  \param ldap_uid [OUT] The LDAP uid of the associated profile.
	 *  \return true on success, false on failure.
	 */
	bool MarkCertificateForRevocation(const PKI_CERT & UserCert, bool CheckOwner, unsigned long id, unsigned long & serial, mString & ca_name, mString & ldap_uid);

	/*! \brief This function marks a certificate for suspension.
	 *  \param UserCert [IN] The certificate of the operator requesting the operation.
	 *  \param id [IN] The certificate's internal UID.
	 *  \param serial [OUT] The serial of the certificate.
	 *  \param ca_name [OUT] The name of the CA that generated the certificate.
	 *  \param ldap_uid [OUT] The LDAP uid of the associated profile.
	 *  \return true on success, false on failure.
	 */
	bool MarkCertificateForSuspension(const PKI_CERT & UserCert, unsigned long id, unsigned long & serial, mString & ca_name, mString & ldap_uid);

	/*! \brief This function marks a certificate for unsuspension.
	 *  \param UserCert [IN] The certificate of the operator requesting the operation.
	 *  \param id [IN] The certificate's internal UID.
	 *  \param serial [OUT] The serial of the certificate.
	 *  \param ca_name [OUT] The name of the CA that generated the certificate.
	 *  \param ldap_uid [OUT] The LDAP uid of the associated profile.
	 *  \return true on success, false on failure.
	 */
	bool MarkCertificateForUnsuspension(const PKI_CERT & UserCert, unsigned long id, unsigned long & serial, mString & ca_name, mString & ldap_uid);

	/*! \brief This function unmarks a certificate for revocation.
	 *  \param id [IN] The certificate's internal UID.
	 *  \return true on success, false on failure.
	 */
	bool UnmarkCertificateForRevocation(unsigned long id);

	/*! \brief This function unmarks a certificate for suspension.
	 *  \param id [IN] The certificate's internal UID.
	 *  \return true on success, false on failure.
	 */
	bool UnmarkCertificateForSuspension(unsigned long id);

	/*! \brief This function unmarks a certificate for unsuspension.
	 *  \param id [IN] The certificate's internal UID.
	 *  \return true on success, false on failure.
	 */
	bool UnmarkCertificateForUnsuspension(unsigned long id);

	/*! \brief This function suspends a certificate.
	 *  \param sender_name [IN] The name of the Entity sending the response.
	 *  \param ra_id [IN] The id of the profile.
	 *  \param resp [IN] The response.
	 *  \return true on success, false on failure.
	 */
	bool InsertResponse(const mString & sender_name, unsigned long ra_id, const NewpkiResponse & resp);

	/*! \brief This function generates a new certificate request for a profile.
	 *  \param UserCert [IN] The certificate of the operator requesting the operation.
	 *  \param Request [IN] The request.
	 *  \param Csr [OUT] The CSR.
	 *  \param ldap_uid [OUT] The LDAP UID.
	 *  \param CertReqId [OUT] The request's internal UID.
	 *  \return true on success, false on failure.
	 */
	bool RequestCertificate(const PKI_CERT &UserCert, const NewpkiRequestCert & Request, PKI_CSR & Csr, mString & ldap_uid, unsigned long & CertReqId);

	/*! \brief This function generates a new certificate request for an EE profile.
	 *  \param transactionId [IN] The transaction Id from the EE request.
	 *  \param ee_name [IN] The name of the EE entity.
	 *  \param Request [IN] The request.
	 *  \param Csr [OUT] The CSR.
	 *  \param CertReqId [OUT] The request's internal UID.
	 *  \return true on success, false on failure.
	 */
	bool RequestCertificateForEE(const Asn1OctetString & transactionId, const mString & ee_name, const NewpkiEeRequestCert & Request, PKI_CSR & Csr, unsigned long & CertReqId);

	/*! \brief This function changes the DN of a profile.
	 *  \param UserCert [IN] The certificate of the operator requesting the operation.
	 *  \param id [IN] The profile's internal UID.
	 *  \param dn [IN] The new DN.
	 *  \return true on success, false on failure.
	 */
	bool ChangeProfileDN(const PKI_CERT & UserCert, unsigned long id, const X509_NAME * dn);
	
	/*! \brief This function changes the owner of a profile.
	 *  \param UserCert [IN] The certificate of the operator requesting the operation.
	 *  \param id [IN] The profile's internal UID.
	 *  \param group_id [IN] The new group id.
	 *  \return true on success, false on failure.
	 */
	bool ChangeProfileOwner(const PKI_CERT & UserCert, unsigned long id, unsigned long group_id);
	
	/*! \brief This function changes the LDAP UID of a profile.
	 *  \param UserCert [IN] The certificate of the operator requesting the operation.
	 *  \param id [IN] The profile's internal UID.
	 *  \param uid [IN] The new UID.
	 *  \return true on success, false on failure.
	 */
	bool ChangeProfileUID(const PKI_CERT & UserCert, unsigned long id, const mString & uid);

	/*! \brief This function marks the profile as accepted.
	 *  \param UserCert [IN] The certificate of the operator requesting the operation.
	 *  \param id [IN] The profile's internal UID.
	 *  \return true on success, false on failure.
	 */
	bool AcceptProfile(const PKI_CERT & UserCert, unsigned long id);
private:
	bool GetProfileAccessCheck(const PKI_CERT &UserCert, unsigned long ProfileId, SQL & sql);
	bool GetCertificateAccessCheck(const PKI_CERT &UserCert, unsigned long CertId, SQL & sql);


	bool DoInsertProfile(const PKI_CERT & UserCert, const NewpkiProfile & Profile, bool admin_validation, const mString & ee_id, const Asn1OctetString & transactionId, long & ProfileId);
	bool InsertResponseCa(const mString & ca_name, unsigned long ra_id, const NewpkiResponse & resp);
	bool InsertResponseEe(const mString & ee_name, unsigned long ra_id, const NewpkiResponse & resp);
	bool InsertResponseCert(const NewpkiCertResponse & cert_response, const mString & ca_name, PKI_CRL & Crl, LOG_MESSAGE_TYPE & LogMessage, LOG_MESSAGE_STATUS & LogStatus, int & Status, mString & strError, const HashTable_Dn & ProfileDn, mString & strPrivDatas, mString & x509, mString & p7b, unsigned long & serial, MailInfo & CertificateMail, MailInfo & PasswordMail, bool & SendCertificate, bool & SendPkcs12Pwd, bool & GenPkcs12Pwd);
	void GeneratePassword(mString & strP12Password);
	bool UpdateFromCRL(const mString & ca_name, const PKI_CRL & Crl);
	bool CanUserAccessProfile(const PKI_CERT & UserCert, unsigned long owner_serial, unsigned long owner_type);
	bool Sql2Cert(SQL * sql, NewpkiProfileDatasCert & Cert, int index);
	bool GetCerts(unsigned long ProfileId, mVector<NewpkiProfileDatasCert> & Certs);
	bool MarkCertificate(const PKI_CERT & UserCert, bool CheckOwner, unsigned long id, int NewStatus, unsigned long & serial, mString & ca_name, mString & ldap_uid);

	#define OWNER_TYPE_USER		1		//!< The owner of the profile is a user
	#define OWNER_TYPE_GROUP	2		//!< The owner of the profile is a group


	#define RASTORE_PROFILES_TABLE			"profiles"
	#define RASTORE_CERTS_TABLE				"certs"
	#define RASTORE_CREATE_1				"create table "RASTORE_PROFILES_TABLE" (profile_id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, ee_id TEXT NOT NULL, ee_tid VARCHAR(41) NOT NULL, owner_type INT UNSIGNED NOT NULL, owner_serial INT UNSIGNED NOT NULL, dn_hash VARCHAR(41) NOT NULL, dn LONGBLOB NOT NULL, ldap_uid LONGBLOB NOT NULL, state INT UNSIGNED NOT NULL, datas LONGBLOB NOT NULL, signature LONGBLOB NOT NULL, INDEX (profile_id), INDEX (dn_hash), INDEX (ee_tid));"
	#define RASTORE_CREATE_2				"create table "RASTORE_CERTS_TABLE" (id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, profile_id INT UNSIGNED NOT NULL, ee_tid VARCHAR(41) NOT NULL, ca_name BLOB NOT NULL, pkcs10 LONGBLOB NOT NULL, priv_datas LONGBLOB NOT NULL, status INT UNSIGNED NOT NULL, type INT UNSIGNED NOT NULL, flags LONGBLOB NOT NULL, admin_mail BLOB NOT NULL, error LONGBLOB NOT NULL, serial INT UNSIGNED NOT NULL, x509 LONGBLOB NOT NULL, p7b LONGBLOB NOT NULL, INDEX (profile_id), INDEX (ee_tid));"
	
	#define RASTORE_INSERT_PROFILE			"INSERT INTO "RASTORE_PROFILES_TABLE" (ee_id, ee_tid, owner_type, owner_serial, dn_hash, dn, ldap_uid, state, datas, signature) VALUES ('%s', '%s', '%d', '%ld', '%s', '%s', '%s', '%d', '%s', '%s');"
	#define RASTORE_GET_PROFILES			"SELECT profile_id, ee_id, state, datas, signature FROM "RASTORE_PROFILES_TABLE" %s ORDER BY profile_id DESC LIMIT %ld,%ld;"
	#define RASTORE_GET_PROFILES_CTRL		"SELECT datas, signature FROM "RASTORE_PROFILES_TABLE";"
	#define RASTORE_SEARCH_PROFILE			"SELECT profile_id FROM "RASTORE_PROFILES_TABLE" WHERE profile_id='%ld';"
	#define RASTORE_SEARCH_PROFILE_DN		"SELECT profile_id FROM "RASTORE_PROFILES_TABLE" WHERE dn_hash='%s' %s;"
	#define RASTORE_UPDATE_PROFILE_STATE	"UPDATE "RASTORE_PROFILES_TABLE" SET state='%d' WHERE profile_id='%ld';"
	#define RASTORE_DELETE_PROFILE			"DELETE FROM "RASTORE_PROFILES_TABLE" WHERE profile_id='%ld';"
	#define RASTORE_GET_ALL_PROFILES		"SELECT profile_id, datas FROM "RASTORE_PROFILES_TABLE" LIMIT %ld,%ld;"
	#define RASTORE_GET_PROFILE				"SELECT * FROM "RASTORE_PROFILES_TABLE" WHERE profile_id='%ld';"
	#define RASTORE_GET_KNOWN_RDNS			"SELECT ldap_uid FROM "RASTORE_PROFILES_TABLE" WHERE ldap_uid!='';"
	#define RASTORE_SET_UID					"UPDATE "RASTORE_PROFILES_TABLE" SET ldap_uid='%s', datas='%s', signature='%s' WHERE profile_id='%ld';"
	#define RASTORE_SET_OWNER				"UPDATE "RASTORE_PROFILES_TABLE" SET owner_type='%d', owner_serial='%ld', datas='%s', signature='%s' WHERE profile_id='%ld';"
	#define RASTORE_SET_DN					"UPDATE "RASTORE_PROFILES_TABLE" SET dn_hash='%s', dn='%s', datas='%s', signature='%s', state='%d' WHERE profile_id='%ld';"
	#define RASTORE_SET_STATE				"UPDATE "RASTORE_PROFILES_TABLE" SET state='%d' WHERE profile_id='%ld';"
	#define RASTORE_GET_EE_PROFILE			"SELECT * FROM "RASTORE_PROFILES_TABLE" WHERE ee_id='%s';"
	#define RASTORE_GET_PROFILE_BY_CERT		"SELECT * FROM "RASTORE_PROFILES_TABLE", "RASTORE_CERTS_TABLE" WHERE "RASTORE_CERTS_TABLE".profile_id = "RASTORE_PROFILES_TABLE".profile_id AND id='%ld';"

	#define RASTORE_INSERT_CERT				"INSERT INTO "RASTORE_CERTS_TABLE" (profile_id, ee_tid, ca_name, pkcs10, priv_datas, type, status, flags, admin_mail, error, serial, x509, p7b) VALUES ('%ld', '%s', '%s', '%s', '%s', '%d', '%d', '%s', '%s', '', 0, '', '');"
	#define RASTORE_GET_CERTS				"SELECT * FROM "RASTORE_CERTS_TABLE" WHERE profile_id=%ld ORDER BY id DESC;"
	#define RASTORE_SET_CERT_STATE			"UPDATE "RASTORE_CERTS_TABLE" SET status=%d WHERE id='%ld';"
	#define RASTORE_GET_CERT				"SELECT * FROM "RASTORE_CERTS_TABLE" WHERE id=%ld;"
	#define RASTORE_GET_CERT_WITH_PROFILE	"SELECT "RASTORE_CERTS_TABLE".*, "RASTORE_PROFILES_TABLE".datas FROM "RASTORE_CERTS_TABLE", "RASTORE_PROFILES_TABLE" WHERE "RASTORE_PROFILES_TABLE".profile_id = "RASTORE_CERTS_TABLE".profile_id AND "RASTORE_CERTS_TABLE".id=%ld;"
	#define RASTORE_DELETE_CERT				"DELETE FROM "RASTORE_CERTS_TABLE" WHERE id='%ld';"
	#define RASTORE_SET_CERT_RESP			"UPDATE "RASTORE_CERTS_TABLE" SET pkcs10='', status='%d', x509='%s', p7b='%s', error='%s', priv_datas='%s', serial='%ld' WHERE id='%ld';"
	#define RASTORE_GET_CERT_4_REV			"SELECT "RASTORE_CERTS_TABLE".x509, "RASTORE_CERTS_TABLE".ca_name, "RASTORE_CERTS_TABLE".status, "RASTORE_PROFILES_TABLE".owner_type, "RASTORE_PROFILES_TABLE".owner_serial, "RASTORE_PROFILES_TABLE".datas FROM "RASTORE_PROFILES_TABLE", "RASTORE_CERTS_TABLE" WHERE "RASTORE_CERTS_TABLE".id ='%ld' AND "RASTORE_PROFILES_TABLE".profile_id = "RASTORE_CERTS_TABLE".profile_id;"
	#define RASTORE_DELETE_PKCS12			"UPDATE "RASTORE_CERTS_TABLE" SET priv_datas='' WHERE id='%ld';"
	#define RASTORE_UPDATE_FROM_CRL_REV		"UPDATE "RASTORE_CERTS_TABLE" SET pkcs10='', priv_datas='', x509='', p7b='', status='%d', serial='0' WHERE ca_name='%s' AND serial='%ld';"
	#define RASTORE_UPDATE_FROM_CRL_SUSP	"UPDATE "RASTORE_CERTS_TABLE" SET pkcs10='', status='%d' WHERE ca_name='%s' AND serial='%ld';"
	#define RASTORE_UPDATE_FROM_CRL_UNSUSP	"UPDATE "RASTORE_CERTS_TABLE" SET pkcs10='', status='%d' WHERE ca_name='%s' AND status='%d' AND serial NOT IN (%s);"
	#define RASTORE_UPDATE_NO_SUSP			"UPDATE "RASTORE_CERTS_TABLE" SET pkcs10='', status='%d' WHERE ca_name='%s' AND status='%d';"
	#define RASTORE_GET_CERT_OWNER			"SELECT "RASTORE_CERTS_TABLE".*, "RASTORE_PROFILES_TABLE".owner_type, "RASTORE_PROFILES_TABLE".owner_serial FROM "RASTORE_PROFILES_TABLE", "RASTORE_CERTS_TABLE" WHERE "RASTORE_CERTS_TABLE".id ='%ld' AND "RASTORE_PROFILES_TABLE".profile_id = "RASTORE_CERTS_TABLE".profile_id;"
	#define RASTORE_SET_CERT_TID			"UPDATE "RASTORE_CERTS_TABLE" SET ee_tid='%s' WHERE id='%ld';"


	#define NEWPKI_PROFILE_sign(x,sig,pkey) \
		ASN1_item_sign(NewpkiProfile::get_ASN1_ITEM(), sig.get_sigAlg(), NULL, sig.get_signature(), (char *)x, pkey, EVP_sha1())

	#define NEWPKI_PROFILE_verify(x,sig,pkey) \
		ASN1_item_verify(NewpkiProfile::get_ASN1_ITEM(), sig.get_sigAlg(), sig.get_signature(),(char *)x, pkey)

	bool InsertCertReq(const mString & CaName, unsigned long ProfileId, int Type, const PKI_CSR & pkcs10, const RaStorePrivdatas & PrivDatas, const ASN1_BIT_STRING * flags, const mString & admin_mail, const Asn1OctetString & transactionId, unsigned long & CertReqId);

	bool ProfileExists(const mString & dn_hash, bool & exists, unsigned long & ProfileId);
	bool IsValidGroupId(unsigned long group_id);
	bool SignProfile(const NewpkiProfile & Profile, mString & pem_signature);
	bool SqlToNewpkiProfileDatas(SQL & sql, long pos, NewpkiProfileDatas & Profile);

	unsigned long m_MinPasswdLen;
	HashTable_String m_Policies;
	ReadersWriter ConfAccessLock;
	mVector<UsersGroup> m_groups;
	const X509_ACL_Validator * m_AclValidator;
};

#endif
