/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// AsynchJobs.h: interface for the AsynchJobs class.
//
//////////////////////////////////////////////////////////////////////

#ifndef ASYNCHJOBS_H
#define ASYNCHJOBS_H

#include <newpki_threads.h>
#include "EntityLog.h"
#include <mString.h>
#include "SmtpClient.h"
#include <PkiClient.h>
#include "X509_ACL_Validator.h"
#include "MailInfo.h"
#include "AsynchJobs_ASN1.h"

class StackRequest;



#ifndef ASYNCH_JOBS_INTERVAL
	#define ASYNCH_JOBS_INTERVAL 60 //!< The time to wait after a job is done, to restart it
#endif
#ifndef MAILS_DB_RELOAD
	#define MAILS_DB_RELOAD 1800  //!< The time to wait before reloading all the mails from the DB
#endif

/*!
	This class holds the jobs that need to be done asynchronously
*/
class AsynchJobs  
{
public:
	/*! \brief This is the constructor.
	 */
	AsynchJobs();

	/*! \brief This is the destructor.
	 */
	virtual ~AsynchJobs();

	/*! \brief This function stops all the asynchronous jobs.
	 */
	void StopAll();

	/*! \brief This function creates the tables needed to work.
	 *  \param DbConn [IN] A SQL connection.
	 *  \return true on success, false on failure.
	 */
	bool CreateTables(SQL_Connection * DbConn);

	/*! \brief This function sets the SQL connection.
	 *  \param DbConn [IN] A SQL connection.
	 *  \return true on success, false on failure.
	 */
	bool SetDbConn(SQL_Connection * DbConn);

	/*! \brief This function starts the configuration push job.
	 *  \param PushToAllRep [IN] When a new conf is declared, should it be sent to all known repositories?
	 *  \return true on success, false on failure.
	 */
	bool StartConfPush(bool PushToAllRep);

	/*! \brief This function stops the configuration push job.
	 */
	void StopConfPush();
	
	/*! \brief This function starts the mailer job.
	 *  \return true on success, false on failure.
	 */
	bool StartMailer();

	/*! \brief This function stops the mailer job.
	 */
	void StopMailer();

	/*! \brief This function sets the name of the entity using this class.
	 *  \param EntityName [IN] The name of the entity.
	 */
	void SetEntityName(const mString & EntityName);

	/*! \brief This function sets the certificate of the entity using this class.
	 *  \param cert [IN] The certificate of the entity.
	 */
	void SetEntityCert(const PKI_CERT & cert);
	
	/*! \brief This function sets the list of parents' certificate.
	 *  \param EntityParentCerts [IN] The list.
	 *  \return true on success, false on failure.
	 */
	bool SetEntityParentsCert(const mVector<PKI_CERT> & EntityParentCerts);
	
	/*! \brief This function sets the logging class.
	 *  \param log [IN] The logging class.
	 */
	void SetLogger(EntityLog * log);
	
	/*! \brief This function sets the mailer's info.
	 *  \param MailFrom [IN] The mail from.
	 *  \param conf [IN] The server configuration.
	 *  \return true on success, false on failure.
	 */
	bool SetMailerInfo(mString MailFrom, const EmailConf & conf);
	
	/*! \brief This function sets the list of admin mails.
	 *  \param Admins [IN] The list.
	 *  \return true on success, false on failure.
	 */
	bool SetAdminMails(const mVector<PkiAdminEntry> & Admins);
	
	/*! \brief This function sets the list of repositories.
	 *  \param Repositories [IN] The list.
	 *  \return true on success, false on failure.
	 */
	bool SetRepositories(const mVector<RepEntryInfo> & Repositories);
	
	/*! \brief This function sets the validator.
	 *  \param AclValidator [IN] The validator.
	 */
	void SetX509_ACL_Validator(const X509_ACL_Validator * AclValidator);
	
	/*! \brief Checks if the mailer is up and running.
	 *  \return true on success, false on failure.
	 */
	bool MailerIsUp();

	/*! \brief This function returns the mailer.
	 *  \return The mailer.
	 */
	const SmtpClient & GetMailer() const;
	
	/*! \brief This function is called to declare a new PKI conf, it will sent it to the repositories.
	 *  \param PkiConf [IN] The PKI conf.
	 *  \return true on success, false on failure.
	 */
	bool SetPkiConf(const ExportedPkiConf & PkiConf);
	
	
	/*! \brief This function is used to send an email.
	 *  \param Author [IN] The author.
	 *  \param Mail [IN] The email.
	 *  \param AdminMail [IN] Is it an admin mail?
	 *  \return true on success, false on failure.
	 */
	bool SendMail(const mString & Author, const MailInfo & Mail, bool AdminMail) const;

	/*! \brief This function is used to connect to a Repository.
	 *  \param EntityCert [IN] The certificate to use for the authentication.
	 *  \param currRep [IN] The repository info.
	 *  \param ClientPki [IN] The instance of PkiClient to use.
	 *  \param TimeOut [IN] TimeOut>0: the connection timeout.
	 *  \param err [OUT] The error.
	 *  \return true on success, false on failure.
	 */
	static bool ConnectToRepository(const PKI_CERT & EntityCert, const RepEntryInfo & currRep, PkiClient * ClientPki, int TimeOut, mString & err);

	/*! \brief This function is used to connect to a Repository.
	 *  \param EntityCert [IN] The certificate to use for the authentication.
	 *  \param RepName [IN] The name of the repository.
	 *  \param RepAddress [IN] The address of the repository.
	 *  \param RepPort [IN] The port of the repository.
	 *  \param RepCert [IN] The certificate of the repository.
	 *  \param ClientPki [IN] The instance of PkiClient to use.
	 *  \param TimeOut [IN] TimeOut>0: the connection timeout.
	 *  \param err [OUT] The error.
	 *  \return true on success, false on failure.
	 */
	static bool ConnectToRepository(const PKI_CERT & EntityCert, const mString & RepName, const mString & RepAddress, unsigned int RepPort, const PKI_CERT & RepCert, PkiClient * ClientPki, int TimeOut, mString & err);


	/*! \brief This function verifies if the public key of a certificate is in rep_path.
	 *  \param rep_path [IN] The list of public keys.
	 *  \param Cert [IN] The certificate to verifiy.
	 *  \return true if Cert is in the list, false is Cert is not known.
	 */
	static bool RepositoryKnowsObject(const STACK_OF(X509_PUBKEY) * rep_path, const PKI_CERT & Cert);

	/*! \brief This function adds the public key of a certificate to rep_path.
	 *  \param rep_path [IN/OUT] The list of public keys.
	 *  \param Cert [IN] The certificate to add.
	 *  \return true on success, false on failure.
	 */
	static bool AddRepositoryToObjectsPath(STACK_OF(X509_PUBKEY) * rep_path, const PKI_CERT & Cert);
private:
	#define MAIL_JOBS_STACK "create table mails (mail_id INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY, mail LONGBLOB NOT NULL);"
	#define MAIL_INSERT "insert into mails (mail) values ('%s');"
	#define MAIL_DELETE "delete from mails where mail_id=%ld;"
	#define MAIL_SELECT "select * from mails;"

	PKI_CERT m_EntityCert;
	mString m_EntityName;
	EntityLog * m_Logging;


	ReadersWriter m_RepositoriesLock;
	mVector<RepEntryInfo> m_Repositories;

	
	ExportedPkiConf m_PkiConf;
	CriticalSection ConfPushLock;
	NewpkiThread hThreadConfigurationPush;
	bool m_PushToAllRep;
	static void ThreadConfigurationPush(const NewpkiThread * Thread, void * param);
	bool Private_PushConfiguration(const NewpkiThread * Thread);

	
	EmailConf m_MailerConf;
	SmtpClient m_Mailer;
	mVector<MailQueueEntry> m_AdminMailQueue;
	CriticalSection m_AdminMailQueueLock;
	NewpkiThread hThreadMailQueue;
	static void ThreadMailQueue(const NewpkiThread * Thread, void * param);
	void LoadMails();
	void RemoveMail(unsigned long id);
	bool InsertMail(MailQueueEntry & q) const;
	mVector<mString> m_AdminMails;

	const X509_ACL_Validator * m_AclValidator;
	SQL_Connection * m_DbConn;
	mVector<PKI_CERT> m_EntityParentCerts;
};

#endif
