/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

namespace Sioux
{
    using Nemerle.Collections;
    
    using System;

    /**
     *  Class representing http session
     */
    public class Session
    {
        /* -- CONSTRUCTORS ------------------------------------------------ */

        /**
         *  Constructs new session
         */
        internal this()
        {
            creation_time = DateTime.Now;
            last_request  = DateTime.Now;
            session_id = SessionID.GetSessionID();
            session_objects = Hashtable();
            is_new_session = true;
            is_valid = true;
            timeout = 10;
        }
     /* UNUSED?
        internal this(cookieless : bool)
        {
            creation_time = DateTime.Now;
            last_request  = DateTime.Now;
            session_id = SessionID.GetSessionID();
            session_objects = Hashtable();
            this.cookieless = cookieless;
            is_new_session = true;
            is_valid = true;
            timeout = 10;
        }
      */
        /* UNUSED?
         *  Constructs session with given id
         
        internal this(session_id : string)
        {
            creation_time = DateTime.Now;
            last_request  = DateTime.Now;
            session_objects = Hashtable();
            this.session_id = session_id;
            is_new_session = true;
            is_valid = true;
            timeout = 10;
        }
        */
        
        /* -- PUBLIC PROPERTIES ------------------------------------------- */

        /**
         *  returns a string containing the unique identifier assigned to this session
         */
        public SessionID : string
        {
            get { session_id }
        }

        /**
         *  Time in minutes, allowed beetwen request before session is invalidated
         *  Value -1 means, that this period is infinite
         *  Value 10 is default
         */
        public Timeout : int
        {
            get { timeout }
            set { timeout = value }
        }

        /**
         *  gets the time when session was created
         */
        public CreationTime : DateTime
        {
            get { creation_time }
        }

        /**
         *  returns time when client last sent request associated with session
         */ 
        public LastAccesedTime : DateTime
        {
            get { last_request }
        }

        /**
         *  eturns true when session was created with last client request 
         */
        public IsNew : bool
        {
            get { is_new_session }
        }

        /**
         *  gets a value indicating if session is stall valid 
         */
        public IsValid : bool
        {
            get { is_valid }
        }

        /**
         * returns a list of all attributes associated with session, where first element of pair is attribute name, and second 
         * is attribute
         */
        public Attributes : list [string * object]
        {
            get
            {
                mutable lst = [];
                def add_to_list (s : string,obj : object) : void
                {
                    lst = (s,obj) :: lst
                }
                session_objects.Iter(add_to_list);
                lst
            }
        }
        /* -- PUBLIC METHODS ---------------------------------------------- */

        /**
         * Invalidates this session
         */
        public Invalidate () : void
        {
            is_valid = false
        }

        /**
         * Adds attribute to session using specified name
         */
        public AddAttribute(name : string,obj : object) : void
        {
            session_objects.Add(name,obj)
        }

        /**
         *  Removes attribute with specified name from session 
         */
        public RemoveAttribute(name : string) : void
        {
            session_objects.Remove(name)
        }

        /**
         *  Returns true if session contains attribute with given name, else returns false
         */
        public ContainsAttribute(name : string) : bool
        {
            session_objects.Contains(name)
        }

        /**
         *  Returns Some(attribute) if session contains attribute with specified name, else returns None
         */
        public GetAttribute(name : string) : option [object]
        {
            session_objects.Get(name);
        }
        
        /* -- INTERNAL METHODS -------------------------------------------- */

        /**
         * Sets a value is_new_session to false
         */
        internal SetIsNotNew() : void
        {
            is_new_session = false;
        }

        /**
         *  Updates last request time
         */
        internal SetAccesTime () : void
        {
            last_request = DateTime.Now;
        }
        
        /* -- PRIVATE FIELDS ---------------------------------------------- */

        private session_id : string;
//        private cookieless : bool;
        private creation_time : DateTime;

        private mutable is_new_session : bool;
        private mutable is_valid : bool;
        private mutable timeout  : int;
        private mutable last_request : DateTime;
        
        private session_objects : Hashtable [string,object];
    }
}
