/*
 * Copyright (c) 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Compiler.Typedtree;
using PT = Nemerle.Compiler.Parsetree;

namespace Nemerle.Compiler
{
  /** Represents a immutable typing context, corresponding to a specific
      place in the source code.

      It is stored with expressions pending to be typed.
    */
  public class LocalContext
  {
    mutable locals : Map [PT.Name, LocalValue];
    mutable is_checked : bool; 


    public FindLocal (name : PT.Name) : option [LocalValue]
    {
      if (name.color == -1)
        locals.Fold (None (), fun (n, v, acc) {
          if (n.idl == name.idl) {
            when (acc.IsSome)
              Message.Warning ($ "more than one hit for $$(\"$(name.idl)\" : dyn)");
            Some (v)
          } else acc
        })
      else
        locals.Find (name)
    }

    public Fold ['a] (acc : 'a, f : PT.Name * LocalValue * 'a -> 'a) : 'a
    {
      locals.Fold (acc, f)
    }

    public GetLocals () : Map [PT.Name, LocalValue]
    {
      locals
    }
                              
    
    /** [false] if we are in the [unchecked { ... }] block. */
    public IsChecked : bool
    {
      get { is_checked }
    }


    public static Empty : LocalContext
    {
      mutable static empty : LocalContext;
      get {
        when (empty == null)
          empty = LocalContext ();
        empty
      }
    }


    // just for Empty above
    private this ()
    {
      locals = NemerleMap ();
      is_checked = true;
    }
    

    private this (other : LocalContext)
    {
      locals = other.locals;
      is_checked = other.is_checked;
    }

    
    public WithChecked (c : bool) : LocalContext
    {
      def l = LocalContext (this);
      l.is_checked = c;
      l
    }

    
    public WithLocal (name : PT.Name, local : LocalValue) : LocalContext
    {
      def l = LocalContext (this);
      l.locals = l.locals.Replace (name, local);
      l
    }
  }
}
