/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using System.Data;
using System.Text.RegularExpressions;
using Nemerle.Collections;
using Npgsql;
using NpgsqlTypes;

using Nemerle.Compiler;

namespace Nemerle.Data.Npgsql
{
  using Helper;
  
  /**
    Define connection string, which will be used by application
    (also for compile-time verification of SQL queries by compiler)
   */
  [Nemerle.MacroUsage (Nemerle.MacroPhase.BeforeInheritance,
                       Nemerle.MacroTargets.Class,
                       Inherited = false, AllowMultiple = true)]
  macro ConfigureConnection (_ : TypeBuilder, con_str : string, name : string = "")
  {
    if (connections.Contains (name))
      Message.FatalError ("Connection with name `" + name + "' is already defined")
    else {
      try {
        def connection = NpgsqlConnection (con_str);
        connection.Open ();
        connections.Add (name, connection);
      }
      catch {
        | e =>
          Message.FatalError ($"connecting to database failed: $e")
      }
    };
  }

  macro ExecuteNonQuery (query : string, conn, con_name : string = "")
  {
    def (query, tpars, pars_init) =
      Tools.ExtractParameters (Nemerle.Macros.ImplicitCTX (), query, ':');

    // create compile-time query to check syntax and types in query
    def mytran = get_connection (con_name).BeginTransaction ();        
    def mycmd = NpgsqlCommand (query, get_connection (con_name), mytran);    

    try { 
      tpars.Iter (fun (name, tvar : Typedtree.TExpr) {
        def (dbtype, dbvalue) = type_representant (tvar.ty.Fix ());
        mycmd.Parameters.Add (name, dbtype).Value = dbvalue;
      });
      // try to execute query chcecking its syntax and typecorrectness
      _ = mycmd.ExecuteNonQuery ()
    }
    catch {
      | e is NpgsqlException =>
        Message.FatalError ("sql query error: " + e.Message)
    }
    finally {
      mytran.Rollback ();
      mycmd.Dispose ();
    };

    <[
      using (querycmd = NpgsqlCommand ($(query : string), $conn))
      {
        { .. $pars_init };
        querycmd.ExecuteNonQuery ();
      }
    ]>
  }

  macro ExecuteScalar (query : string, conn, con_name : string = "")
  {
    def (query, tpars, pars_init) =
      Tools.ExtractParameters (Nemerle.Macros.ImplicitCTX (), query, ':');

    // create compile-time query to check syntax and types in query
    def mytran = get_connection (con_name).BeginTransaction ();          
    def mycmd = NpgsqlCommand (query, get_connection (con_name), mytran);

    mutable col_type = null;
    try {
      tpars.Iter (fun (name, tvar : Typedtree.TExpr) {
        def (dbtype, dbvalue) = type_representant (tvar.ty.Fix ());
        mycmd.Parameters.Add (name, dbtype).Value = dbvalue;
      });

      // try to execute query chcecking its syntax and aquiring names of columns
      def myreader = mycmd.ExecuteReader(CommandBehavior.SchemaOnly);
      def table = myreader.GetSchemaTable ();
      if (table.Rows.Count < 1)
        Message.FatalError ("this query doesn't return any value")
      else 
        col_type = Util.ExprOfQid (table.Rows[0]["DataType"].ToString ());
      myreader.Close ();
    }
    catch {
      | e is NpgsqlException =>
        Message.FatalError ("sql query error: " + e.Message)
    }
    finally {
      mytran.Rollback ();
      mycmd.Dispose ();
    };

    /// final code for entire sql loop
    <[
      using (querycmd = NpgsqlCommand ($(query : string), $conn))
      {
        { .. $pars_init };
        (querycmd.ExecuteScalar () :> $col_type);
      }
    ]>
  }

  macro ExecuteReader (query : string, conn, con_name : string = "")
  {
    def (query, tpars, pars_init) =
      Tools.ExtractParameters (Nemerle.Macros.ImplicitCTX (), query, ':');

    // create compile-time query to check syntax and types in query
    def mytran = get_connection (con_name).BeginTransaction ();                
    def mycmd = NpgsqlCommand (query, get_connection (con_name), mytran);
    try {
      tpars.Iter (fun (name, tvar : Typedtree.TExpr) {
        def (dbtype, dbvalue) = type_representant (tvar.ty.Fix ());
        mycmd.Parameters.Add (name, dbtype).Value = dbvalue;
      });
      // try to execute query chcecking its syntax
      _ = mycmd.ExecuteNonQuery ();
    }
    catch {
      | e is NpgsqlException =>
        Message.FatalError ("sql query error: " + e.Message)
    }
    finally {
      mytran.Rollback ();
      mycmd.Dispose ();
    };

    /// final code for entire sql loop
    <[
      using (querycmd = NpgsqlCommand ($(query : string), $conn)) 
      {
        { .. $pars_init };
        querycmd.ExecuteReader ();
      }
    ]>
  }

  macro ExecuteReaderLoop (query : string, conn, body)
  {
    def (query, tpars, pars_init) =
      Tools.ExtractParameters (Nemerle.Macros.ImplicitCTX (), query, ':');

    // list of definitions of query results inside loop body
    mutable bodyseq = [body];

    // create compile-time query to check syntax and types in query
    def mytran = get_connection ("").BeginTransaction ();
    def mycmd = NpgsqlCommand (query, get_connection (""), mytran);
    try {
      tpars.Iter (fun (name, tvar : Typedtree.TExpr) {
        def (dbtype, dbvalue) = type_representant (tvar.ty.Fix ());
        mycmd.Parameters.Add (name, dbtype).Value = dbvalue;
      });

      // try to execute query chcecking its syntax and aquiring names of columns
      def myreader = mycmd.ExecuteReader(CommandBehavior.SchemaOnly %|
                                         CommandBehavior.SingleRow);
      def table = myreader.GetSchemaTable ();
      mutable col_num = 0;
      foreach (myRow :> DataRow in table.Rows){
        def col_type = myRow["DataType"].ToString ();
        def col_name = myRow["ColumnName"].ToString ();
        def type_suff = 
          if (col_type.StartsWith ("System."))
            col_type.Substring (7)
          else col_type;

        // create runtime variables definition according to extracted types
        bodyseq = <[ def $(col_name : usesite) =
                        reader.$("Get" + type_suff : usesite) ($(col_num : int)) ]>
                    :: bodyseq;
        ++col_num;
      };

      myreader.Close ();
    }
    catch {
      | e is NpgsqlException =>
        Message.FatalError ("sql query error: " + e.Message)
    }
    finally {
      mytran.Rollback ();
      mycmd.Dispose ();
    };

    /// final code for entire sql loop
    <[
      using (querycmd = NpgsqlCommand ($(query : string), $conn)) 
      {
        { .. $pars_init };
        def reader = querycmd.ExecuteReader ();
        while (reader.Read ()) { ..$bodyseq };
        reader.Close ();
      }
    ]>
  }

  module Helper {
    internal connections : Hashtable [string, NpgsqlConnection] = Hashtable ();

    internal get_connection (name : string) : NpgsqlConnection
    {
      match (connections.Get (name)) {
        | Some (c) => c
        | None =>
          if (name == "")
            Message.FatalError ("default connection was not found")
          else
            Message.FatalError ("connection `" + name + "' was not found")
      }
    }

    internal type_representant (t : MType) : NpgsqlDbType * object
    {
      match (t) {
        | MType.Class (tc, []) =>
          match (tc.FullName) {
            | "System.String" | "Nemerle.Core.string" =>
              (NpgsqlDbType.Text, ("string parameter" : object))
            | "System.Int32" | "Nemerle.Core.int" =>
              (NpgsqlDbType.Integer, 234)
            | "System.Boolean" | "Nemerle.Core.bool" =>
              (NpgsqlDbType.Boolean, true)
            | "System.UInt32" | "Nemerle.Core.uint" =>
              (NpgsqlDbType.Integer, 234u)
  //          | "System.Byte" | "Nemerle.Core.byte" =>
  //            (NpgsqlDbType.Byte, 34ub)
            | "System.DateTime" =>
              (NpgsqlDbType.Date, System.DateTime.Now)
            | "System.Decimal" | "Nemerle.Core.decimal" =>
              (NpgsqlDbType.Money, System.Convert.ToDecimal (45.3))
            | "System.Double" | "Nemerle.Core.double" =>
              (NpgsqlDbType.Real, 34.4)
  //          | "System.Guid" =>
  //            (NpgsqlDbType.Guid, System.Guid.NewGuid ())
            | "System.Int16" | "Nemerle.Core.short" =>
              (NpgsqlDbType.Smallint, 34s)
            | "System.UInt16" | "Nemerle.Core.ushort" =>
              (NpgsqlDbType.Smallint, 34us)
            | "System.Int64" | "Nemerle.Core.long" =>
              (NpgsqlDbType.Bigint, 34l)
            | "System.UInt64" | "Nemerle.Core.ulong" =>
              (NpgsqlDbType.Bigint, 34ul)
  //          | "System.Object" | "Nemerle.Core.object" =>
  //            (NpgsqlDbType.Object, null)
  //          | "System.SByte" | "Nemerle.Core.sbyte" =>
  //            (NpgsqlDbType.SByte, 34b)
            | "System.Single" | "Nemerle.Core.float" =>
              (NpgsqlDbType.Real, 34.4f)
            | x =>
              Message.FatalError (x + " type not supported")
          }
        | _ =>
          Message.FatalError ("only basic types supported in sql query")
      }
    }
  }
}
