--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
with Interfaces;
with Ada.Strings.Unbounded;
with System;

with Types;
with Utils;
with Low_Level_Network.Constants; use Low_Level_Network.Constants;

package body Low_Level_Network.Utils is

   function Ioctl (Fd : Integer;
                   Cmd : Interfaces.Unsigned_32;
                   Address : System.Address) return Integer;
   pragma Import (C, Ioctl, "ioctl");
   function Socket (Domain : Integer;
                    Socket_Type : Integer;
                    Protocol : Integer) return Integer;
   pragma Import (C, Socket, "socket");
   function Close (Fd : Integer) return Integer;
   pragma Import (C, Close, "close");

   ---------------
   -- To_String --
   ---------------

   function To_String
     (Network_Address : Network_Address_Type)
      return String
   is
      First : constant String := Network_Address (1)'Img;
      Second : String := Network_Address (2)'Img;
      Third : String := Network_Address (3)'Img;
      Fourth : String := Network_Address (4)'Img;
   begin
      Second (Second'First) := '.';
      Third (Third'First) := '.';
      Fourth (Fourth'First) := '.';
      return First (First'First + 1 .. First'Last) &
        Second & Third & Fourth;
   end To_String;

   ------------------------
   -- To_Network_Address --
   ------------------------

   function To_Network_Address
     (Network_Address : String)
      return Network_Address_Type
   is
      use Ada.Strings.Unbounded;
      Invalid_Address : exception;
      Sub_Strings : constant Standard.Types.String_Array :=
        Standard.Utils.Split (Network_Address, '.');
   begin
      if Sub_Strings'Length /= 4 then
         raise Invalid_Address;
      end if;
      return (Interfaces.Unsigned_8'Value (To_String (Sub_Strings (1))),
              Interfaces.Unsigned_8'Value (To_String (Sub_Strings (2))),
              Interfaces.Unsigned_8'Value (To_String (Sub_Strings (3))),
              Interfaces.Unsigned_8'Value (To_String (Sub_Strings (4))));
   exception
      when Invalid_Address =>
         raise;
      when others =>
         raise Invalid_Address;
   end To_Network_Address;

   -----------------
   -- Get_Address --
   -----------------

   function Get_Address (Interface_Name : String) return String is
   begin
      return To_String (Get_Address (Interface_Name));
   end Get_Address;

   -----------------
   -- Get_Address --
   -----------------

   function Get_Address
     (Interface_Name : String)
      return Network_Address_Type
   is
      C_Fct_Return : Integer;
      Fd : Integer;
      Req_Record : Address_Req_Type;
      Index : Positive := 1;
   begin
      Fd := Socket (PF_INET, SOCK_DGRAM, IPPROTO_IP);
      if Fd = -1 then
         raise Bad_Call;
      end if;
      for I in Interface_Name'Range loop
         Req_Record.Interface_Name (Index) := Interface_Name (I);
         Index := Index + 1;
      end loop;
      Req_Record.Interface_Name (Index) := ASCII.NUL;
      C_Fct_Return := Ioctl (Fd, SIOCGIFADDR, Req_Record'Address);
      if C_Fct_Return = -1 then
         raise Bad_Call;
      end if;
      C_Fct_Return := Close (Fd);
      if C_Fct_Return = -1 then
         raise Bad_Call;
      end if;
      return (Req_Record.Address (3),
              Req_Record.Address (4),
              Req_Record.Address (5),
              Req_Record.Address (6));
   end Get_Address;

   --------------------
   -- Get_Interfaces --
   --------------------

   function Get_Interfaces return Interfaces_Array is separate;

end Low_Level_Network.Utils;
