/* 
 * Copyright (c) 2009, 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; version 2 of the
 * License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */

#include "stdafx.h"
#include "wf_textentry.h"

using namespace MySQL::Forms;

//--------------------------------------------------------------------------------------------------

bool TextEntryImpl::create(::mforms::TextEntry *self, ::mforms::TextEntryType type)
{
  TextEntryImpl ^entry= gcnew TextEntryImpl(self);

  if (entry != nullptr)
  {
    TextBox ^tb= ViewImpl::create<TextBox>(self, entry);
    tb->ForeColor = Color::Black;
    tb->TextChanged += gcnew EventHandler(change);
    tb->KeyDown += gcnew KeyEventHandler(key_pressed);
    tb->Multiline= false;
    switch (type)
    {
    case ::mforms::NormalEntry:
      break;
    case ::mforms::PasswordEntry:
      tb->PasswordChar= '*';
      break;
    case ::mforms::SearchEntry:
      break;
    }
    tb->Size= Size(100, tb->PreferredSize.Height); // DefaultSize is not accessible here. 
    return true;
  }
  return false;
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::set_text(::mforms::TextEntry *self, const std::string &text)
{
  TextEntryImpl^ cb= (TextEntryImpl^)ObjectImpl::FromUnmanaged(self);

  if (cb != nullptr)
  {
    TextBox^ box= cb->get_control<TextBox>();
    box->Text= CppStringToNative(text);

    // Make sure the text box is never made taller than their preferred height.
    Size size = box->PreferredSize;
    size.Width = box->MaximumSize.Width;
    box->MaximumSize = size;
  }
}

//--------------------------------------------------------------------------------------------------

std::string TextEntryImpl::get_text(::mforms::TextEntry *self)
{
  TextEntryImpl^ cb= (TextEntryImpl^)ObjectImpl::FromUnmanaged(self);

  if (cb != nullptr)
  {
    return NativeToCppString(cb->get_control<TextBox>()->Text);
  }
  return "";
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::set_max_length(::mforms::TextEntry *self, int length)
{
  TextEntryImpl^ cb= (TextEntryImpl^)ObjectImpl::FromUnmanaged(self);

  if (cb != nullptr)
  {
    cb->get_control<TextBox>()->MaxLength= length;
  }
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::change(Object^ sender, EventArgs^ args)
{
  TextBox^ tb= (TextBox^) sender;

  if (tb->Tag != nullptr)
  {
    ::mforms::TextEntry* box= ViewImpl::get_backend_control<::mforms::TextEntry>(tb);
    if (box != NULL)
      box->callback();
  }
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::key_pressed(Object^ sender, KeyEventArgs^ args)
{
  TextBox^ tb= (TextBox^) sender;

  if (tb->Tag != nullptr)
  {
    ::mforms::TextEntry* box= ViewImpl::get_backend_control<::mforms::TextEntry>(tb);
    if (box != NULL)
    {
      switch (args->KeyCode)
      {
      case Keys::Return:
        box->action(mforms::EntryActivate);
        args->SuppressKeyPress= true;
        break;
      case Keys::Up:
        if (args->Control)
            box->action(mforms::EntryCKeyUp);
        else
            box->action(mforms::EntryKeyUp);
        args->SuppressKeyPress= true;
        break;
      case Keys::Down:
        if (args->Control)
            box->action(mforms::EntryCKeyDown);
        else
            box->action(mforms::EntryKeyDown);
        args->SuppressKeyPress= true;
        break;
      }
    }
  }
}

//--------------------------------------------------------------------------------------------------

void TextEntryImpl::set_read_only(::mforms::TextEntry *self, bool flag)
{
  TextEntryImpl^ cb= (TextEntryImpl^) ObjectImpl::FromUnmanaged(self);

  if (cb != nullptr)
    cb->get_control<TextBox>()->ReadOnly= flag;
}

//--------------------------------------------------------------------------------------------------

TextEntryImpl::TextEntryImpl(::mforms::TextEntry *text)
  : ViewImpl(text)
{
}

//--------------------------------------------------------------------------------------------------
