/*
 *   idevice.c
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                         
 *   blwtt.h
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
//  This class encapsulates everything needed to track call restrictions
//  for an individual device such as modem, fax, or voice.  This class is
//  built up using an array of the UnsuccessfulRecord class.  An overview
//  of operations can be found in mwbl.txt.
//
//  All public functions in this class must be protected so that only
//  one operates at a time.  This is the responsibility of the calling code.
//  This will prevent problems between the viewer and the devices using
//  blacklist services.  The case where a reset occurs while a device is
//  dialing (has a dial handle checked out) is allowed for in the DialResults
//  function.
//
//
//
//****************************************************************************
#define __need_time_t
#include <time.h>
#include <idevice.h>  // Header file for this class
#include <mwblapi.h>  // The API header file needed for information structures
#include <blobject.h> // The WTT settings are imbeded in the main object
#include <string.h>
#include <mwwttbl.h>  // For the WT_COUNTRY_CANADA define
#include <stdio.h>    // debug support

//------- external references ------------------------------------------------
extern BLobject BL; //reference the main object which includes the WTTsettings

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  Constructor
//
//     This constructor initializes all overhead variables.  The array of
//     UnsuccessfulRecord objects is initialized by it's constructor.
//
//     Params:   none
//
//     Returns:  none
//-----------------------------------------------------------------------------
void devInitDevice(blDevice *pDev)
{

  pDev->bDeviceReserved = FALSE;
  pDev->ulDeviceType = 0;
  pDev->szDeviceName[0] = '\0';
  pDev->bInitialized = TRUE;
  pDev->LastDeviceDialTime = 0;
  pDev->ulDialHandleInUse  = NO_DIAL_HANDLE; // clear handle in use
  pDev->ulDialTypeInUse    = MANUAL_DIAL;    // clear the dial type
} 

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  Reset
//
//     This function resets call restrictions for a device, it does not reset
//     the high level information about the device such as device type and
//     name which is set during registration.
//
//
//     Params:   none
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG devReset (blDevice *pDev)
{
  ULONG ulError;

  MW_SYSLOG_2(TRACE_MWMBL,"idevice::devReset entry pDev %p\n",pDev);

  ulError=devInternalReset(pDev);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWMBL,"idevice::devReset, ERROR call to InternalReset returned ulError %lx\n", ulError);
  }
  
  return BL_SUCCESSFUL;
} 

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  Register
//
//     This function needs to be called first to register the device.  The
//     registration includes the type of device (modem, fax, voice) and
//     the name of the device to be reported with information about the
//     current status.
//
//
//     Params:   ulLocalDeviceType - The device type defined above.
//
//               szDeviceName      - The name that the device is known by
//                                   in the system such as in the config.
//                                   This will be used by the blacklist viewer
//                                   in identifying status for a particular
//                                   device/line.
//
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG devRegister(blDevice *pDev, ULONG ulLocalDeviceType, char FAR * lpszDeviceName)
{
  ULONG ulError;
  
  MW_SYSLOG_4(TRACE_MWMBL,"idevice::devRegister, entry pDev %p, DeviceType %lx  DeviceName %s\n", pDev, ulLocalDeviceType, lpszDeviceName);
  
  // Check to make sure the device memory is initialized
  if (pDev->bInitialized == FALSE) {
    pDev->bDeviceReserved = FALSE;
    pDev->ulDeviceType = 0;
    pDev->szDeviceName[0] = '\0';
    
    ulError=devReset(pDev); //This will also get the UnsuccessfulRecord class
    if (ulError)
      return ulError;
  }

  // To prevent two calls to this function to proceed in parallel, we will
  // increment the reserved flag instead of setting it to true.  Only the
  // tread that sees it as a '1' will get the registration.
  pDev->bDeviceReserved++;

  // if there is a race, only one of the treads will see this as a '1'
  if (pDev->bDeviceReserved > 1) {
    return BLERR_DEVICE_REGISTRATION_FAILED;
  }
  
  // Now let's register the device type checking to see if it is proper
  switch (ulLocalDeviceType) {
  case DEVICE_MODEM:
  case DEVICE_FAX:
    pDev->ulDeviceType=ulLocalDeviceType;
    break;
  default:
    {
      MW_SYSLOG_2(TRACE_MWMBL,"idevice::devRegister, ERROR invalid device type %lx\n", ulLocalDeviceType);
      return BLERR_INVALID_DEVICE_TYPE;
    }
  }
  
  // Next let's record the device name
  if (strlen(lpszDeviceName) > MAX_DEVICE_NAME_SIZE) {
    MW_SYSLOG_1(TRACE_MWMBL,"idevice::devRegister, ERROR failed with the szDeviceName too large\n");
    pDev->bDeviceReserved = FALSE;  // release the device
    return BLERR_DEVICE_NAME_TOO_LARGE;
  }
  
  strcpy (pDev->szDeviceName, lpszDeviceName);
  return BL_SUCCESSFUL;
} 

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  UnRegister
//
//     This function removes the registration information and resets member
//     memory.
//
//     Params:   none
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG devUnregister (blDevice *pDev)
{
  ULONG ulError;

  MW_SYSLOG_2(TRACE_MWMBL,"idevice::devUnregister entry, pDev %p\n",pDev);
  
  // Clear the device type
  pDev->ulDeviceType = 0;

  // Clear out the device name
  pDev->szDeviceName[0] = '\0';

  // Now, reset everything else
  ulError=devInternalReset(pDev);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWMBL,"idevice::devUnregister, ERROR call to devInternalReset returned ulError %lx\n", ulError);
    return ulError;
  }
  
  // Last, unreserved this instance
  pDev->bDeviceReserved = FALSE;
  return BL_SUCCESSFUL;
} 

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  RequestDialPermission
//
//     This function is used to request permission to dial.  Permission is
//     granted if BL_DIAL_GRANTED is returned.  In this case, a valid
//     ulDialHandle will be returned which needs to be used in reporting
//     DialResults.
//
//     In the case of a manual dial, permission will always be granted.
//     This is used to obtain the ulDialHandle which can then be used
//     to clean calling restrictions for the number if the manual dial
//     succeeded.
//
//
//     Params:   ulDialType     - The type of dial:  MANUAL_DIAL or AUTOMATIC_DIAL.
//               pszPhoneNumber - A pointer to the phone number that dialing is requested for.
//               lpulDialStatus - A pointer to return the status of the request:
//                                BL_DIAL_GRANTED       - OK to dial
//                                BL_NUMBER_BLACKLISTED - permission denied
//                                BL_NUMBER_DELAYED     - permission denied for
//                                                        lpulDelayTime more seconds
//
//               lpulDialHandle - A pointer to the returned dial handle.
//               lpulDelayTime  - A pointer to the returned delay time in seconds.
//
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from the list in mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG devRequestDialPermission (blDevice *pDev, ULONG ulDialType,
				char  FAR *lpszPhoneNumber,
				ULONG FAR *lpulDialStatus,
				ULONG FAR *lpulDialHandle,
				ULONG FAR *lpulDelayTime)
{ 
  ULONG ulError;  
  time_t CurrentTime, ElapsedTime;
  
  MW_SYSLOG_4(TRACE_MWMBL,"idevice::devRequestDialPermission, entry pDev %p, phone number %s with DialType %lx\n",pDev, lpszPhoneNumber, ulDialType);

  // Check to make sure the device memory is initialized
  if (pDev->bInitialized == FALSE) {
    MW_SYSLOG_1(TRACE_MWMBL,"idevice::devRequestDialPermission, ERROR device not initialized\n");
    return BLERR_DEVICE_NOT_INITIALIZED;
  }
  
  // Check to see if a dial handle is currently in use pending a results call
  if (!(pDev->ulDialHandleInUse == NO_DIAL_HANDLE)) {
    MW_SYSLOG_1(TRACE_MWMBL,"idevice::devRequestDialPermission, ERROR device has an outstanding dial handle\n");
    return BLERR_DIAL_HANDLE_IN_USE;
  }
  
  // Check to make sure the dial type is set to a good value
  if (! ((ulDialType == MANUAL_DIAL) || (ulDialType == AUTOMATIC_DIAL)) ) {
    MW_SYSLOG_1(TRACE_MWMBL,"idevice::devRequestDialPermission, ERROR ulDialType is invalid\n");
    return BLERR_INCORRECT_DIAL_TYPE;
  }
  
  // Get the current time
  time(&CurrentTime);


  MW_SYSLOG_3(TRACE_MWMBL,"idevice::devRequestDialPermission, pWTT %p, pWTT->bBlacklistingEnabled %x\n",&BL.WTT,BL.WTT.bBlacklistingEnabled);
  //----------------------------------------------------------------------------
  // Now we can begin the permission algorithm:

  //////////////////////////////////////////////////////////////////////////////
  //----------------------------------------------------------------------------
  // Special code for Japan - disable data and fax modem calls to the
  // emergency numbers 110, 119.  In Japan, to get an outside line from
  // a PBX '0' is used so 0110 and 0119 will also be disabled.
  if (BL.WTT.bUseSpecialAlgorithm &&
      (BL.WTT.ulCountryNumber == WT_COUNTRY_JAPAN)) {
    if ((pDev->ulDeviceType == DEVICE_MODEM) || (pDev->ulDeviceType == DEVICE_FAX)) {
      if ((strcmp (lpszPhoneNumber,  "110") == 0) ||
          (strcmp (lpszPhoneNumber,  "119") == 0) ||
          (strcmp (lpszPhoneNumber, "0110") == 0) ||
          (strcmp (lpszPhoneNumber, "0119") == 0))
      {
        *lpulDialStatus = BL_NUMBER_BLACKLISTED;
        *lpulDialHandle = NO_DIAL_HANDLE;
        *lpulDelayTime  = 0;

        return BL_SUCCESSFUL;
      }
    }
  }
  //-- end special code for Japan ----------------------------------------------
  //////////////////////////////////////////////////////////////////////////////


  // If there are no blacklist settings for the country then OK the dial
  if (BL.WTT.bBlacklistingEnabled == FALSE) {
    *lpulDialStatus = BL_DIAL_GRANTED;
    *lpulDialHandle = NO_RECORD;

    MW_SYSLOG_1(TRACE_MWMBL,"idevice::devRequestDialPermission returned BL_DIAL_GRANTED because blacklisting is not enabled\n");
    return BL_SUCCESSFUL;
  }

  // See if a special algorithm is suppose to be used in the country
  if (BL.WTT.bUseSpecialAlgorithm) {
    // Currently there is only special code for Canada and Japan
    if ((BL.WTT.ulCountryNumber != WT_COUNTRY_CANADA) &&
        (BL.WTT.ulCountryNumber != WT_COUNTRY_JAPAN)) {
      MW_SYSLOG_1(TRACE_MWMBL,"idevice::devRequestDialPermission, ERROR no special code is available\n");
      return BLERR_NO_SPECIAL_CODE_AVAILABLE;
    }
  }

  // If the country (Austria) is only looking for a maximum total number of
  // unsuccessful attempts and does not care about status on individual numbers
  // then map all numbers in to one dummy number to run through the algorithm.
  if (BL.WTT.bDoNotDistiguishPhoneNumbers) {
    strcpy (lpszPhoneNumber, STANDARD_NUMBER);
  }

  // Now we need to get the existing dial handle or a new one.
  ulError=devFindDialHandle(pDev, lpszPhoneNumber, ulDialType, lpulDialHandle);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWMBL,"idevice::devRequestDialPermission, ERROR call to FindPhoneNumber returned ulError %lx\n", ulError);
    return ulError;
  } else {
    MW_SYSLOG_2(TRACE_MWMBL,"idevice::devRequestDialPermission, devFindDialHandle returns ulDialHandle %lx\n",*lpulDialHandle);
  }

  // Is the number blacklisted?
  if (*lpulDialHandle != NO_RECORD) {
    if (failIsNumberBlacklisted(&pDev->Unsuccessful[((int)*lpulDialHandle)-1])) {
      *lpulDialStatus = BL_NUMBER_BLACKLISTED;
      *lpulDialHandle = NO_DIAL_HANDLE;
      *lpulDelayTime  = 0;
      
      MW_SYSLOG_1(TRACE_MWMBL,"idevice::devRequestDialPermission returned BL_NUMBER_BLACKLISTED\n");
      return BL_SUCCESSFUL;
    }
  }

  // All calls must meet the BL.WTT.ulAllCallDelay time
  ElapsedTime = CurrentTime - pDev->LastDeviceDialTime;
  if (BL.WTT.ulAllCallDelay > (ULONG)ElapsedTime) {
    // Reject the call as delayed
    *lpulDialStatus = BL_NUMBER_DELAYED;
    *lpulDialHandle = NO_DIAL_HANDLE;
    *lpulDelayTime  = BL.WTT.ulAllCallDelay - ElapsedTime;

    MW_SYSLOG_2(TRACE_MWMBL,"idevice::devRequestDialPermission BL.WTT.ulAllCallDelay %lx\n",BL.WTT.ulAllCallDelay);
    MW_SYSLOG_2(TRACE_MWMBL,"idevice::devRequestDialPermission returned BL_NUMBER_DELAYED for %lx seconds\n", *lpulDelayTime);
    return BL_SUCCESSFUL;
  }

  // If the phone number was not found for a manual dial, the dial handle will not point
  // to a UnsuccessfulRecord.  Handle this now by taking care of manual dials
  if (ulDialType == MANUAL_DIAL) {
    pDev->ulDialHandleInUse = *lpulDialHandle;
    pDev->ulDialTypeInUse = ulDialType;
    *lpulDialStatus = BL_DIAL_GRANTED;
    *lpulDelayTime = 0;

    MW_SYSLOG_1(TRACE_MWMBL,"idevice::devRequestDialPermission returned BL_DIAL_GRANTED because of manual dial\n");
    return BL_SUCCESSFUL;
  }

  // At this point, the call is auto dial and we have a dial handle which may
  // point to a previously existing UnsuccessfulRecord or a new one.  Regardless,
  // we will now pass the request to the UnsuccessfulRecord which is handling
  // this call to resolve dial permission.
  ulError=failRequestDialPermission(&pDev->Unsuccessful[((int)*lpulDialHandle)-1],CurrentTime,
				lpulDialStatus, lpulDelayTime);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWMBL, "idevice::devRequestDialPermission, ERROR call to failRequestDialPermission returned ulError %lx\n", ulError);
    return ulError;
  }

  // If the request was granted, then record the info so that the dial handle
  // is in use.
  if (*lpulDialStatus == BL_DIAL_GRANTED) {
    pDev->ulDialHandleInUse = *lpulDialHandle;
    pDev->ulDialTypeInUse = ulDialType;
    MW_SYSLOG_1(TRACE_MWMBL,"idevice::devRequestDialPermission returned BL_DIAL_GRANTED\n");
  }

  MW_SYSLOG_3(TRACE_MWMBL,"idevice::devRequestDialPermission returned DialStatus %lx for DialHandle %lx\n", *lpulDialStatus, *lpulDialHandle);
  return BL_SUCCESSFUL;
}

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  DialResults
//
//     This function is used to handle the results of a dial from the list
//     in mwblapi.h.  For manual dials, unsuccessful results do not change any
//     phone number status, however, a successful result will clear call
//     restrictions for auto-dialing.
//
//
//
//     Params:   lpulDialHandle - The dial handle from RequestDialPermission.
//               ulDialStatus   - The dial status defined in mwblapi.h as
//                                   SUCCESSFUL
//                                   NO_DIAL_TONE
//                                   LINE_BUSY
//                                   USER_ABORT
//
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from the list in mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG devDialResults (blDevice *pDev, ULONG ulDialHandle, ULONG ulDialStatus)
{
  ULONG ulError;
  time_t CurrentTime;

  MW_SYSLOG_3(TRACE_MWMBL,"idevice::devDialResults entry pDev %p, DialHandle %lx\n",pDev,ulDialHandle);

  // Check to make sure the device memory is initialized
  if (pDev->bInitialized == FALSE) {
    MW_SYSLOG_1(TRACE_MWMBL,"idevice::devDialResults, ERROR device not initialized\n");
    return BLERR_DEVICE_NOT_INITIALIZED;
  }
  
  
  // Check to see if a dial handle is currently in use pending a results call
  if (pDev->ulDialHandleInUse == NO_DIAL_HANDLE) {
    MW_SYSLOG_1(TRACE_MWMBL,"idevice::devDialResults, WARNING received a call with ulDialHandleInUse == NO_DIAL_HANDLE\n");

    // Because it is possible for a utility to reset the blacklister while a
    // dial is occuring, this error will not be flagged as a problem and this
    // call to DialResults will be tossed.
    return BL_SUCCESSFUL;
  }
  
  // Check to see if the dial handle matches the in use stored one
  if (pDev->ulDialHandleInUse != ulDialHandle) {
    // The given dial handle does not match the stored dial handle.  Assume
    // that the calling routine did not correctly store the dial handle
    // after permission to dial was given to it.
    MW_SYSLOG_1(TRACE_MWMBL,"idevice::devDialResults, ERROR incorrect dial handle\n");
    return BLERR_INCORRECT_DIAL_HANDLE;
  }
  
  //----------------------------------------------------------------------------
  //OK, at this point we can proceed to handling the dial results
  
  // Get the current time
  time (&CurrentTime);
  
  // Record this time as the last dial time
  pDev->LastDeviceDialTime = CurrentTime;


  // see if the dial handle is actually associated to memory
  if (ulDialHandle == NO_RECORD) {
    // It was a manual call with no previous auto-dialed unsuccessful counts
    // in memory.  Therefore there is only clean up to do now.
    pDev->ulDialHandleInUse = NO_DIAL_HANDLE;
    return BL_SUCCESSFUL;
  }

  // At this point the dial handle matches the one issued by RequestDialPermission,
  // and it should be a good index(+1) into the Unsuccessful array.  First, if the
  // call was successful then we can toss the phone number out of the unsuccessful
  // list.
  if (ulDialStatus == SUCCESSFUL) {
    // Delete the UnsuccessfulRecord
    ulError=failReset(&pDev->Unsuccessful[((int)ulDialHandle)-1]);
    if (ulError) {
      MW_SYSLOG_2(TRACE_MWMBL,"idevice::devDialResults, ERROR call to failReset returned with ulError %lx\n", ulError);
      return ulError;
    }
    pDev->ulDialHandleInUse = NO_DIAL_HANDLE;
    return BL_SUCCESSFUL;
  }
  
  // Now if the call was a manual call, there is nothing else to do.
  if (pDev->ulDialTypeInUse == MANUAL_DIAL) {
    pDev->ulDialHandleInUse = NO_DIAL_HANDLE;
    return BL_SUCCESSFUL;
  }

  // If the result was a user abort, then we may not have to do anything further.
  if ((ulDialStatus == USER_ABORT) && !(BL.WTT.bUserAbortIsUnsuccessful)) {
    pDev->ulDialHandleInUse = NO_DIAL_HANDLE;
    return BL_SUCCESSFUL;
  }

  // OK, we have a auto-dialed call and it was unsuccessful.  What we need to do
  // next depends on the type of blacklisting being used.  All results that
  // were not already handled are assumed a type of unsuccessful.  At this
  // point we will pass the dial results to the UnsuccessfulRecord handling
  // the call.

  ulError=failDialResults(&pDev->Unsuccessful[((int)ulDialHandle)-1],ulDialStatus, CurrentTime);
  pDev->ulDialHandleInUse = NO_DIAL_HANDLE;

  if (ulError) {
    MW_SYSLOG_2 (TRACE_MWMBL,"idevice::devDialResults, ERROR call to failDialResults returned with ulError %lx\n", ulError);
    return ulError;
  }
  return BL_SUCCESSFUL;
} 

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  GetInfo
//
//     This function is used to get information about the call restrictions
//     in effect for a device.
//
//
//     Params:   lpBLsettings   - A pointer to where to return the settings
//                                as defined in mwbiapi.h
//
//
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from the list in mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG devGetInfo  (blDevice *pDev, struct BL_Settings FAR *lpBLsettings)
{
  // Check to make sure the device handle is reserved (registered)
  if (pDev->bDeviceReserved == FALSE) {
    MW_SYSLOG_1(TRACE_MWMBL,"idevice::devGetInfo, ERROR device is not registered\n");
    return BLERR_DEVICE_HANDLE_NOT_IN_USE;
  }

  // Check to make sure the device memory is initialized
  if (pDev->bInitialized == FALSE) {
    MW_SYSLOG_1(TRACE_MWMBL,"idevice::devGetInfo, ERROR device not initialized\n");
    return BLERR_DEVICE_NOT_INITIALIZED;
  }
  

  // OK, let's return the info
  lpBLsettings->ulCountryNumber = BL.WTT.ulCountryNumber;

  strcpy (lpBLsettings->szDeviceName, pDev->szDeviceName);

  lpBLsettings->ulDeviceType = pDev->ulDeviceType;

  lpBLsettings->ulMaxPhoneNumbersTracked = MAX_BL_NUMBER_LIST;


  lpBLsettings->bBlacklistingEnabled = BL.WTT.bBlacklistingEnabled;

  lpBLsettings->bUseSpecialAlgorithm = BL.WTT.bUseSpecialAlgorithm;

  lpBLsettings->bUseTimeWindowing = BL.WTT.bUseTimeWindowing;

  lpBLsettings->bUserAbortIsUnsuccessful = BL.WTT.bUserAbortIsUnsuccessful;

  lpBLsettings->bBlacklistAfterFirstSeries = BL.WTT.bBlacklistAfterFirstSeries;

  lpBLsettings->bBlacklistAfterSecondSeries = BL.WTT.bBlacklistAfterSecondSeries;

  lpBLsettings->bDoNotDistiguishPhoneNumbers = BL.WTT.bDoNotDistiguishPhoneNumbers;

  lpBLsettings->ulAllCallDelay = BL.WTT.ulAllCallDelay;

  lpBLsettings->ulFirstRetryDelay = BL.WTT.ulFirstRetryDelay;

  lpBLsettings->ulUnsuccessfulRetryDelay = BL.WTT.ulUnsuccessfulRetryDelay;

  lpBLsettings->ulMaxUnsuccessfulCount = BL.WTT.ulMaxUnsuccessfulCount;

  lpBLsettings->ulTimeWindow = BL.WTT.ulTimeWindow;

  lpBLsettings->ulSeriesToSeriesDelay = BL.WTT.ulSeriesToSeriesDelay;

  return BL_SUCCESSFUL;
} 

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  GetNumberList
//
//     This function is used to get a list of numbers with call restrictions.
//     Memory must be provided by the calling routine.  This function will
//     fill in up to ulMaxListSize number of NumberListItem structures in
//     the specified array with call restricted information.  The first
//     structure in the array with a NULL phone number ends the list.
//
//
//
//     Params:   ulMaxListSize  - The size of the array pointed to by lpNumberList.
//                                The maximum number of phone numbers tracked.
//
//               lpNumberList   - A pointer to the array where call restriction
//                                information needs to be returned.
//
//
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from the list in mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG devGetNumberList (blDevice *pDev, ULONG ulMaxListSize,
			struct BL_NumberListItem FAR *lpNumberList)
{
  ULONG ulError;
  time_t CurrentTime;
  USHORT usNumberListIndex, usRecordIndex;
  
  // Check to make sure the device handle is reserved (registered)
  if (pDev->bDeviceReserved == FALSE) {
    MW_SYSLOG_1(TRACE_MWMBL,"idevice::devGetNumberList, ERROR device is not registered\n");
    return BLERR_DEVICE_HANDLE_NOT_IN_USE;
  }
  
  // Check to make sure the device memory is initialized
  if (pDev->bInitialized == FALSE) {
    MW_SYSLOG_1(TRACE_MWMBL,"idevice::devGetNumberList, ERROR device not initialized\n");
    return BLERR_DEVICE_NOT_INITIALIZED;
  }
  
  // Check to make sure enough memory is allocated
  if (ulMaxListSize < MAX_BL_NUMBER_LIST)
  {
    MW_SYSLOG_2(TRACE_MWMBL,"idevice::devGetNumberList, ERROR MaxListSize %lx is not large enough\n", ulMaxListSize);
    return BLERR_NUMBER_LIST_TOO_SMALL;
  }
  
  
  // OK, let's find out what's going on with the device.
  
  // Get the current time
  time (&CurrentTime);
  
  usNumberListIndex = 0;
  
  for (usRecordIndex = 0; usRecordIndex < MAX_BL_NUMBER_LIST; usRecordIndex++) {
    // A blank phone number must mark the end of the number list
    (lpNumberList+usNumberListIndex)->szPhoneNumber[0] = '\0';

    // if the phone number stored in the unsuccessful array is blank, skip to the next location
    if (! failIsPhoneNumber(&pDev->Unsuccessful[usRecordIndex],"")) {
      // OK, we have a unsuccessful phone number, let's have the UnsuccessfulRecord class
      // fill in the details.
      ulError=failGetNumberInfo(&pDev->Unsuccessful[usRecordIndex],lpNumberList+usNumberListIndex,
				CurrentTime);
      
      // It is possible for a phone number in the list to drop off the list while
      // information is being gathered about it.  This would be the case for old
      // times in a time window environment.
      if (ulError == BLERR_PHONE_NUMBER_IS_BLANK)
        continue;  //don't increment the usNumberListIndex
      
      if (ulError) {
        MW_SYSLOG_2(TRACE_MWMBL,"idevice::devGetNumberList, ERROR call to failGetNumberInfo returned ulError %lx\n", ulError);
        return ulError;
      }
      
      usNumberListIndex++;
    }
  }
  
  return BL_SUCCESSFUL;
  
} 

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  InternalReset
//
//     This function resets call restrictions for a device, it does not reset
//     the high level information about the device such as device type and
//     name which is set during registration.  This function is called
//     by other functions which are protected by a semiphore.
//
//
//     Params:   none
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from mwblapi.h
//
//-----------------------------------------------------------------------------
ULONG devInternalReset (blDevice *pDev)
{
  ULONG ulError;
  USHORT usRecordIndex;

  MW_SYSLOG_2(TRACE_MWMBL,"idevice::devInternalReset entry pDev %p\n",pDev);

  pDev->bInitialized = FALSE;  // mark this device as uninitialized

  // loop through and reset all the records for this device
  for (usRecordIndex = 0; usRecordIndex < MAX_BL_NUMBER_LIST; usRecordIndex++) {
    ulError = failReset(&pDev->Unsuccessful[usRecordIndex]);
    if (ulError) {
      MW_SYSLOG_3(TRACE_MWMBL,"idevice::devInternalReset, ERROR call to failReset returned ulError %lx for ulRecordIndex %x\n", ulError, usRecordIndex);
      return ulError;
    }
  }
  
  pDev->LastDeviceDialTime = 0;              // clear time used in all call delays
  pDev->ulDialHandleInUse  = NO_DIAL_HANDLE; // clear handle in use, zero is a valid handle
  pDev->ulDialTypeInUse    = MANUAL_DIAL;    // clear the dial type
  pDev->bInitialized = TRUE;                 // mark this device as initialized
  return BL_SUCCESSFUL;
} 

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//
//  FindDialHandle
//
//     This function searches the UnsuccessfulArray to see if a phone number
//     is already listed.  If it is, it returns the handle to that UnsuccessfulRecord.
//     If it can not find the phone number, it returns the first available free dial
//     handle or reclaims the oldest.
//
//     This function assumes that the phone number string has already been cleaned
//     of extraneous characters before calling this function.
//
//
//     Params:   lpszPhoneNumber - The phone number string to clean.
//               ulDialType      - The dial type, manual or automatic
//               lpulDialHandle  - return location for the dial handle.
//
//
//     Returns:  ULONG Successful:  0
//                     Error:       BLERR_.... from mwblapi.h
//
//---private-------------------------------------------------------------------
ULONG devFindDialHandle (blDevice *pDev, char FAR *lpszPhoneNumber, ULONG ulDialType,
                              ULONG FAR *lpulDialHandle)
{
  ULONG ulError;
  time_t OldestTimeStamp, TempTimeStamp;
  USHORT usOldestIndex, usIndex;  // used as an array index, the compiler expects 16 bits

  MW_SYSLOG_2(TRACE_MWMBL,"idevice::devFindDialHandle entry pDev %p\n",pDev);
  
  // Check to make sure the device memory is initialized
  if (pDev->bInitialized == FALSE) {
    MW_SYSLOG_1(TRACE_MWMBL,"idevice::devFindDialHandle, ERROR device not initialized\n");
    return BLERR_DEVICE_NOT_INITIALIZED;
  }
  
  // Assume that error checking has already been done on ulDeviceHandle in the calling routine
  
  for (usIndex = 0; usIndex < MAX_BL_NUMBER_LIST; usIndex++) {
    if (failIsPhoneNumber(&pDev->Unsuccessful[usIndex],lpszPhoneNumber) ) {
      // the phone numbers compare, return this index as the dial handle
      *lpulDialHandle = (ULONG) usIndex + 1; //The dial handle is a 1 based value, not a zero based index
      return BL_SUCCESSFUL;
    }
  }
  
  // If we got this far then the phone number was not found.  If this is a manual
  // dial then we do not need a location, else locate an available location and
  // create a new dial handle.
  
  if (ulDialType == MANUAL_DIAL) {
    *lpulDialHandle = NO_RECORD;
    return BL_SUCCESSFUL;
  }

  for (usIndex = 0; usIndex < MAX_BL_NUMBER_LIST; usIndex++) {
    if (failIsPhoneNumber(&pDev->Unsuccessful[usIndex],"")) {
      // If the phone number is blank then it is available
      *lpulDialHandle = (ULONG) usIndex + 1; //The dial handle is a 1 based value, not a zero based index
      
      // Record the phone number to create a new record.
      ulError=failMakeNewRecord(&pDev->Unsuccessful[usIndex],lpszPhoneNumber);
      if (ulError) {
        MW_SYSLOG_2(TRACE_MWMBL,"idevice::devFindDialHandle, ERROR call to failMakeNewRecord returned ulError %lx\n", ulError);
        return ulError;
      }
      return BL_SUCCESSFUL;
    }
  }
  
  // If we got this far then no handles are available to pass out.  Find the
  // oldest entry and replace it.  This loop assumes that all locations are
  // in use which should have been verified before this point.
  
  OldestTimeStamp = 0;
  usOldestIndex   = 0;
  
  for (usIndex = 0; usIndex < MAX_BL_NUMBER_LIST; usIndex++) {
    TempTimeStamp = failFindLatestTimeStamp(&pDev->Unsuccessful[usIndex]);

    if (TempTimeStamp > OldestTimeStamp) {
      OldestTimeStamp = TempTimeStamp;
      usOldestIndex = usIndex;
    }
  }
  
  // Assign the oldest one
  *lpulDialHandle = (ULONG) usOldestIndex + 1;  //The dial handle is a 1 based value, not a zero based index
  
  
  // Reclaim the oldest one
  ulError = failMakeNewRecord(&pDev->Unsuccessful[usOldestIndex],lpszPhoneNumber);
  if (ulError) {
    MW_SYSLOG_2(TRACE_MWMBL,"idevice::devFindDialHandle, ERROR call to failMakeNewRecord returned ulError %lx\n", ulError);
    return ulError;
  }
  
  return BL_SUCCESSFUL;
  
} 

ULONG devIsDeviceReserved(blDevice *pDev) {
  return pDev->bDeviceReserved;
}
  
