//=========================================================
//  MusE
//  Linux Music Editor
//    $Id: filedialog.cpp,v 1.1 2002/01/30 14:54:03 muse Exp $
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#include <errno.h>

#include <qwidget.h>
#include <qurl.h>
#include <qpixmap.h>
#include <qmessagebox.h>
#include <qtoolbutton.h>

#include "filedialog.h"
#include "fdialogbuttons.h"
#include "globals.h"

//---------------------------------------------------------
//   globalToggled
//---------------------------------------------------------

void MFileDialog::globalToggled(bool flag)
      {
      if (flag) {
            buttons->userButton->setOn(!flag);
            buttons->projectButton->setOn(!flag);
            setDir(museGlobalShare+"/"+baseDir);
            }
      }

//---------------------------------------------------------
//   userToggled
//---------------------------------------------------------

void MFileDialog::userToggled(bool flag)
      {
      if (flag) {
            buttons->globalButton->setOn(!flag);
            buttons->projectButton->setOn(!flag);
            setDir(museUser+"/"+baseDir);
            }
      }

//---------------------------------------------------------
//   projectToggled
//---------------------------------------------------------

void MFileDialog::projectToggled(bool flag)
      {
      if (flag) {
            buttons->globalButton->setOn(!flag);
            buttons->userButton->setOn(!flag);
            setDir(museProject+"/"+baseDir);
            }
      }

//---------------------------------------------------------
//   MFileDialog
//---------------------------------------------------------

MFileDialog::MFileDialog(const QString& dir,
   const QString& filter = QString::null, QWidget* parent = 0, bool writeFlag = false)
   : QFileDialog(".", filter, parent, 0, true)
      {
      showButtons = false;
      if (dir[0] == '/') {
            buttons = 0;
            setDir(dir);
            }
      else {
            baseDir = dir;
            showButtons = true;
            buttons = new FileDialogButtons(this, "fdialogbuttons");
            addLeftWidget(buttons);
            connect(buttons->globalButton, SIGNAL(toggled(bool)), SLOT(globalToggled(bool)));
            connect(buttons->userButton, SIGNAL(toggled(bool)), SLOT(userToggled(bool)));
            connect(buttons->projectButton, SIGNAL(toggled(bool)), SLOT(projectToggled(bool)));
            buttons->globalButton->setOn(true);
            if (writeFlag) {
                  buttons->globalButton->setEnabled(false);
                  buttons->projectButton->setOn(true);
                  }
            else
                  buttons->globalButton->setOn(true);
            }
      }

//---------------------------------------------------------
//   ContentsPreview
//---------------------------------------------------------

ContentsPreview::~ContentsPreview()
      {
      if (bg)
            delete bg;
      }

//---------------------------------------------------------
//   ContentsPreview::showPreview
//---------------------------------------------------------

void ContentsPreview::previewUrl(const QUrl& url)
      {
      if (!url.isLocalFile())
            return;
      if (url.path() == path)
            return;
      path = url.path();
      if (bg)
            delete bg;
      bg  = new QPixmap(path);
      if (bg)
            setBackgroundPixmap(*bg);
      }

//---------------------------------------------------------
//   getOpenFileName
//---------------------------------------------------------

QString getOpenFileName(const QString &startWith,
   const char** filters, QWidget* parent, const char* name=0)
      {
      QString initialSelection;
#if 0
	QString* workingDirectory = new QString(QDir::currentDirPath());

      if (!startWith.isEmpty()) {
            QFileInfo fi(startWith);
            if (fi.exists() && fi.isDir()) {
                  *workingDirectory = startWith;
                  }
            else if (fi.exists() && fi.isFile()) {
                  *workingDirectory = fi.dirPath(TRUE);
                  initialSelection = fi.absFilePath();
                  }
            }
#endif
      MFileDialog *dlg = new MFileDialog(startWith, QString::null,
					parent);
      dlg->setFilters(filters);
      dlg->setCaption(name);
      if (!initialSelection.isEmpty())
            dlg->setSelection(initialSelection);
      dlg->setMode(QFileDialog::ExistingFile);
      QString result;
      if (dlg->exec() == QDialog::Accepted) {
            result = dlg->selectedFile();
            }
      delete dlg;
      return result;
      }

//---------------------------------------------------------
//   getSaveFileName
//---------------------------------------------------------

QString getSaveFileName(const QString &startWith,
   const char** filters, QWidget* parent, const char* name=0)
      {
#if 0
      QString initialSelection;
	QString* workingDirectory = new QString(QDir::currentDirPath());
      if (!startWith.isEmpty()) {
            QFileInfo fi( startWith );
            if (fi.exists() && fi.isDir()) {
                  *workingDirectory = startWith;
                  }
            else if (!fi.exists() || fi.isFile()) {
                  *workingDirectory = fi.dirPath(TRUE);
                  initialSelection = fi.absFilePath();
                  }
            }
#endif
      MFileDialog *dlg = new MFileDialog(startWith, QString::null,
         parent, true);
      CHECK_PTR(dlg);
      dlg->setFilters(filters);
      dlg->setCaption(name);
      dlg->setMode(QFileDialog::AnyFile);
      QString result;
//      if (!initialSelection.isEmpty() )
//            dlg->setSelection(initialSelection);
      if (dlg->exec() == QDialog::Accepted) {
            result = dlg->selectedFile();
            }
      delete dlg;
      return result;
      }

//---------------------------------------------------------
//   getImageFileName
//---------------------------------------------------------

QString getImageFileName(const QString& startWith,
   const char** filters, QWidget* parent, const char* name=0)
      {
      QString initialSelection;
	QString* workingDirectory = new QString(QDir::currentDirPath());
      if (!startWith.isEmpty() ) {
            QFileInfo fi(startWith);
            if (fi.exists() && fi.isDir()) {
                  *workingDirectory = startWith;
                  }
            else if (fi.exists() && fi.isFile()) {
                  *workingDirectory = fi.dirPath(TRUE);
                  initialSelection = fi.absFilePath();
                  }
            }
      MFileDialog *dlg = new MFileDialog(*workingDirectory, QString::null,
         parent);

      dlg->setContentsPreviewEnabled(true);
      ContentsPreview* preview = new ContentsPreview(dlg);
      dlg->setContentsPreview(preview, preview);
      dlg->setPreviewMode(QFileDialog::Contents);

      dlg->setCaption(QFileDialog::tr("Open"));
      dlg->setFilters(filters);
      dlg->setMode(QFileDialog::ExistingFile);
      QString result;
      if (!initialSelection.isEmpty())
            dlg->setSelection( initialSelection);
      if (dlg->exec() == QDialog::Accepted) {
            result = dlg->selectedFile();
            }
      delete dlg;
      return result;
      }

//---------------------------------------------------------
//   fileOpen
//    opens file "name" with extension "ext" in mode "mode"
//    handles "name.ext.bz2" and "name.ext.gz"
//
//    mode = "r" or "w"
//    popenFlag   set to true on return if file was opened
//                with popen() (and therefore must be closed
//                with pclose())
//    noError     show no error if file was not found in "r"
//                mode. Has no effect in "w" mode
//    overwriteWarning
//                warn in "w" mode, if file exists
//---------------------------------------------------------

FILE* fileOpen(QWidget* parent, QString name, const QString& ext,
   const char* mode, bool& popenFlag, bool noError = false,
   bool overwriteWarning = false)
      {
      QFileInfo info(name);
      QString zip;

      popenFlag = false;
      if (info.extension(true) == "") {
            name += ext;
            info.setFile(name);
            }
      else if (info.extension(false) == "gz") {
            popenFlag = true;
            zip = "gzip";
            }
      else if (info.extension(false) == "bz2") {
            popenFlag = true;
            zip = "bzip2";
            }

      if (strcmp(mode,"w") == 0 && overwriteWarning && info.exists()) {
            QString s;
            s = "File\n" + name + "\nexists";
            int rv = QMessageBox::warning(parent,
               "MusE: write",
               s,
               "Overwrite", "Quit", 0, 0, 1);
            switch(rv) {
                  case 0:  // overwrite
                        break;
                  case 1:  // quit
                        return 0;
                  }
            }
      FILE* fp = 0;
      if (popenFlag) {
            if (strcmp(mode, "r") == 0)
                  zip += " -d < ";
            else
                  zip += " > ";
            zip += name;
            fp  = popen(zip.ascii(), mode);
            }
      else {
            fp = fopen(name.ascii(), mode);
            }
      if (fp == 0 && !noError) {
            QString s = "Open File\n" + name + "\nfailed: "
               + strerror(errno);
            QMessageBox::critical(parent, "MusE: Open File", s);
            return 0;
            }
      return fp;
      }

//---------------------------------------------------------
//   MFile
//---------------------------------------------------------

MFile::MFile(const QString& _path, const QString& _ext)
   : path(_path), ext(_ext)
      {
      f = 0;
      isPopen = false;
      }

MFile::~MFile()
      {
      if (f) {
            if (isPopen)
                  pclose(f);
            else
                  fclose(f);
            }
      }

//---------------------------------------------------------
//   open
//---------------------------------------------------------

FILE* MFile::open(const char* mode, const char** pattern,
   QWidget* parent, bool noError=false, bool warnIfOverwrite=false)
      {
      QString name;
      if (strcmp(mode, "r") == 0)
           name = getOpenFileName(path, pattern, parent);
      else
           name = getSaveFileName(path, pattern, parent);
      if (name.isEmpty())
            return 0;
      return fileOpen(parent, name, ext, mode, isPopen, noError,
         warnIfOverwrite);
      }

