//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: wav.cpp,v 1.2 2002/02/13 11:42:56 muse Exp $
//
//  (C) Copyright 2001 Werner Schweer (ws@seh.de)
//  based on libsndfile:
//  Copyright (C) 1999-2000 Erik de Castro Lopo <erikd@zip.com.au>
//=========================================================

#include	<stdio.h>
#include	<unistd.h>
#include	<string.h>
#include	<ctype.h>

#include	"sndfile.h"
#include	"config.h"
#include	"sfendian.h"
#include	"pcm.h"
#include	"wav.h"

/*---------------------------------------
** List of known WAV format tags
*/

enum
      {	
	WAVE_FORMAT_UNKNOWN				= 0x0000,		/* Microsoft Corporation */
	WAVE_FORMAT_PCM     		    = 0x0001, 		/* Microsoft PCM format */

	WAVE_FORMAT_MS_ADPCM			= 0x0002,		/* Microsoft ADPCM */
	WAVE_FORMAT_IEEE_FLOAT			= 0x0003,		/* Micrososft 32 bit float format */
	
	WAVE_FORMAT_IBM_CVSD			= 0x0005,		/* IBM Corporation */
	WAVE_FORMAT_ALAW				= 0x0006,		/* Microsoft Corporation */
	WAVE_FORMAT_MULAW				= 0x0007,		/* Microsoft Corporation */
	WAVE_FORMAT_OKI_ADPCM			= 0x0010,		/* OKI */
	WAVE_FORMAT_IMA_ADPCM			= 0x0011,		/* Intel Corporation */
	WAVE_FORMAT_MEDIASPACE_ADPCM	= 0x0012,		/* Videologic */
	WAVE_FORMAT_SIERRA_ADPCM		= 0x0013,		/* Sierra Semiconductor Corp */
	WAVE_FORMAT_G723_ADPCM			= 0x0014,		/* Antex Electronics Corporation */
	WAVE_FORMAT_DIGISTD				= 0x0015,		/* DSP Solutions, Inc. */
	WAVE_FORMAT_DIGIFIX				= 0x0016,		/* DSP Solutions, Inc. */
	WAVE_FORMAT_DIALOGIC_OKI_ADPCM	= 0x0017,		/*  Dialogic Corporation  */
	WAVE_FORMAT_MEDIAVISION_ADPCM	= 0x0018,		/*  Media Vision, Inc. */

	WAVE_FORMAT_YAMAHA_ADPCM		= 0x0020,		/* Yamaha Corporation of America */
	WAVE_FORMAT_SONARC				= 0x0021,		/* Speech Compression */
	WAVE_FORMAT_DSPGROUP_TRUESPEECH = 0x0022,		/* DSP Group, Inc */
	WAVE_FORMAT_ECHOSC1				= 0x0023,		/* Echo Speech Corporation */
	WAVE_FORMAT_AUDIOFILE_AF18  	= 0x0024,		/* Audiofile, Inc. */
	WAVE_FORMAT_APTX				= 0x0025,		/* Audio Processing Technology */
	WAVE_FORMAT_AUDIOFILE_AF10  	= 0x0026,		/* Audiofile, Inc. */

	WAVE_FORMAT_DOLBY_AC2			= 0x0030,		/* Dolby Laboratories */
	WAVE_FORMAT_GSM610				= 0x0031,		/* Microsoft Corporation */
	WAVE_FORMAT_MSNAUDIO			= 0x0032,		/* Microsoft Corporation */
	WAVE_FORMAT_ANTEX_ADPCME		= 0x0033, 		/* Antex Electronics Corporation */
	WAVE_FORMAT_CONTROL_RES_VQLPC	= 0x0034,		/* Control Resources Limited */
	WAVE_FORMAT_DIGIREAL			= 0x0035,		/* DSP Solutions, Inc. */
	WAVE_FORMAT_DIGIADPCM			= 0x0036,		/* DSP Solutions, Inc. */
	WAVE_FORMAT_CONTROL_RES_CR10	= 0x0037,		/* Control Resources Limited */
	WAVE_FORMAT_NMS_VBXADPCM		= 0x0038,		/* Natural MicroSystems */
	WAVE_FORMAT_ROCKWELL_ADPCM		= 0x003B,		/* Rockwell International */
	WAVE_FORMAT_ROCKWELL_DIGITALK	= 0x003C, 		/* Rockwell International */

	WAVE_FORMAT_G721_ADPCM			= 0x0040,		/* Antex Electronics Corporation */
	WAVE_FORMAT_MPEG				= 0x0050,		/* Microsoft Corporation */

	WAVE_FORMAT_MPEGLAYER3			= 0x0055,		/* MPEG 3 Layer 1 */

	IBM_FORMAT_MULAW				= 0x0101,		/* IBM mu-law format */
	IBM_FORMAT_ALAW					= 0x0102,		/* IBM a-law format */
	IBM_FORMAT_ADPCM				= 0x0103,		/* IBM AVC Adaptive Differential PCM format */

	WAVE_FORMAT_CREATIVE_ADPCM		= 0x0200,		/* Creative Labs, Inc */

	WAVE_FORMAT_FM_TOWNS_SND		= 0x0300,		/* Fujitsu Corp. */
	WAVE_FORMAT_OLIGSM				= 0x1000,		/* Ing C. Olivetti & C., S.p.A. */
	WAVE_FORMAT_OLIADPCM			= 0x1001,		/* Ing C. Olivetti & C., S.p.A. */
	WAVE_FORMAT_OLICELP				= 0x1002,		/* Ing C. Olivetti & C., S.p.A. */
	WAVE_FORMAT_OLISBC				= 0x1003,		/* Ing C. Olivetti & C., S.p.A. */
	WAVE_FORMAT_OLIOPR				= 0x1004,		/* Ing C. Olivetti & C., S.p.A. */

	WAVE_FORMAT_EXTENSIBLE			= 0xFFFE
      };

/*------------------------------------------------------------------------------
 * Macros to handle big/little endian issues.
 */

#if (CPU_IS_LITTLE_ENDIAN == 1)
#	define	MAKE_MARKER(a,b,c,d)		((a)|((b)<<8)|((c)<<16)|((d)<<24))
#elif (CPU_IS_BIG_ENDIAN == 1)
#	define	MAKE_MARKER(a,b,c,d)		(((a)<<24)|((b)<<16)|((c)<<8)|(d))
#else
#	error "Cannot determine endian-ness of processor."
#endif

#define RIFF_MARKER	(MAKE_MARKER ('R', 'I', 'F', 'F'))
#define WAVE_MARKER	(MAKE_MARKER ('W', 'A', 'V', 'E'))
#define fmt_MARKER	(MAKE_MARKER ('f', 'm', 't', ' '))
#define data_MARKER	(MAKE_MARKER ('d', 'a', 't', 'a'))
#define cue_MARKER	(MAKE_MARKER ('c', 'u', 'e', ' '))
#define LIST_MARKER	(MAKE_MARKER ('L', 'I', 'S', 'T'))
#define slnt_MARKER	(MAKE_MARKER ('s', 'l', 'n', 't'))
#define wavl_MARKER	(MAKE_MARKER ('w', 'a', 'v', 'l'))
#define INFO_MARKER	(MAKE_MARKER ('I', 'N', 'F', 'O'))
#define plst_MARKER	(MAKE_MARKER ('p', 'l', 's', 't'))
#define adtl_MARKER	(MAKE_MARKER ('a', 'd', 't', 'l'))
#define labl_MARKER	(MAKE_MARKER ('l', 'a', 'b', 'l'))
#define note_MARKER	(MAKE_MARKER ('n', 'o', 't', 'e'))
#define fact_MARKER	(MAKE_MARKER ('f', 'a', 'c', 't'))
#define smpl_MARKER	(MAKE_MARKER ('s', 'm', 'p', 'l'))
#define bext_MARKER	(MAKE_MARKER ('b', 'e', 'x', 't'))
#define MEXT_MARKER	(MAKE_MARKER ('M', 'E', 'X', 'T'))
#define DISP_MARKER	(MAKE_MARKER ('D', 'I', 'S', 'P'))

/*------------------------------------------------------------------------------
 * Private static functions.
 */

static void	le2h_wav_fmt (WAV_FMT *fmt);
static void	h2le_wav_fmt (WAV_FMT *fmt);

//---------------------------------------------------------
//   wav_open_read
//---------------------------------------------------------

int SndFile::wav_open_read()
      {
      WAV_FMT wav_fmt;
	FACT_CHUNK fact_chunk;
	unsigned int dword, marker, RIFFsize;
	int parsestage = 0, error;
      int _fmt = 0;

	_seekable = true;

	for (;;) {
	      fread (&marker, sizeof (marker), 1, _file);
		switch (marker) {	
                  case RIFF_MARKER:
                        if (parsestage != 0)
				      return SFE_WAV_NO_RIFF;
                        fread (&dword, sizeof (dword), 1, _file);
				RIFFsize = LE2H_INT(dword);	
			      if (_filelength  < RIFFsize + 2 * sizeof (dword)) {
                              dword = _filelength - 2 * sizeof (dword);
					RIFFsize = dword;
					}
		            parsestage = 1;
                        break;
					
			case WAVE_MARKER:
                        if (parsestage != 1)
				      return SFE_WAV_NO_WAVE;
			      parsestage = 2;
				break;
			
			case fmt_MARKER:
                        if (parsestage != 2)
				      return SFE_WAV_NO_FMT;
			      if ((error = read_fmt_chunk (&wav_fmt)))
				      return error;
                        _fmt = wav_fmt.format;
			      parsestage = 3;
                        break;
					
			case data_MARKER:
                        if (parsestage < 3)
                              return SFE_WAV_NO_DATA;
                        fread(&dword, sizeof (dword), 1, _file);
			      _datalength = LE2H_INT (dword);
				_dataoffset = ftell (_file);
                        if (_filelength < _dataoffset + _datalength) {	
                              _datalength = _filelength - _dataoffset;
				      }

			      if (_format == WAVE_FORMAT_MS_ADPCM && _datalength % 2) {
				      ++_datalength;
					}
		
			      fseek(_file, _datalength, SEEK_CUR);
				dword = ftell(_file);
				if (dword != _dataoffset + _datalength) {
                              }
			      break;

			case fact_MARKER:
                        fread (&dword, sizeof(dword), 1, _file);
		            dword = LE2H_INT(dword);
			      fread (&fact_chunk, sizeof (fact_chunk), 1, _file);
                        if (dword > sizeof (fact_chunk))
                              fseek (_file, (int) (dword - sizeof (fact_chunk)), SEEK_CUR);
                        fact_chunk.samples = LE2H_INT (fact_chunk.samples);
                        break;

			case cue_MARKER:
			case LIST_MARKER:
			case INFO_MARKER:
			case smpl_MARKER:
			case bext_MARKER:
			case MEXT_MARKER:
			case DISP_MARKER:
			      fread(&dword, sizeof (dword), 1, _file);
				dword = LE2H_INT(dword);
				fseek(_file, (int) dword, SEEK_CUR);
				break;

			default:
			      if (isprint ((marker >> 24) & 0xFF) && isprint ((marker >> 16) & 0xFF)
				   && isprint ((marker >> 8) & 0xFF) && isprint (marker & 0xFF)) {
			            fread (&dword, sizeof (dword), 1, _file) ;
					fseek (_file, (int) dword, SEEK_CUR) ;
					break ;
					}
			      if ((dword = ftell (_file)) & 0x03) {	
                              fseek (_file, -3, SEEK_CUR) ;
					break ;
					}
				break ;
			}

		if (ferror (_file)) {
			clearerr (_file) ;
			break ;
			}
		if (ftell (_file) >= (int) (_filelength - (2 * sizeof (dword))))
			break;
		}
      if (!_dataoffset)
	      return SFE_WAV_NO_DATA;

      current     = 0 ;
	endian      = SF_ENDIAN_LITTLE ;		/* All WAV files are little endian. */
	_sections = 1;
	
	fseek (_file, _dataoffset, SEEK_SET) ;
	
	if (_blockwidth) {	
            if (_filelength - _dataoffset < _datalength)
                  _samples = (_filelength - _dataoffset) / _blockwidth;
		else
			_samples = _datalength / _blockwidth;
		}

	switch (_fmt) {	
            case  WAVE_FORMAT_PCM:
		case	WAVE_FORMAT_EXTENSIBLE:
                  break;
#if 0					
		case	WAVE_FORMAT_MULAW :
		      _format   = (SF_FORMAT_WAV | SF_FORMAT_ULAW) ;
                  fmt = new SndFileFormatUlaw(this);
		      return 0 ;
	
		case	WAVE_FORMAT_ALAW :
			_format   = (SF_FORMAT_WAV | SF_FORMAT_ALAW) ;
                  fmt = new SndFileFormatAlaw(this);
			return 0 ;
		
		case	WAVE_FORMAT_MS_ADPCM :
	            _format   = (SF_FORMAT_WAV | SF_FORMAT_MS_ADPCM) ;
                  fmt = new SndFileFormatADPCM(this);
		      if ((error = ((SndFileFormatADPCM*)(fmt))->reader_init (&wav_fmt)))
                        return error ;
		      return 0 ;

		case	WAVE_FORMAT_IMA_ADPCM :
		      _format   = (SF_FORMAT_WAV | SF_FORMAT_IMA_ADPCM) ;
                  fmt = new SndFileFormatIMA(this);
			if ((error = ((SndFileFormatIMA*)(fmt))->reader_init (&wav_fmt)))
                        return error ;
			return 0 ;
#endif
		case	WAVE_FORMAT_IEEE_FLOAT :
		      _format   = (SF_FORMAT_WAV | SF_FORMAT_FLOAT) ;

                  fmt = new SndFileFormatWaveFloat(this);
		      return 0;
		
		default:	
                  return SFE_UNIMPLEMENTED;
		}

	_format = (SF_FORMAT_WAV | SF_FORMAT_PCM) ;
	switch (_bytewidth) {	
            case  1:
                  fmt = new SndFileFormatWavePCM1(this);
		      break;
		case  2:
                  fmt = new SndFileFormatWavePCM2(this);
		      break;
		case  3:
                  fmt = new SndFileFormatWavePCM3(this);
			break;
		case  4:
                  fmt = new SndFileFormatWavePCM4(this);
			break;
		default:
                  return SFE_UNIMPLEMENTED;
		}
	return 0;
      }

//---------------------------------------------------------
//   wav_open_write
//---------------------------------------------------------

int SndFile::wav_open_write()
      {	
      WAV_FMT wav_fmt;
	unsigned int dword, subformat;
	int error;
	
	if ((_format & SF_FORMAT_TYPEMASK) != SF_FORMAT_WAV)
	      return SFE_BAD_OPEN_FORMAT;
	
	endian      = SF_ENDIAN_LITTLE;     // All WAV files are little endian.
	_seekable = true;
	error       = 0;

	subformat = _format & SF_FORMAT_SUBMASK;
	if (subformat == SF_FORMAT_ULAW || subformat == SF_FORMAT_ALAW)
		_bytewidth = 1;
	else
		_bytewidth = BITWIDTH2BYTES (_pcmbitwidth);

	_blockwidth             = _bytewidth * _channels;
	wav_fmt.min.channels    = _channels;
	wav_fmt.min.samplerate  = _samplerate;
	wav_fmt.min.bytespersec = _samplerate * _bytewidth * _channels;
	wav_fmt.min.blockalign  = _bytewidth * _channels;
	wav_fmt.min.bitwidth    = _pcmbitwidth;


	switch (_format & SF_FORMAT_SUBMASK) {
            case	SF_FORMAT_PCM:
		      wav_fmt.format = WAVE_FORMAT_PCM;
			_dataoffset  = 7 * sizeof (dword) + sizeof (MIN_WAV_FMT);
			_datalength  = _blockwidth * _samples;
			_filelength  = _datalength + _dataoffset;
			write_header (&wav_fmt, sizeof (MIN_WAV_FMT), 0);
			break;

		case	SF_FORMAT_FLOAT:
                  wav_fmt.format = WAVE_FORMAT_IEEE_FLOAT;
                  _dataoffset  = 9 * sizeof (dword) + sizeof (MIN_WAV_FMT) + sizeof (FACT_CHUNK);
			_datalength  = _blockwidth * _samples;
			_filelength  = _datalength + _dataoffset;
			write_header (&wav_fmt, sizeof (MIN_WAV_FMT), 1);
			break;

		case	SF_FORMAT_ULAW:
                  wav_fmt.format = WAVE_FORMAT_MULAW;
			wav_fmt.size20.bitwidth = 8;
			wav_fmt.size20.extrabytes = 2;
			wav_fmt.size20.dummy = 0;
			_dataoffset  = 9 * sizeof (dword) + sizeof (WAV_FMT_SIZE20) + sizeof (FACT_CHUNK);
			_datalength  = _blockwidth * _samples;
			_filelength  = _datalength + _dataoffset;
			write_header (&wav_fmt, sizeof (WAV_FMT_SIZE20), 1);
			break;
					
		case	SF_FORMAT_ALAW:
                  wav_fmt.format = WAVE_FORMAT_ALAW;
                  wav_fmt.size20.bitwidth = 8;
			wav_fmt.size20.extrabytes = 2;
			wav_fmt.size20.dummy = 0;
			_dataoffset  = 9 * sizeof (dword) + sizeof (WAV_FMT_SIZE20) + sizeof (FACT_CHUNK);
			_datalength  = _blockwidth * _samples;
			_filelength  = _datalength + _dataoffset;
			write_header (&wav_fmt, sizeof (WAV_FMT_SIZE20), 1);
			break;

		case	SF_FORMAT_IMA_ADPCM:
                  wav_fmt.format = WAVE_FORMAT_IMA_ADPCM;
//			if ((error = wav_ima_writer_init (&wav_fmt)))
//			      return error;
                  _dataoffset  = 9 * sizeof (dword) + sizeof (IMA_ADPCM_WAV_FMT) + sizeof (FACT_CHUNK);
			if (_samples % wav_fmt.ima.samplesperblock)
			      _datalength  = ((_samples / wav_fmt.ima.samplesperblock) + 1) * wav_fmt.ima.samplesperblock;
			else
				_datalength  = _samples;
			_filelength  = _datalength + _dataoffset;
			write_header (&wav_fmt, sizeof (IMA_ADPCM_WAV_FMT), 1);
			break;

		case	SF_FORMAT_MS_ADPCM :
                  wav_fmt.format = WAVE_FORMAT_MS_ADPCM;
//			msadpcm_writer_init (&wav_fmt);
			_dataoffset  = 9 * sizeof (dword) + sizeof (MS_ADPCM_WAV_FMT) + sizeof (FACT_CHUNK);
			if (_samples % wav_fmt.msadpcm.samplesperblock)
			      _datalength  = ((_samples / wav_fmt.msadpcm.samplesperblock) + 1) * wav_fmt.msadpcm.samplesperblock;
			else
				_datalength  = _samples;
			_filelength  = _datalength + _dataoffset;
			write_header (&wav_fmt, sizeof (MS_ADPCM_WAV_FMT), 1);
			break;

		default: 	
                  return SFE_UNIMPLEMENTED;
		}

	dword = data_MARKER;
	fwrite(&dword, sizeof (dword), 1, _file);

	dword = H2LE_INT (_datalength);
	fwrite (&dword, sizeof (dword), 1, _file);
	
	if ((_format & SF_FORMAT_SUBMASK) == SF_FORMAT_FLOAT) {	
            _format = (SF_FORMAT_WAV | SF_FORMAT_FLOAT) ;
            fmt = new SndFileFormatWaveFloat(this);
		return 0;
            }
#if 0	
	if ((_format & SF_FORMAT_SUBMASK) == SF_FORMAT_IMA_ADPCM) {
	      _format    = (SF_FORMAT_WAV | SF_FORMAT_IMA_ADPCM);
            fmt = new SndFileFormatIMA(this);
		return 0;
		}

	if ((_format & SF_FORMAT_SUBMASK) == SF_FORMAT_MS_ADPCM) {
	      _format    = (SF_FORMAT_WAV | SF_FORMAT_MS_ADPCM);
            fmt = new SndFileFormatADPCM(this);
		return 0;
		}
	if ((_format & SF_FORMAT_SUBMASK) == SF_FORMAT_ULAW) {
	      _format    = (SF_FORMAT_WAV | SF_FORMAT_ULAW);
            fmt = new SndFileFormatUlaw(this);
		return 0;
		}
	if ((_format & SF_FORMAT_SUBMASK) == SF_FORMAT_ALAW) {
            _format    = (SF_FORMAT_WAV | SF_FORMAT_ALAW);
            fmt = new SndFileFormatAlaw(this);
		return 0;
		}
#endif
	if ((_format & SF_FORMAT_SUBMASK) != SF_FORMAT_PCM)
		return SFE_UNIMPLEMENTED ;

	switch (_bytewidth) {	
            case  1:
                  fmt = new SndFileFormatWavePCM1(this);
		      break;
		case  2:
                  fmt = new SndFileFormatWavePCM2(this);
		      break;
		case  3:
                  fmt = new SndFileFormatWavePCM3(this);
			break;
		case  4:
                  fmt = new SndFileFormatWavePCM4(this);
			break;
		default:
                  return SFE_UNIMPLEMENTED;
		}
	return 0;
      }

//---------------------------------------------------------
//   close
//---------------------------------------------------------

void SndFileFormatWave::close ()
      {	
	if (sfile->mode() == SF_MODE_WRITE) {	
            /*  Now we know for certain the length of the file we can
		 *  re-write correct values for the RIFF and data chunks.
		 */
		fseek(sfile->file(), 0, SEEK_END);
		sfile->setFilelength(ftell(sfile->file()));

		/* Fix RIFF size. */
		unsigned int dword = H2LE_INT (sfile->filelength() - 2 * sizeof (dword));
		fseek (sfile->file(), sizeof (dword), SEEK_SET);
		fwrite (&dword, sizeof (dword), 1, sfile->file());
		
		sfile->setDatalength(sfile->filelength() - sfile->dataoffset());
		sfile->setSamples(sfile->datalength() / (sfile->channels() * sfile->bytewidth()));
		fseek (sfile->file(), (int) (sfile->dataoffset() - sizeof (dword)), SEEK_SET);
		dword = H2LE_INT (sfile->datalength());
		fwrite (&dword, sizeof (dword), 1, sfile->file());
		}
      }

//---------------------------------------------------------
//   read_fmt_chunk
//---------------------------------------------------------

int SndFile::read_fmt_chunk(WAV_FMT *wav_fmt)
      {	
      unsigned int dword, bytesread, structsize, bytespersec = 0;
	
	memset (wav_fmt, 0, sizeof (WAV_FMT));
	bytesread = 0;

	fread(&dword, sizeof (dword), 1, _file);
	structsize = LE2H_INT (dword);
	
	if (structsize < 16)
		return SFE_WAV_FMT_SHORT ;
	if (structsize > sizeof (WAV_FMT))
		return SFE_WAV_FMT_TOO_BIG;
					
	fread(wav_fmt, structsize, 1, _file);
	bytesread += structsize;
	if (CPU_IS_BIG_ENDIAN)
		le2h_wav_fmt (wav_fmt);

	_samplerate		= wav_fmt->min.samplerate;
	_samples 		= 0;	    /* Correct this when reading data chunk. */
	_channels		= wav_fmt->min.channels;
	
	switch (wav_fmt->format) {
            case WAVE_FORMAT_PCM:
		case WAVE_FORMAT_IEEE_FLOAT:
		      bytespersec = wav_fmt->min.samplerate * wav_fmt->min.blockalign;
		      _pcmbitwidth = wav_fmt->min.bitwidth;
			_bytewidth = BITWIDTH2BYTES (wav_fmt->min.bitwidth);
			break;

		case WAVE_FORMAT_ALAW:
		case WAVE_FORMAT_MULAW:
		      _pcmbitwidth = 16;
                  _bytewidth      = 1;
		      break;

		case WAVE_FORMAT_MS_ADPCM:
                  if (wav_fmt->msadpcm.bitwidth != 4)
			      return SFE_WAV_ADPCM_NOT4BIT;
		      if (wav_fmt->msadpcm.channels < 1 || wav_fmt->msadpcm.channels > 2)
			      return SFE_WAV_ADPCM_CHANNELS;

		      bytespersec = (wav_fmt->msadpcm.samplerate * wav_fmt->msadpcm.blockalign) / wav_fmt->msadpcm.samplesperblock;
		      _pcmbitwidth = 16;
		      _bytewidth      = 2;
			if (wav_fmt->msadpcm.numcoeffs > sizeof (MS_ADPCM_WAV_FMT) / sizeof (int)) {
			      wav_fmt->msadpcm.numcoeffs = sizeof (MS_ADPCM_WAV_FMT) / sizeof (int);
			      }
			break;
				
		case WAVE_FORMAT_IMA_ADPCM :
			if (wav_fmt->ima.bitwidth != 4)
			      return SFE_WAV_ADPCM_NOT4BIT;
		      if (wav_fmt->ima.channels < 1 || wav_fmt->ima.channels > 2)
			      return SFE_WAV_ADPCM_CHANNELS;
		      bytespersec = (wav_fmt->ima.samplerate * wav_fmt->ima.blockalign) / wav_fmt->ima.samplesperblock;
		      _pcmbitwidth = 16;
			_bytewidth      = 2;
			break;
				
		case WAVE_FORMAT_EXTENSIBLE :
			_pcmbitwidth = wav_fmt->ext.bitwidth ;
			_bytewidth      = BITWIDTH2BYTES (wav_fmt->ext.bitwidth) ;
			break;

		default:
                  break;
		}
	_blockwidth = wav_fmt->min.channels * _bytewidth;
	return 0;
      }

//---------------------------------------------------------
//   write_header
//---------------------------------------------------------

int SndFile::write_header(WAV_FMT *wav_fmt, unsigned int size, int do_fact)
      {	
      FACT_CHUNK        fact_chunk ;
	unsigned int dword, RIFFsize ;

	RIFFsize   = _filelength - 2 * sizeof (dword) ;

	dword = RIFF_MARKER ;
	fwrite (&dword, sizeof (dword), 1, _file) ;
	dword = H2LE_INT (RIFFsize) ;
	fwrite (&dword, sizeof (dword), 1, _file) ;
	
	dword = WAVE_MARKER ;
	fwrite (&dword, sizeof (dword), 1, _file) ;
	dword = fmt_MARKER ;
	fwrite (&dword, sizeof (dword), 1, _file) ;
	
	dword = H2LE_INT (size) ;
	fwrite (&dword, sizeof (dword), 1, _file) ;
	if (CPU_IS_BIG_ENDIAN)
		h2le_wav_fmt (wav_fmt) ;	
	fwrite (wav_fmt, size, 1, _file) ;
	
	if (do_fact) {	
            dword = fact_MARKER ;
		fwrite (&dword, sizeof (dword), 1, _file) ;
		dword = H2LE_INT (sizeof (FACT_CHUNK)) ;
		fwrite (&dword, sizeof (dword), 1, _file) ;
		fact_chunk.samples = H2LE_INT (_samples) ;
		fwrite (&fact_chunk, sizeof (fact_chunk), 1, _file) ;
		}

	return 0;
      }

static void	le2h_wav_fmt (WAV_FMT *fmt)
      {	
      int         k ;

	fmt->min.format      = ENDSWAP_SHORT (fmt->min.format) ;
	fmt->min.channels    = ENDSWAP_SHORT (fmt->min.channels) ;
	fmt->min.samplerate  = ENDSWAP_INT   (fmt->min.samplerate) ;
	fmt->min.bytespersec = ENDSWAP_INT   (fmt->min.bytespersec) ;
	fmt->min.blockalign  = ENDSWAP_SHORT (fmt->min.blockalign) ;
	fmt->min.bitwidth    = ENDSWAP_SHORT (fmt->min.bitwidth) ;

	switch (fmt->format)
	{	case WAVE_FORMAT_MS_ADPCM :	
				fmt->msadpcm.extrabytes      = ENDSWAP_SHORT (fmt->msadpcm.extrabytes) ;
				fmt->msadpcm.samplesperblock = ENDSWAP_SHORT (fmt->msadpcm.samplesperblock) ;
				fmt->msadpcm.numcoeffs       = ENDSWAP_SHORT (fmt->msadpcm.numcoeffs) ;
				for (k = 0 ; k < fmt->msadpcm.numcoeffs  ; k++)
				{	fmt->msadpcm.coeffs [k].coeff1 = ENDSWAP_SHORT (fmt->msadpcm.coeffs [k].coeff1) ;
					fmt->msadpcm.coeffs [k].coeff2 = ENDSWAP_SHORT (fmt->msadpcm.coeffs [k].coeff2) ;
					} ;
				break ;
				
		case WAVE_FORMAT_IMA_ADPCM :	
				fmt->ima.extrabytes      = ENDSWAP_SHORT (fmt->ima.extrabytes) ;
				fmt->ima.samplesperblock = ENDSWAP_SHORT (fmt->ima.samplesperblock) ;
				break ;
				
		case WAVE_FORMAT_ALAW :	
		case WAVE_FORMAT_MULAW :	
				fmt->size20.extrabytes = ENDSWAP_SHORT (fmt->size20.extrabytes) ;
				fmt->size20.dummy      = ENDSWAP_SHORT (fmt->size20.dummy) ;
				break ;
				
		default : break ;
		} ;

      }

static void	h2le_wav_fmt (WAV_FMT *fmt)
      {
      int k ;

	switch (fmt->format)
	{	case WAVE_FORMAT_MS_ADPCM :	
				for (k = 0 ; k < fmt->msadpcm.numcoeffs  ; k++)
				{	fmt->msadpcm.coeffs [k].coeff1 = ENDSWAP_SHORT (fmt->msadpcm.coeffs [k].coeff1) ;
					fmt->msadpcm.coeffs [k].coeff2 = ENDSWAP_SHORT (fmt->msadpcm.coeffs [k].coeff2) ;
					} ;
				fmt->msadpcm.numcoeffs       = ENDSWAP_SHORT (fmt->msadpcm.numcoeffs) ;
				fmt->msadpcm.extrabytes      = ENDSWAP_SHORT (fmt->msadpcm.extrabytes) ;
				fmt->msadpcm.samplesperblock = ENDSWAP_SHORT (fmt->msadpcm.samplesperblock) ;
				break ;
				
		case WAVE_FORMAT_IMA_ADPCM :	
				fmt->ima.extrabytes      = ENDSWAP_SHORT (fmt->ima.extrabytes) ;
				fmt->ima.samplesperblock = ENDSWAP_SHORT (fmt->ima.samplesperblock) ;
				break ;
				
		case WAVE_FORMAT_ALAW :	
		case WAVE_FORMAT_MULAW :	
				fmt->size20.extrabytes = ENDSWAP_SHORT (fmt->size20.extrabytes) ;
				fmt->size20.dummy      = ENDSWAP_SHORT (fmt->size20.dummy) ;
				break ;
				
		default : break ;
		} ;

	fmt->min.format      = ENDSWAP_SHORT (fmt->min.format) ;
	fmt->min.channels    = ENDSWAP_SHORT (fmt->min.channels) ;
	fmt->min.samplerate  = ENDSWAP_INT   (fmt->min.samplerate) ;
	fmt->min.bytespersec = ENDSWAP_INT   (fmt->min.bytespersec) ;
	fmt->min.blockalign  = ENDSWAP_SHORT (fmt->min.blockalign) ;
	fmt->min.bitwidth    = ENDSWAP_SHORT (fmt->min.bitwidth) ;
      }

#if 0
static const char* wav_format_str (int k)
      {
      switch (k) {	
            case WAVE_FORMAT_UNKNOWN :
			return "WAVE_FORMAT_UNKNOWN" ;
		case WAVE_FORMAT_PCM          :
			return "WAVE_FORMAT_PCM         " ;
		case WAVE_FORMAT_MS_ADPCM :
			return "WAVE_FORMAT_MS_ADPCM" ;
		case WAVE_FORMAT_IEEE_FLOAT :
			return "WAVE_FORMAT_IEEE_FLOAT" ;
		case WAVE_FORMAT_IBM_CVSD :
			return "WAVE_FORMAT_IBM_CVSD" ;
		case WAVE_FORMAT_ALAW :
			return "WAVE_FORMAT_ALAW" ;
		case WAVE_FORMAT_MULAW :
			return "WAVE_FORMAT_MULAW" ;
		case WAVE_FORMAT_OKI_ADPCM :
			return "WAVE_FORMAT_OKI_ADPCM" ;
		case WAVE_FORMAT_IMA_ADPCM :
			return "WAVE_FORMAT_IMA_ADPCM" ;
		case WAVE_FORMAT_MEDIASPACE_ADPCM :
			return "WAVE_FORMAT_MEDIASPACE_ADPCM" ;
		case WAVE_FORMAT_SIERRA_ADPCM :
			return "WAVE_FORMAT_SIERRA_ADPCM" ;
		case WAVE_FORMAT_G723_ADPCM :
			return "WAVE_FORMAT_G723_ADPCM" ;
		case WAVE_FORMAT_DIGISTD :
			return "WAVE_FORMAT_DIGISTD" ;
		case WAVE_FORMAT_DIGIFIX :
			return "WAVE_FORMAT_DIGIFIX" ;
		case WAVE_FORMAT_DIALOGIC_OKI_ADPCM :
			return "WAVE_FORMAT_DIALOGIC_OKI_ADPCM" ;
		case WAVE_FORMAT_MEDIAVISION_ADPCM :
			return "WAVE_FORMAT_MEDIAVISION_ADPCM" ;
		case WAVE_FORMAT_YAMAHA_ADPCM :
			return "WAVE_FORMAT_YAMAHA_ADPCM" ;
		case WAVE_FORMAT_SONARC :
			return "WAVE_FORMAT_SONARC" ;
		case WAVE_FORMAT_DSPGROUP_TRUESPEECH  :
			return "WAVE_FORMAT_DSPGROUP_TRUESPEECH " ;
		case WAVE_FORMAT_ECHOSC1 :
			return "WAVE_FORMAT_ECHOSC1" ;
		case WAVE_FORMAT_AUDIOFILE_AF18   :
			return "WAVE_FORMAT_AUDIOFILE_AF18  " ;
		case WAVE_FORMAT_APTX :
			return "WAVE_FORMAT_APTX" ;
		case WAVE_FORMAT_AUDIOFILE_AF10   :
			return "WAVE_FORMAT_AUDIOFILE_AF10  " ;
		case WAVE_FORMAT_DOLBY_AC2 :
			return "WAVE_FORMAT_DOLBY_AC2" ;
		case WAVE_FORMAT_GSM610 :
			return "WAVE_FORMAT_GSM610" ;
		case WAVE_FORMAT_MSNAUDIO :
			return "WAVE_FORMAT_MSNAUDIO" ;
		case WAVE_FORMAT_ANTEX_ADPCME :
			return "WAVE_FORMAT_ANTEX_ADPCME" ;
		case WAVE_FORMAT_CONTROL_RES_VQLPC :
			return "WAVE_FORMAT_CONTROL_RES_VQLPC" ;
		case WAVE_FORMAT_DIGIREAL :
			return "WAVE_FORMAT_DIGIREAL" ;
		case WAVE_FORMAT_DIGIADPCM :
			return "WAVE_FORMAT_DIGIADPCM" ;
		case WAVE_FORMAT_CONTROL_RES_CR10 :
			return "WAVE_FORMAT_CONTROL_RES_CR10" ;
		case WAVE_FORMAT_NMS_VBXADPCM :
			return "WAVE_FORMAT_NMS_VBXADPCM" ;
		case WAVE_FORMAT_ROCKWELL_ADPCM :
			return "WAVE_FORMAT_ROCKWELL_ADPCM" ;
		case WAVE_FORMAT_ROCKWELL_DIGITALK :
			return "WAVE_FORMAT_ROCKWELL_DIGITALK" ;
		case WAVE_FORMAT_G721_ADPCM :
			return "WAVE_FORMAT_G721_ADPCM" ;
		case WAVE_FORMAT_MPEG :
			return "WAVE_FORMAT_MPEG" ;
		case WAVE_FORMAT_MPEGLAYER3 :
			return "WAVE_FORMAT_MPEGLAYER3" ;
		case IBM_FORMAT_MULAW :
			return "IBM_FORMAT_MULAW" ;
		case IBM_FORMAT_ALAW :
			return "IBM_FORMAT_ALAW" ;
		case IBM_FORMAT_ADPCM :
			return "IBM_FORMAT_ADPCM" ;
		case WAVE_FORMAT_CREATIVE_ADPCM :
			return "WAVE_FORMAT_CREATIVE_ADPCM" ;
		case WAVE_FORMAT_FM_TOWNS_SND :
			return "WAVE_FORMAT_FM_TOWNS_SND" ;
		case WAVE_FORMAT_OLIGSM :
			return "WAVE_FORMAT_OLIGSM" ;
		case WAVE_FORMAT_OLIADPCM :
			return "WAVE_FORMAT_OLIADPCM" ;
		case WAVE_FORMAT_OLICELP :
			return "WAVE_FORMAT_OLICELP" ;
		case WAVE_FORMAT_OLISBC :
			return "WAVE_FORMAT_OLISBC" ;
		case WAVE_FORMAT_OLIOPR :
			return "WAVE_FORMAT_OLIOPR" ;
		case WAVE_FORMAT_EXTENSIBLE :
			return "WAVE_FORMAT_EXTENSIBLE" ;
		break ;
		}
	return "Unknown format";
      }
#endif

