//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: plugin.h,v 1.2 2001/11/20 15:19:33 muse Exp $
//
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#ifndef __PLUGIN_H__
#define __PLUGIN_H__

#include <qstring.h>
#include <qwidget.h>
#include <list>
#include "ladspa.h"
#include "globals.h"
#include <qdialog.h>
#include "desk.h"

class Xml;
class QWidget;
class QLabel;
class Slider;
class QListView;

//---------------------------------------------------------
//   Plugin
//---------------------------------------------------------

class Plugin {
      QString libname;
      LADSPA_Descriptor_Function ladspa;
      const LADSPA_Descriptor *plugin;
      int _inports;
      int _outports;

   public:
      Plugin(const QString& libname,
         LADSPA_Descriptor_Function df, const LADSPA_Descriptor* d);
      QString label() const { return QString(plugin->Label); }
      QString name() const { return QString(plugin->Name); }
      QString lib() const { return libname; }
      LADSPA_Handle instantiate() {
            return plugin->instantiate(plugin, sampleRate);
            }
      void activate(LADSPA_Handle handle) {
            if (plugin->activate)
                  plugin->activate(handle);
            }
      void connectPort(LADSPA_Handle handle, int port, float* value) {
            plugin->connect_port(handle, port, value);
            }
      void apply(LADSPA_Handle handle, int n) {
            plugin->run(handle, n);
            }
      int ports() { return plugin->PortCount; }
      LADSPA_PortDescriptor portd(int k) const {
            return plugin->PortDescriptors[k];
            }
      LADSPA_PortRangeHint range(int i) const {
            return plugin->PortRangeHints[i];
            }
      const char* portName(int i) {
            return plugin->PortNames[i];
            }
      int inports() const  { return _inports; }
      int outports() const { return _outports; }
      };

typedef std::list<Plugin>::iterator iPlugin;

//---------------------------------------------------------
//   PluginList
//---------------------------------------------------------

class PluginList : public std::list<Plugin> {
   public:
      void add(const QString& n, LADSPA_Descriptor_Function df,
         const LADSPA_Descriptor* d) {
            push_back(Plugin(n, df, d));
            }
      Plugin* find(const QString&, const QString&);
      PluginList() {}
      };

struct Port {
      int idx;
      float val;
      float tmpVal;
      };

//---------------------------------------------------------
//   PluginI
//    plugin instance
//---------------------------------------------------------

#define AUDIO_IN (LADSPA_PORT_AUDIO  | LADSPA_PORT_INPUT)
#define AUDIO_OUT (LADSPA_PORT_AUDIO | LADSPA_PORT_OUTPUT)

class PluginI {
      Plugin* plugin;
      int instances;

      LADSPA_Handle* handle;         // per instance
      Port* controls;

      int controlPorts;
      QWidget* gui;
      bool _on;
      bool initControlValues;

      void init();
      void makeGui();

   public:
//      PluginI(Plugin* p, int channels);
      PluginI();
      ~PluginI();

      bool on() const       { return _on; }
      void setOn(bool val)  { _on = val; }
      bool initPluginInstance(Plugin*, int channels);
      void connect(float*, int segmentSize);

      void apply(int n);
      void activate();
      QString label() const { return plugin->label(); }
      QString name() const  { return plugin->name(); }
      QString lib() const   { return plugin->lib(); }

      void writeConfiguration(int level, Xml& xml);
      bool readConfiguration(Xml& xml);
      bool loadControl(Xml& xml);
      bool setControl(const QString& s, double val);
      void showGui();
      void showGui(bool);
      bool guiVisible();
      int parameters() const           { return controlPorts; }
      void setParam(int i, double val) { controls[i].tmpVal = val; }
      double param(int i) const        { return controls[i].val; }
      const char* paramName(int i)     { return plugin->portName(controls[i].idx); }
      LADSPA_PortDescriptor portd(int i) const { return plugin->portd(controls[i].idx); }
      LADSPA_PortRangeHint range(int i) const { return plugin->range(controls[i].idx); }
      bool isAudioIn(int k) {
            return (plugin->portd(k) & AUDIO_IN) == AUDIO_IN;
            }
      bool isAudioOut(int k) {
            return (plugin->portd(k) & AUDIO_OUT) == AUDIO_OUT;
            }
      };

//---------------------------------------------------------
//   Pipeline
//    chain of connected efx inserts
//---------------------------------------------------------

class Pipeline : public std::list<PluginI*> {
   public:
      PluginI* find(int index) const;
      void insert(PluginI* p, int index);
      void remove(int index);
      bool isOn(int idx) const;
      void setOn(int, bool);
      QString label(int idx) const;
      QString name(int idx) const;
      void showGui(int, bool);
      bool guiVisible(int);
      void apply(float* buffer, int segmentSize, int channel);
      };

typedef Pipeline::iterator iPluginI;
typedef Pipeline::const_iterator ciPluginI;

//---------------------------------------------------------
//   PluginGui
//---------------------------------------------------------

class PluginGui : public MainWindow {
      Q_OBJECT

      PluginI* plugin;        // plugin instance
      Slider** slider;
      QLabel** valLabel;

   private slots:
      void parameterChanged(int);
      void load();
      void save();

   public:
      PluginGui(PluginI*);
      };


//---------------------------------------------------------
//   PluginDialog
//---------------------------------------------------------

class PluginDialog : public QDialog {
      QListView* pList;

      Q_OBJECT

   public:
      PluginDialog(QWidget* parent=0, const char* name=0, bool modal=true);
      static Plugin* getPlugin(QWidget* parent);
      Plugin* value();
      };

extern void initPlugins();
extern PluginList plugins;

#endif

