//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: amixer.cpp,v 1.2 2002/02/13 11:42:56 muse Exp $
//
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#include "amixer.h"
#include "song.h"
#include "meter.h"
#include "seq.h"
#include "xml.h"
#include "globals.h"
#include "plugins/plugin.h"
#include "audiomix.h"
#include "driver/audiodev.h"
#include "icons.h"
#include "panknob.h"
#include "doublelabel.h"
#include "mlabel.h"
#include "rack.h"
#include "audiothread.h"
#include "volslider.h"
#include "routecombo.h"
#include "audionodebutton.h"
#include "sf/sndfile.h"

#include <qlabel.h>
#include <qpushbutton.h>
#include <math.h>
#include <qlistbox.h>
#include <qtooltip.h>
#include <qtoolbutton.h>
#include <qpopupmenu.h>
#include <qsignalmapper.h>
#include <qlist.h>
#include <qobjectlist.h>
#include <qlayout.h>
#include <list>

static const int STRIP_WIDTH = 70;

enum { MASTER, GROUP, TRACK, INPUT, SYNTHI };

//---------------------------------------------------------
//   Strip
//---------------------------------------------------------

struct Strip {
      int channel;
      AudioNode* src;
      Meter* meter[2];

      Strip() {
            channel = 0;
            }
      ~Strip() {
            }
      };

typedef std::list<Strip*> StripList;
typedef std::list<Strip*>::iterator iStrip;

static StripList stripList;

//---------------------------------------------------------
//   recordToggled
//---------------------------------------------------------

void AudioMixerApp::recordToggled(bool val, AudioNode*)
      {
      if (val)
            audioThread->msgOpenMixdownRecord();
      else
            audioThread->msgCloseMixdownRecord();
      }

//---------------------------------------------------------
//   muteToggled
//---------------------------------------------------------

void AudioMixerApp::muteToggled(bool val, AudioNode* node)
      {
      audioThread->msgSetMute(node, val);
      }

//---------------------------------------------------------
//   preToggled
//---------------------------------------------------------

void AudioMixerApp::preToggled(bool val, AudioNode* node)
      {
      audioThread->msgSetPrefader(node, val);
      }

//---------------------------------------------------------
//   stereoToggled
//---------------------------------------------------------

void AudioMixerApp::stereoToggled(bool val, AudioNode* node)
      {
      audioThread->msgSetChannels(node, val ? 2 : 1);
      }

//---------------------------------------------------------
//   soloToggled
//---------------------------------------------------------

void AudioMixerApp::soloToggled(bool val, AudioNode* node)
      {
      song->setSolo(val ? node : 0);
      }

//---------------------------------------------------------
//   recFileDialog
//---------------------------------------------------------

void AudioMixerApp::recFileDialog()
      {
      SndFile* sf = audioThread->mixdownFile();
      SndFile* nsf = getSndFile(sf, this, "recFileDialog");
      if (nsf)
            audioThread->msgSetMixdownFile(nsf);
      }

//---------------------------------------------------------
//   genStrip
//    create mixer strip
//---------------------------------------------------------

void AudioMixerApp::genStrip(const QString& txt, int col, AudioNode* src,
   int type)
      {
      Strip* strip = new Strip;
      strip->channel = src->channels();
      for (int ch = 0; ch < strip->channel; ++ch) {
            strip->meter[ch] = new Meter(central);
            }
      strip->src = src;

      int row = 1;

      //---------------------------------------------
      //    channel label
      //---------------------------------------------

      QLabel* label = new QLabel(txt, central);
      label->setAlignment(AlignCenter);
      label->setFixedWidth(STRIP_WIDTH);
      grid->addWidget(label, row++, col);

      //---------------------------------------------------
      //    plugin rack
      //---------------------------------------------------

      EffectRack* rack = new EffectRack(central, src, strip->channel);
      rack->setFixedWidth(STRIP_WIDTH);
      grid->addWidget(rack, row++, col);

      //---------------------------------------------------
      //    mono/stereo  pre/post
      //---------------------------------------------------

      QHBoxLayout* ppBox = new QHBoxLayout;
      AudioNodeButton* stereo  = new AudioNodeButton(central, src);
      stereo->setFont(font1);
      stereo->setText("S");
      stereo->setToggleButton(true);
      QToolTip::add(stereo, tr("1/2 channel"));
      connect(stereo, SIGNAL(audioNodeToggled(bool, AudioNode*)),
         SLOT(stereoToggled(bool, AudioNode*)));
// printf("STrip <%s> %d\n", txt.latin1(), src->channels());
      stereo->setOn(src->channels() == 2);

      // disable mono/stereo for Synthesizer-Plugins and Master
      for (iSynthI ii = synthiInstances.begin(); ii != synthiInstances.end(); ++ii) {
            if (src == *ii) {
                  stereo->setEnabled(false);
                  break;
                  }
            }
      if (src == song->master())
            stereo->setEnabled(false);

      AudioNodeButton* pre  = new AudioNodeButton(central, src);
      pre->setFont(font1);
      pre->setToggleButton(true);
      pre->setText("Pre");
      connect(pre, SIGNAL(audioNodeToggled(bool, AudioNode*)),
         SLOT(preToggled(bool, AudioNode*)));
      QToolTip::add(pre, tr("pre fader - post fader"));
      pre->setOn(src->prefader());

      ppBox->addWidget(stereo);
      ppBox->addWidget(pre);
      grid->addLayout(ppBox, row++, col);

      //---------------------------------------------------
      //    slider, label, meter
      //---------------------------------------------------

      QGridLayout* sliderGrid = new QGridLayout;
      sliderGrid->setRowStretch(0, 100);

      VolSlider* slider = new VolSlider(central, src);
      slider->setFixedWidth(20);
      slider->setRange(minSlider, 10.0);
      slider->setFont(font1);

      sliderGrid->addWidget(slider, 0, 0, AlignHCenter);

      for (int i = 0; i < strip->channel; ++i) {
            strip->meter[i]->setRange(minSlider, 10.0);
            strip->meter[i]->setFixedWidth(10);
            connect(strip->meter[i], SIGNAL(mousePress()), src, SLOT(resetPeaks()));
            sliderGrid->addWidget(strip->meter[i], 0, i+1, AlignHCenter);
            sliderGrid->setColStretch(i, 50);
            }
      grid->addLayout(sliderGrid, row++, col);

      DoubleLabel* sl = new DoubleLabel(0.0, minSlider, 10.0, central);
      sl->setFrame(true);
      sl->setPrecision(0);
      sl->setFixedWidth(STRIP_WIDTH);
      connect(slider, SIGNAL(valueChanged(double)), sl, SLOT(setValue(double)));
      slider->setValue(log10(src->volume())*20);
      grid->addWidget(sl, row++, col);

      //---------------------------------------------------
      //    pan, balance
      //---------------------------------------------------

      PanKnob* pan = new PanKnob(central, src);
      pan->setFixedWidth(STRIP_WIDTH);
      pan->setRange(-1.0, +1.0);
      grid->addWidget(pan, row++, col);
      pan->setValue(src->pan());

      //---------------------------------------------------
      //    mute, solo
      //    or
      //    record, mixdownfile
      //---------------------------------------------------

      QHBoxLayout* smBox = new QHBoxLayout;
      if (type == MASTER) {
            AudioNodeButton* record  = new AudioNodeButton(central, src);
            record->setToggleButton(true);
            QIconSet iconSet;
            iconSet.setPixmap(*record1Icon, QIconSet::Automatic, QIconSet::Normal, QIconSet::On);
            iconSet.setPixmap(*dot1Icon, QIconSet::Automatic, QIconSet::Normal, QIconSet::Off);
            record->setIconSet(iconSet);
            QToolTip::add(record, tr("record downmix"));
            connect(record, SIGNAL(audioNodeToggled(bool, AudioNode*)),
               SLOT(recordToggled(bool, AudioNode*)));
            record->setOn(audioThread->mixdownRecord());

            QToolButton* buttonPath = new QToolButton(central, "path");
            buttonPath->setPixmap(*openIconS);
            connect(buttonPath, SIGNAL(clicked()), SLOT(recFileDialog()));
            QToolTip::add(buttonPath, tr("select downmix file"));

            smBox->addWidget(record);
            smBox->addWidget(buttonPath);
            }
      else {
            AudioNodeButton* mute  = new AudioNodeButton(central, src);
            mute->setFont(font1);
            mute->setText("M");
            connect(mute, SIGNAL(audioNodeToggled(bool, AudioNode*)), SLOT(muteToggled(bool, AudioNode*)));
            mute->setToggleButton(true);
            QToolTip::add(mute, tr("mute"));

            AudioNodeButton* solo  = new AudioNodeButton(central, src);
            solo->setFont(font1);
            connect(solo, SIGNAL(audioNodeToggled(bool, AudioNode*)), SLOT(soloToggled(bool, AudioNode*)));
            solo->setText("S");
            QToolTip::add(solo, tr("solo"));
            solo->setToggleButton(true);

            smBox->addWidget(mute);
            smBox->addWidget(solo);
            }
      grid->addLayout(smBox, row++, col);

      //---------------------------------------------------
      //    output routing
      //---------------------------------------------------

      RouteComboBox* cb1 = new RouteComboBox(central, src);
      cb1->setFixedWidth(STRIP_WIDTH);
      cb1->setFont(font4);
      AudioPort* port  = &audioPort;
      AudioDevice* dev = port->device();
      AudioNode* route = src->route();
      int currentRoute = 0;

      switch(type) {
            case GROUP:
                  cb1->insertItem("Master");
                  break;
            case TRACK:
            case INPUT:
            case SYNTHI:
                  cb1->insertItem("Master");
                  cb1->insertItem("Group A");
                  cb1->insertItem("Group B");
                  cb1->insertItem("Group C");
                  cb1->insertItem("Group D");
                  if (route == song->master())
                        currentRoute = 0;
                  else if (route == song->group(0))
                        currentRoute = 1;
                  else if (route == song->group(1))
                        currentRoute = 2;
                  else if (route == song->group(2))
                        currentRoute = 3;
                  else if (route == song->group(3))
                        currentRoute = 4;
                  cb1->setCurrentItem(currentRoute);
                  break;
            case MASTER:
                  if (dev && port->rwFlags() & 1)
                        cb1->insertItem(dev->name());
                  else
                        cb1->insertItem("None");
                  break;
            }
      QToolTip::add(cb1, tr("output routing"));
      grid->addWidget(cb1, row++, col);
      stripList.push_back(strip);
      }

//---------------------------------------------------------
//   AudioMixer
//
//    inputs | synthis | tracks | groups | master
//---------------------------------------------------------

AudioMixerApp::AudioMixerApp()
   : TopWin(0, "audiomixer", WType_TopLevel /*|WDestructiveClose*/)
      {
      setCaption("MusE: Audio Mixer");

      view = new QScrollView(this);
      view->setResizePolicy(QScrollView::AutoOneFit);
      view->setVScrollBarMode(QScrollView::AlwaysOff);

      setCentralWidget(view);
      central = new QWidget(view);
      central->setSizePolicy(
         QSizePolicy(QSizePolicy::Preferred, QSizePolicy::Expanding));
      view->addChild(central);

      createMixer();

      setFixedWidth(STRIP_WIDTH * stripList.size() + 8);

      connect(song, SIGNAL(heartBeat()), SLOT(heartBeat()));
      connect(song, SIGNAL(songChanged(int)), SLOT(songChanged(int)));
      }

//---------------------------------------------------------
//   createMixer
//---------------------------------------------------------

void AudioMixerApp::createMixer()
      {
      grid = new QGridLayout(central);
      grid->setRowStretch(4, 100);   // slider

      // clear old strip list:

      for (iStrip i = stripList.begin(); i != stripList.end(); ++i)
            delete *i;
      stripList.clear();

      int col = 0;

      //---------------------------------------------------
      //  generate Input Strips
      //---------------------------------------------------

      AudioPort* port  = &audioPort;
      AudioDevice* dev = port->device();
      if (dev && (port->rwFlags() & 2)) {
            QLabel* label = new QLabel("Inputs", central);
            label->setBackgroundColor(red);
            label->setAlignment(AlignCenter);
            grid->addMultiCellWidget(label, 0, 0, col, col);
            genStrip(dev->name(), col++, port, INPUT);
            }

      //---------------------------------------------------
      //  Synthesizer Strips
      //---------------------------------------------------

      int sstrips = synthiInstances.size();
      if (sstrips) {
            QLabel* label = new QLabel("Synthi", central);
            label->setBackgroundColor(blue);
            label->setAlignment(AlignCenter);
            grid->addMultiCellWidget(label, 0, 0, col, col+sstrips-1);
            for (iSynthI si = synthiInstances.begin(); si != synthiInstances.end(); ++si) {
                  genStrip((*si)->iname(), col++, *si, SYNTHI);
                  }
            }

      //---------------------------------------------------
      //  generate Audio Track Strips
      //---------------------------------------------------

      TrackList* tl = song->tracks();
      int tstrips = 0;
      for (iTrack i = tl->begin(); i != tl->end(); ++i) {
            if ((*i)->type() != Track::WAVE)
                  continue;
            ++tstrips;
            }
      if (tstrips) {
            QLabel* label = new QLabel("Tracks", central);
            label->setBackgroundColor(green);
            label->setAlignment(AlignCenter);
            grid->addMultiCellWidget(label, 0, 0, col, col+tstrips-1);
            for (iTrack t = tl->begin(); t != tl->end(); ++t) {
                  if ((*t)->type() != Track::WAVE)
                        continue;
                  WaveTrack* track = (WaveTrack*)(*t);
                  genStrip(track->tname(), col++, track, TRACK);
                  }
            }

      //---------------------------------------------------
      //  Groups
      //---------------------------------------------------

      int groups = song->mixerGroups();
      QLabel* label = new QLabel("Groups", central);
      label->setBackgroundColor(yellow);
      label->setAlignment(AlignCenter);
      grid->addMultiCellWidget(label, 0, 0, col, col+groups-1);

      char* grouplabel[] = { "A","B","C","D","E","F","G","H" };
      for (int i = 0; i < groups; ++i) {
            AudioNode* src = song->group(i);
            genStrip(grouplabel[i], col++, src, GROUP);
            }

      //---------------------------------------------------
      //    Master
      //---------------------------------------------------

      label = new QLabel("Master", central);
      label->setFixedHeight(17);
      label->setBackgroundColor(white);
      label->setAlignment(AlignCenter);
      grid->addMultiCellWidget(label, 0, 0, col, col);
      genStrip("", col++, song->master(), MASTER);

      grid->setColStretch(col, 1000);
      }

//---------------------------------------------------------
//   songChanged
//---------------------------------------------------------

void AudioMixerApp::songChanged(int /*val*/)
      {
//      masterStrip->setPorts();
//      masterStrip->setOutputRoute(song->outRoute(masterStrip->id()));
      }

//---------------------------------------------------------
//   closeEvent
//---------------------------------------------------------

void AudioMixerApp::closeEvent(QCloseEvent* e)
      {
      emit deleted((int)this);
      e->accept();
      }

//---------------------------------------------------------
//   readStatus
//---------------------------------------------------------

void AudioMixerApp::readStatus(Xml& xml)
      {
      for (;;) {
            Xml::Token token = xml.parse();
            const QString& tag = xml.s1();
            switch (token) {
                  case Xml::Error:
                  case Xml::End:
                        return;
                  case Xml::TagStart:
                        if (tag == "topwin")
                              TopWin::readStatus(xml);
                        else
                              xml.unknown("AudioMixer");
                        break;
                  case Xml::TagEnd:
                        if (tag == "audiomixer")
                              return;
                  default:
                        break;
                  }
            }
      }

//---------------------------------------------------------
//   writeStatus
//---------------------------------------------------------

void AudioMixerApp::writeStatus(int level, Xml& xml) const
      {
      xml.tag(level++, "audiomixer");
      TopWin::writeStatus(level, xml);
      xml.tag(level, "/audiomixer");
      }

//---------------------------------------------------------
//   heartBeat
//    update meter & peak display
//---------------------------------------------------------

void AudioMixerApp::heartBeat()
      {
      for (iStrip i = stripList.begin(); i != stripList.end(); ++i) {
            AudioNode* as = (*i)->src;
            for (int ch = 0; ch < as->channels(); ++ch) {
                  int meter = as->activity(ch);
                  int peak  = as->peak(ch);
                  (*i)->meter[ch]->setVal(meter, peak, false);
                  }
            }
      }

//---------------------------------------------------------
//   updateMixer
//---------------------------------------------------------

void AudioMixerApp::updateMixer()
      {
      delete central;

      central = new QWidget(view);
      central->setSizePolicy(
         QSizePolicy(QSizePolicy::Preferred, QSizePolicy::Expanding));
      view->addChild(central);

      createMixer();
      central->show();

      setFixedWidth(STRIP_WIDTH * stripList.size() + 8);
      }

