//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: audiothread.cpp,v 1.4 2002/02/27 11:52:58 muse Exp $
//
//  (C) Copyright 2001 Werner Schweer (ws@seh.de)
//=========================================================

#include <poll.h>
#include "audiothread.h"
#include "song.h"
#include "audionode.h"
#include "audioport.h"
#include "driver/audiodev.h"
#include "driver/mididev.h"
#include "seq.h"
#include "sf/sndfile.h"
#include "synth.h"
#include "ssource.h"
#include "audioprefetch.h"

//---------------------------------------------------------
//   AudioThread
//---------------------------------------------------------

AudioThread::AudioThread(bool rt=true, int prio=50, bool ml = true)
   : Thread(rt, prio, ml)
      {
      _masterMix     = 0;
      playState      = false;
      curSamplePos   = 0;
      _mixdownRecord = false;
      }

//---------------------------------------------------------
//   start
//---------------------------------------------------------

void AudioThread::start()
      {
      updatePollFd();
      if (audioPort.device())
            audioPort.device()->start();
      Thread::start();
      }

//---------------------------------------------------------
//   AudioThread
//---------------------------------------------------------

AudioThread::~AudioThread()
      {
      if (_masterMix) {
            if (_mixdownRecord)
                  _masterMix->close();
            delete _masterMix;
            }
      }

//---------------------------------------------------------
//   audioTick
//    master audio output is writable
//---------------------------------------------------------

void AudioThread::audioTick(AudioPort* port)
      {
      if (playState) {
            port->write(audioThread->mixdownRecord() ? _masterMix : 0, curSamplePos, rPrefetchIdx);
            curSamplePos += segmentSize;
            rPrefetchIdx += 1;
            rPrefetchIdx %= PREFETCH_BUFFER;

            audioPrefetch->msgTick(prefetchSamplePos, wPrefetchIdx);
            prefetchSamplePos += segmentSize;
            wPrefetchIdx += 1;
            wPrefetchIdx %= PREFETCH_BUFFER;
            }
      else
            port->write(0, 0, 0);
      }

//---------------------------------------------------------
//   defaultTick
//    debug hook
//---------------------------------------------------------

void AudioThread::defaultTick()
      {
      audioTick(&audioPort);
      }

//---------------------------------------------------------
//   audioRead
//---------------------------------------------------------

void AudioThread::audioRead(AudioPort* port)
      {
      port->read();
      }

//---------------------------------------------------------
//   audioWrite
//---------------------------------------------------------

#if 0
void AudioThread::audioWrite(AudioPort* port)
      {
      port->write(0, curSamplePos);
      }
#endif

//---------------------------------------------------------
//   audioTick
//    master audio output is writable
//---------------------------------------------------------

static void audioTick(void* p, void *q)
      {
      AudioThread* at = (AudioThread*)p;
      at->audioTick((AudioPort*)q);
      }

//---------------------------------------------------------
//   audioRead
//---------------------------------------------------------

static void audioRead(void* p, void* q)
      {
      AudioThread* at = (AudioThread*)p;
      at->audioRead((AudioPort*)q);
      }
#if 0
//---------------------------------------------------------
//   audioWrite
//---------------------------------------------------------

static void audioWrite(void* p, void *q)
      {
      AudioThread* at = (AudioThread*)p;
      at->audioWrite((AudioPort*)q);
      }
#endif
//---------------------------------------------------------
//   readMsg
//---------------------------------------------------------

static void readMsg(void* p, void*)
      {
      AudioThread* at = (AudioThread*)p;
      at->readMsg();
      }

//---------------------------------------------------------
//   updatePollFd
//---------------------------------------------------------

void AudioThread::updatePollFd()
      {
      clearPollFd();
      addPollFd(toThreadFdr, POLLIN, ::readMsg, this, 0);

      //---------------------------------------------------
      //  activate audio ports
      //---------------------------------------------------

      AudioPort* port = &audioPort;
      AudioDevice* dev = port->device();
      if (dev == 0)     // no dev - no audio
            return;

      if (port->rwFlags() & 0x1) {
            int fd = dev->selectwfd();
            if (fd != -1) {
                  addPollFd(fd, POLLOUT, ::audioTick, this, port);
                  }
            }

      //---------------------------------------
      // select on audio input only if
      // audioport has capture flag
      //---------------------------------------

      if (port->rwFlags() & 0x2) {
            int fd = dev->selectrfd();
            if (fd != -1) {
                  addPollFd(fd, POLLIN, ::audioRead, this, port);
                  }
            }
      }

//---------------------------------------------------------
//   processMsg
//---------------------------------------------------------

void AudioThread::processMsg(const ThreadMsg* m)
      {
      const AudioMsg* msg = (AudioMsg*)m;
      switch(msg->id) {
            case AUDIO_SET_MIXDOWN:
                  if (_masterMix)
                        delete _masterMix;
                  _masterMix = msg->downmix;
                  break;
            case AUDIO_OPEN_MIXDOWN:
                  _masterMix->openWrite();
                  _mixdownRecord = true;
                  break;
            case AUDIO_CLOSE_MIXDOWN:
                  _masterMix->close();
                  _mixdownRecord = false;
                  break;
            case AUDIO_ROUTEADD:
                  {
                  AudioNode* n = msg->snode->route();
                  if (n)
                        n->disconnect(msg->snode);
                  msg->snode->setRoute(msg->dnode);
                  if (msg->dnode)
                        msg->dnode->connect(msg->snode);
                  }
                  break;
            case AUDIO_ROUTEREMOVE:
                  msg->snode->setRoute(0);
                  msg->dnode->disconnect(msg->snode);
                  break;
            case AUDIO_VOL:
                  msg->snode->setVolume(msg->dval);
                  break;
            case AUDIO_PAN:
                  msg->snode->setPan(msg->dval);
                  break;
            case AUDIO_SET_PREFADER:
                  msg->snode->setPrefader(msg->ival);
                  break;
            case AUDIO_SET_CHANNELS:
                  msg->snode->setChannels(msg->ival);
                  break;
            case AUDIO_ADDPLUGIN:
                  msg->snode->efxPipe()->insert(msg->plugin, msg->ival);
                  break;
            case AUDIO_MUTE:
                  msg->snode->setMute(msg->ival);
                  break;
            case AUDIO_REMOVE_SYNTHI:
                  {
                  SynthI* synthi = msg->synth;

                  removeMidiInstrument(synthi->iname());

#if 0
                  for (int i = 0; i < MIDI_PORTS; ++i) {
                        MidiDevice* dev = midiPorts[i].device();
                        if (dev && dev->name() == synthi->iname()) {
                              midiPorts[i].setMidiDevice(0);
                              midiPorts[i].setInstrument(0);
                              break;
                              }
                        }
#endif

                  synthi->route()->disconnect(synthi);
                  for (iSynthI i = synthiInstances.begin();
                     i != synthiInstances.end(); ++i) {
                        if (*i == synthi) {
                              synthiInstances.erase(i);
                              break;
                              }
                        }
                  delete synthi;
                  }
                  break;
            case AUDIO_ADD_SYNTHI:
                  synthiInstances.push_back(msg->synth);
                  msg->synth->route()->connect(msg->synth);
                  break;

            case AUDIO_SET_DEVICE:
                  msg->port->setAudioDevice(msg->device);
                  break;

            case AUDIO_SET_SEG_SIZE:
                  segmentSize = msg->ival;
                  sampleRate  = msg->iival;
                  song->master()->segmentSizeChanged();
                  for (int i = 0; i < song->mixerGroups(); ++i)
                        song->group(i)->segmentSizeChanged();
                  audioPort.setup(5, segmentSize * audioPort.channels());
                  for (iSynthI ii = synthiInstances.begin(); ii != synthiInstances.end();++ii)
                        (*ii)->segmentSizeChanged();

                  AudioDevice* dev = audioPort.device();
                  if (dev) {
                        dev->close();
                        dev->open(audioPort.rwFlags());
                        }
                  break;
            }
      updatePollFd();
      }

//---------------------------------------------------------
//   msgSetMixdownFile
//---------------------------------------------------------

void AudioThread::msgSetMixdownFile(SndFile* sf)
      {
      AudioMsg msg;
      msg.id = AUDIO_SET_MIXDOWN;
      msg.downmix = sf;
      sendMsg(&msg);
      }

void AudioThread::msgOpenMixdownRecord()
      {
      AudioMsg msg;
      msg.id = AUDIO_OPEN_MIXDOWN;
      sendMsg(&msg);
      }

void AudioThread::msgCloseMixdownRecord()
      {
      AudioMsg msg;
      msg.id = AUDIO_CLOSE_MIXDOWN;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgRemoveRoute
//---------------------------------------------------------

void AudioThread::msgRemoveRoute(AudioNode* src, AudioNode* dst)
      {
      AudioMsg msg;
      msg.id = AUDIO_ROUTEREMOVE;
      msg.snode = src;
      msg.dnode = dst;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgAddRoute
//---------------------------------------------------------

void AudioThread::msgAddRoute(AudioNode* src, AudioNode* dst)
      {
      AudioMsg msg;
      msg.id = AUDIO_ROUTEADD;
      msg.snode = src;
      msg.dnode = dst;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgAddPlugin
//---------------------------------------------------------

void AudioThread::msgAddPlugin(AudioNode* node, int idx, PluginI* plugin)
      {
      AudioMsg msg;
      msg.id     = AUDIO_ADDPLUGIN;
      msg.snode  = node;
      msg.ival   = idx;
      msg.plugin = plugin;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgSetMute
//---------------------------------------------------------

void AudioThread::msgSetMute(AudioNode* node, bool val)
      {
      AudioMsg msg;
      msg.id     = AUDIO_MUTE;
      msg.snode  = node;
      msg.ival   = int(val);
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgSetVolume
//---------------------------------------------------------

void AudioThread::msgSetVolume(AudioNode* src, double val)
      {
      AudioMsg msg;
      msg.id    = AUDIO_VOL;
      msg.snode    = src;
      msg.dval  = val;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgRemoveSynthi
//---------------------------------------------------------

void AudioThread::msgRemoveSynthI(SynthI* synth)
      {
      AudioMsg msg;
      msg.id = AUDIO_REMOVE_SYNTHI;
      msg.synth = synth;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgAddSynthi
//---------------------------------------------------------

void AudioThread::msgAddSynthI(SynthI* synth)
      {
      AudioMsg msg;
      msg.id = AUDIO_ADD_SYNTHI;
      msg.synth = synth;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgSetPan
//---------------------------------------------------------

void AudioThread::msgSetPan(AudioNode* node, double val)
      {
      AudioMsg msg;
      msg.id    = AUDIO_PAN;
      msg.snode = node;
      msg.dval  = val;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgSetPrefader
//---------------------------------------------------------

void AudioThread::msgSetPrefader(AudioNode* node, int val)
      {
      AudioMsg msg;
      msg.id    = AUDIO_SET_PREFADER;
      msg.snode = node;
      msg.ival  = val;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgSetChannels
//---------------------------------------------------------

void AudioThread::msgSetChannels(AudioNode* node, int val)
      {
      AudioMsg msg;
      msg.id    = AUDIO_SET_CHANNELS;
      msg.snode = node;
      msg.ival  = val;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   msgSetAudioDevice
//---------------------------------------------------------

void AudioThread::msgSetAudioDevice(AudioPort* port, AudioDevice* device)
      {
      AudioMsg msg;
      msg.id = AUDIO_SET_DEVICE;
      msg.port = port;
      msg.device = device;
      sendMsg(&msg);
      song->updateAudioMixer();
      }

//---------------------------------------------------------
//   msgSetSegSize
//---------------------------------------------------------

void AudioThread::msgSetSegSize(int bs, int sr)
      {
      AudioMsg msg;
      msg.id = AUDIO_SET_SEG_SIZE;
      msg.ival = bs;
      msg.iival = sr;
      sendMsg(&msg);
      }

//---------------------------------------------------------
//   play
//---------------------------------------------------------

void AudioThread::play(int tickpos)
      {
//      printf("PLAY\n");
      playState = true;
      seek(tickpos);
      }

//---------------------------------------------------------
//   seek
//---------------------------------------------------------

void AudioThread::seek(int tickpos)
      {
      if (noAudio)
            return;
//      printf("SEEK\n");
      curSamplePos = tempomap.tick2samples(tickpos);
      prefetchSamplePos = curSamplePos;
      rPrefetchIdx      = 0;
      //
      // schedule prefetch
      //
      for (wPrefetchIdx = 0; wPrefetchIdx < PREFETCH_BUFFER; ++wPrefetchIdx) {
            audioPrefetch->msgTick(prefetchSamplePos, wPrefetchIdx);
            prefetchSamplePos += segmentSize;
            }
      wPrefetchIdx = 0;
      }

//---------------------------------------------------------
//   stopPlay
//---------------------------------------------------------

void AudioThread::stopPlay()
      {
//      printf("STOP\n");
      playState = false;
      AudioDevice* dev = audioPort.device();
      if (dev)
            audioPort.resetMeter();
      for (int i = 0; i < song->mixerGroups(); ++i)
            song->group(i)->resetMeter();
      song->master()->resetMeter();
      }

