/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <math.h>
#include "MRI.h"

struct ConvolverData {
	struct link *next;
	int	width;
	int	height;
	int	mask;
	double	minval, maxval;
	int	freedata;
	int	radius, tworp1;
	struct MRI_ScanLine **fp;
	double  *M;
};

static void
ConvolverStart (void *private, int width, int height, int freedata)
{
	struct ConvolverData *wd = private;
	int i;

	(*wd->next->start) (wd->next->private, width, height, TRUE);
	wd->width = width;
	wd->height = height;
	wd->freedata = freedata;
	for (i = 0; i < wd->tworp1; i++) {
		wd->fp[i] = (struct MRI_ScanLine *)0;
	}
}

static double
ConvolveR (struct ConvolverData *wd, int x)
{
	double rval = 0.0;
	int i, j, k =0;

	for (i = 0; i < wd->tworp1; i++) {
		double *R = wd->fp[i]->R;
		for (j = -wd->radius; j <= wd->radius; j++)
			rval += wd->M[k++] * R[x+j];
	}
	return rval;
}

static double
ConvolveG (struct ConvolverData *wd, int x)
{
	double rval = 0.0;
	int i, j, k =0;

	for (i = 0; i < wd->tworp1; i++) {
		double *G = wd->fp[i]->G;
		for (j = -wd->radius; j <= wd->radius; j++)
			rval += wd->M[k++] * G[x+j];
	}
	return rval;
}

static double
ConvolveB (struct ConvolverData *wd, int x)
{
	double rval = 0.0;
	int i, j, k =0;

	for (i = 0; i < wd->tworp1; i++) {
		double *B = wd->fp[i]->B;
		for (j = -wd->radius; j <= wd->radius; j++)
			rval += wd->M[k++] * B[x+j];
	}
	return rval;
}

static void
ConvolverRow (void *private, void *data)
{
	struct ConvolverData *wd = private;
	int i;

	wd->fp[2*wd->radius] = (struct MRI_ScanLine *)data;

	if (wd->fp[wd->radius] != (struct MRI_ScanLine *)0) {
		int x;
		struct MRI_ScanLine *np = MRI_NewScanLine (LINETYPE_DOUBLE, wd->width);
		double *R = np->R;
		double *G = np->G;
		double *B = np->B;
		double *wdR = wd->fp[wd->radius]->R;
		double *wdG = wd->fp[wd->radius]->G;
		double *wdB = wd->fp[wd->radius]->B;
		if (wd->fp[0] == (struct MRI_ScanLine *)0) {
			for (x = 0; x < wd->width; x++) {
				R[x] = wdR[x];
				G[x] = wdG[x];
				B[x] = wdB[x];
			}
		}
		else {
			for (x = 0; x < wd->radius; x++) {
				R[x] = wdR[x];
				G[x] = wdG[x];
				B[x] = wdB[x];
			}
			for (x = wd->radius; x < wd->width-wd->radius; x++) {
				R[x] = wdR[x];
				G[x] = wdG[x];
				B[x] = wdB[x];
				if (wd->mask & TONE_R) {
					R[x] = ConvolveR(wd, x);
					if (R[x] < wd->minval)
						R[x] = wd->minval;
					else if (R[x] > wd->maxval)
						R[x] = wd->maxval;
				}
				if (wd->mask & TONE_G) {
					G[x] = ConvolveG(wd, x);
					if (G[x] < wd->minval)
						G[x] = wd->minval;
					else if (G[x] > wd->maxval)
						G[x] = wd->maxval;
				}
				if (wd->mask & TONE_B) {
					B[x] = ConvolveB(wd, x);
					if (B[x] < wd->minval)
						B[x] = wd->minval;
					else if (B[x] > wd->maxval)
						B[x] = wd->maxval;
				}
			}
			for (x = wd->width-wd->radius; x < wd->width; x++) {
				R[x] = wdR[x];
				G[x] = wdG[x];
				B[x] = wdB[x];
			}
			if (wd->freedata)
				MRI_FreeScanLine (wd->fp[0]);
		}
		(*wd->next->row) (wd->next->private, np);
	}
	for (i = 0; i < 2*wd->radius; i++)
		wd->fp[i] = wd->fp[i+1];
}

static void
ConvolverClose (void *private)
{
	int i, x;
	struct ConvolverData *wd = private;

	for (i = wd->radius; i < 2*wd->radius; i++) {
		struct MRI_ScanLine *np = MRI_NewScanLine (LINETYPE_DOUBLE, wd->width);
		double *R = np->R;
		double *G = np->G;
		double *B = np->B;
		double *wdR = wd->fp[i]->R;
		double *wdG = wd->fp[i]->G;
		double *wdB = wd->fp[i]->B;
		for (x = 0; x < wd->width; x++) {
			R[x] = wdR[x];
			G[x] = wdG[x];
			B[x] = wdB[x];
		}
		(*wd->next->row) (wd->next->private, np);
	}
	(*wd->next->close) (wd->next->private);

	if (wd->freedata) {
		for (i = 0; i < 2*wd->radius; i++)
			MRI_FreeScanLine (wd->fp[i]);
	}
	free (wd->next);
	free (wd->M);
	free (wd->fp);
	free (wd);
}

struct link *
MRI_GenConvolver (int mask, double minval, double maxval, int radius, double M[], struct link *next)
{
	int i;
	struct ConvolverData *wd = malloc (sizeof (struct ConvolverData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct ConvolverData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = ConvolverStart;
	ep->row = ConvolverRow;
	ep->close = ConvolverClose;
	ep->private = wd;
	wd->next = next;

	wd->mask = mask;
	wd->minval = minval;
	wd->maxval = maxval;
	wd->radius = radius;
	wd->tworp1 = 2 * radius + 1;
	wd->fp = (struct MRI_ScanLine **)malloc(sizeof(struct MRI_ScanLine *)* wd->tworp1);
	wd->M = (double *)malloc(sizeof(double)* wd->tworp1 * wd->tworp1);
	if (wd->fp == (struct MRI_ScanLine **)0 || wd->M == (double *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	for (i = 0; i < wd->tworp1; i++) wd->fp[i] = (struct MRI_ScanLine *)0;
	for (i = 0; i < (wd->tworp1 * wd->tworp1); i++) wd->M[i] = M[i];
	return ep;
}
