/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2001 CodeFactory AB
 * Copyright (C) 2001 Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Mikael Hallendal <micke@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "util/marshallers.h"
#include "util/type-utils.h"
#include "task-manager-client.h"

static void  task_mc_init            (TaskManagerClient        *tmc);
static void  task_mc_class_init      (TaskManagerClientClass   *klass);
static void  tmc_destroy             (GtkObject                *object);
static void  tmc_listener            (ManagerClient            *mc,
				      gchar                    *subtype,
				      CORBA_any                *any);
static void  tmc_project_set         (ManagerClient            *mc);
static void  tmc_project_unset       (ManagerClient            *mc);

#define d(x)

enum {
	TASK_INSERTED,
	TASKS_REMOVED,
	TASK_UPDATED,
	TASK_REPOSITIONED,
	TASK_REPARENTED,
	TASKS_LINKED,
	TASKS_UNLINKED,
	NOTE_CHANGED,
	LAST_SIGNAL
};

static gint signals[LAST_SIGNAL] = { 0 };

struct _TaskManagerClientPriv {
        GM_TaskManager manager;
};

GNOME_CLASS_BOILERPLATE (TaskManagerClient, task_mc,
                         ManagerClient, manager_client);

static void
task_mc_init (TaskManagerClient *tmc)
{
        TaskManagerClientPriv *priv;
        
        priv = g_new0 (TaskManagerClientPriv, 1);
        
        tmc->priv = priv;
}

static void
task_mc_class_init (TaskManagerClientClass *klass)
{
	ManagerClientClass *mcc;
	GtkObjectClass     *object_class;
	
	mcc          = MANAGER_CLIENT_CLASS (klass);
	object_class = (GtkObjectClass *) klass;

	mcc->listener         = tmc_listener;
	mcc->project_set      = tmc_project_set;
	mcc->project_unset    = tmc_project_unset;

        object_class->destroy = tmc_destroy;

	signals[TASK_INSERTED] = 
		gtk_signal_new ("task_inserted",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (TaskManagerClientClass,
						   task_inserted),
				gtk_marshal_NONE__POINTER_INT_INT,
				GTK_TYPE_NONE,
				3, GTK_TYPE_POINTER, GTK_TYPE_INT, GTK_TYPE_INT);
	signals[TASKS_REMOVED] = 
		gtk_signal_new ("tasks_removed",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (TaskManagerClientClass,
						   tasks_removed),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE,
				1, GTK_TYPE_POINTER);
	signals[TASK_UPDATED] = 
		gtk_signal_new ("task_updated",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (TaskManagerClientClass,
						   task_updated),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE,
				1, GTK_TYPE_POINTER);
	signals[TASK_REPOSITIONED] = 
		gtk_signal_new ("task_repositioned",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (TaskManagerClientClass,
						   task_repositioned),
				mrproject_marshal_NONE__INT_INT_INT,
				GTK_TYPE_NONE,
				3, GTK_TYPE_INT, GTK_TYPE_INT, GTK_TYPE_INT);
	signals[TASK_REPARENTED] = 
		gtk_signal_new ("task_reparented",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (TaskManagerClientClass,
						   task_reparented),
				gtk_marshal_NONE__INT_INT,
				GTK_TYPE_NONE,
				2, GTK_TYPE_INT, GTK_TYPE_INT);
	signals[TASKS_LINKED] = 
		gtk_signal_new ("tasks_linked",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (TaskManagerClientClass,
						   tasks_linked),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE,
				1, GTK_TYPE_POINTER);
	signals[TASKS_UNLINKED] = 
		gtk_signal_new ("tasks_unlinked",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (TaskManagerClientClass,
						   tasks_unlinked),
				gtk_marshal_NONE__POINTER,
				GTK_TYPE_NONE,
				1, GTK_TYPE_POINTER);
	signals[NOTE_CHANGED] =
		gtk_signal_new ("note_changed",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (TaskManagerClientClass,
						   note_changed),
				gtk_marshal_NONE__INT_POINTER,
				GTK_TYPE_NONE,
				2, GTK_TYPE_INT, GTK_TYPE_POINTER);

	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);
}

static void
tmc_destroy (GtkObject *object)
{
        TaskManagerClient     *tmc;
	TaskManagerClientPriv *priv;

	d(puts (__FUNCTION__));
	
        g_return_if_fail (object != NULL);
        g_return_if_fail (IS_TASK_MANAGER_CLIENT (object));
        
        tmc  = TASK_MANAGER_CLIENT (object);
	priv = tmc->priv;

	bonobo_object_release_unref (priv->manager, NULL);
	priv->manager = NULL;

        g_free (tmc->priv);
        tmc->priv = NULL;

	GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static void 
tmc_listener (ManagerClient *mc, gchar *subtype, CORBA_any *any)
{
	TaskManagerClient     *tmc;
	TaskManagerClientPriv *priv;
	GM_Task               *task;
	GM_Dependency         *dependency;
	GSList                *list;
	
	g_return_if_fail (mc != NULL);
	g_return_if_fail (IS_TASK_MANAGER_CLIENT (mc));
	
	tmc  = TASK_MANAGER_CLIENT (mc);
	priv = tmc->priv;
	
	/* Emit signals */

	if (!strcmp (subtype, "inserted")) {
		GM_EventTaskInserted *event;
		gint sibling_id, type;
		
		event = any->_value;

		task = &event->theTask;
		sibling_id = event->siblingId;
		type = event->type;
		
		gtk_signal_emit (GTK_OBJECT (tmc),
				 signals[TASK_INSERTED],
				 task,
				 sibling_id,
				 type);
	}
	else if (!strcmp (subtype, "updated")) {
		task = (GM_Task *) any->_value;
		gtk_signal_emit (GTK_OBJECT (tmc),
				 signals[TASK_UPDATED],
				 task);
	}
	else if (!strcmp (subtype, "removed_seq")) {
		BonoboArg *arg = (BonoboArg *) any;
		gchar *str;
		
		str = BONOBO_ARG_GET_STRING (arg);
		list = corba_util_id_string_to_list (str);
		
		gtk_signal_emit (GTK_OBJECT (tmc),
				 signals[TASKS_REMOVED],
				 list);

		g_slist_free (list);
	}
	else if (!strcmp (subtype, "repositioned")) {
		GNOME_MrProject_EventTaskRepositioned *event;
		
		event = (GNOME_MrProject_EventTaskRepositioned *) any->_value;
		gtk_signal_emit (GTK_OBJECT (tmc),
				 signals[TASK_REPOSITIONED],
				 event->taskId, 
				 event->siblingId, 
				 event->type);
	}
	else if (!strcmp (subtype, "reparented")) {
		GNOME_MrProject_EventTaskReparented *event;
		
		event = (GNOME_MrProject_EventTaskReparented *) any->_value;
		gtk_signal_emit (GTK_OBJECT (tmc),
				 signals[TASK_REPARENTED],
				 event->taskId, 
				 event->newParentId);
	}
	else if (!strcmp (subtype, "linked")) { 
		dependency = (GM_Dependency *) any->_value;
		
		gtk_signal_emit (GTK_OBJECT (tmc),
				 signals[TASKS_LINKED],
				 dependency);
	}
	else if (!strcmp (subtype, "unlinked")) {
		dependency = (GM_Dependency *) any->_value;
		
		gtk_signal_emit (GTK_OBJECT (tmc),
				 signals[TASKS_UNLINKED],
				 dependency);
	}
	else if (!strcmp (subtype, "note_changed")) {
		GNOME_MrProject_EventNoteChanged *event;
		
		event = (GNOME_MrProject_EventNoteChanged *) any->_value;
		
		gtk_signal_emit (GTK_OBJECT (tmc),
				 signals[NOTE_CHANGED],
				 event->taskId, event->note);
	} else {
		d(g_print ("TaskManagerClient: got unhandled event: %s\n",
			   subtype));
	}
}

static void
tmc_project_set (ManagerClient *mc) 
{
	TaskManagerClient     *tmc;
	TaskManagerClientPriv *priv;
	GM_Project             project_co;
	CORBA_Environment      ev;
	
	g_return_if_fail (mc != NULL);
	g_return_if_fail (IS_TASK_MANAGER_CLIENT (mc));

	d(puts (__FUNCTION__));

	tmc  = TASK_MANAGER_CLIENT (mc);
	priv = tmc->priv;

	if (priv->manager) {
		bonobo_object_release_unref (priv->manager, NULL);
	}
	
	if (!mc->shell) {
		g_warning ("TaskManagerClient doesn't have a shell");
		return;
	}
	
	CORBA_exception_init (&ev);

	project_co = GNOME_MrProject_Shell_getProject (mc->shell, &ev);
	
	if (BONOBO_EX (&ev) || !project_co) {
		g_warning ("Shell doesn't have a project");
		CORBA_exception_free (&ev);
		return;
	}

	priv->manager = Bonobo_Unknown_queryInterface (project_co,
						       "IDL:GNOME/MrProject/TaskManager:1.0",
						       &ev);
	
	if (BONOBO_EX (&ev)) {
		g_warning ("Couldn't get TaskManager interface through project");
	}

	CORBA_Object_release (project_co, NULL);

	CORBA_exception_free (&ev);
}

static void
tmc_project_unset (ManagerClient *mc)
{
	TaskManagerClient     *tmc;
	TaskManagerClientPriv *priv;

	g_return_if_fail (mc != NULL);
	g_return_if_fail (IS_TASK_MANAGER_CLIENT (mc));
 	
	tmc  = TASK_MANAGER_CLIENT (mc);
	priv = tmc->priv;

	if (priv->manager) {
		bonobo_object_release_unref (priv->manager, NULL);
	}

	priv->manager = CORBA_OBJECT_NIL; 
}

ManagerClient * 
task_mc_new (GM_Shell shell, gboolean set, CORBA_Environment *ev)
{
	TaskManagerClient     *tmc;
	TaskManagerClientPriv *priv;

	d(puts(__FUNCTION__));
	
	tmc  = gtk_type_new (TASK_MANAGER_CLIENT_TYPE);
	priv = tmc->priv;
	
	manager_client_set_shell (MANAGER_CLIENT (tmc),
				  shell,
				  "task",
				  ev);

	if (set) { 
		tmc_project_set (MANAGER_CLIENT(tmc));
	}

	return MANAGER_CLIENT(tmc);
}

GM_Task *       
task_mc_create_task (TaskManagerClient *tmc, CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	GM_Task               *task;
	
	g_return_val_if_fail (tmc != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MANAGER_CLIENT (tmc), NULL);
	
	priv = tmc->priv;
	
	task = GNOME_MrProject_TaskManager_createTask (priv->manager, ev);

	/* FIX: Check for exceptions */

	return task;
}

GM_Id
task_mc_insert_task (TaskManagerClient *tmc,
		     GM_Task           *task,
		     GM_Id              parent_id,
		     CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	gboolean               free_task = FALSE;
	GM_Id                  id;
	
	g_return_val_if_fail (tmc != NULL, -1);
	g_return_val_if_fail (IS_TASK_MANAGER_CLIENT (tmc), -1);
	
	priv = tmc->priv;

	if (!task) {
		task = task_mc_create_task (tmc, ev);
		free_task = TRUE;
	}

	id = GNOME_MrProject_TaskManager_insertTask (priv->manager,
						     task,
						     parent_id,
						     ev);
	
	/* FIX: Check for exceptions */

	if (free_task) {
		CORBA_free (task);
	}

	return id;
}

GM_Id
task_mc_insert_task_full (TaskManagerClient *tmc,
			  GM_Task           *task,
			  GM_Id              parent_id,
			  GM_Id              sibling_id,
			  GM_TaskOrderType   type,
			  CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	gboolean               free_task = FALSE;
	GM_Id                  id;
	
	g_return_val_if_fail (tmc != NULL, -1);
	g_return_val_if_fail (IS_TASK_MANAGER_CLIENT (tmc), -1);
	
	priv = tmc->priv;

	if (!task) {
		task = task_mc_create_task (tmc, ev);
		free_task = TRUE;
	}

	id = GNOME_MrProject_TaskManager_insertTaskFull (priv->manager,
							 task,
							 parent_id,
							 sibling_id,
							 type,
							 ev);
	
	/* FIX: Check for exceptions */

	if (free_task) {
		CORBA_free (task);
	}

	return id;
}

void
task_mc_update_task (TaskManagerClient *tmc,
		     GM_Id              task_id,
		     GM_Task           *task,
		     CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	
	g_return_if_fail (tmc != NULL);
	g_return_if_fail (IS_TASK_MANAGER_CLIENT (tmc));
	
	priv = tmc->priv;

	GNOME_MrProject_TaskManager_updateTask (priv->manager, 
						task_id, 
						task, 
						ev);
	
	/* FIX: Check for exceptions */
}

void
task_mc_move_task (TaskManagerClient *tmc,
		   GM_Id              id,
		   time_t             start,
		   GM_TaskEdge        edge,
		   CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	
	g_return_if_fail (tmc != NULL);
	g_return_if_fail (IS_TASK_MANAGER_CLIENT (tmc));
	
	priv = tmc->priv;

	GNOME_MrProject_TaskManager_moveTask (priv->manager, 
					      id, 
					      start,
					      edge,
					      ev);
}

void
task_mc_set_task_duration (TaskManagerClient *tmc,
			   GM_Id              task_id,
			   GM_Time            duration,
			   GM_TaskEdge        relativeEdge,
			   CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	
	g_return_if_fail (tmc != NULL);
	g_return_if_fail (IS_TASK_MANAGER_CLIENT (tmc));
	
	priv = tmc->priv;

	GNOME_MrProject_TaskManager_setTaskDuration (priv->manager,
						     task_id,
						     duration,
						     relativeEdge,
						     ev);
}

void
task_mc_reparent_task (TaskManagerClient *tmc,
		       GM_Id              task_id,
		       GM_Id              parent_id,
		       CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	
	g_return_if_fail (tmc != NULL);
	g_return_if_fail (IS_TASK_MANAGER_CLIENT (tmc));
	
	priv = tmc->priv;

	GNOME_MrProject_TaskManager_reparentTask (priv->manager, 
						  task_id,
						  parent_id, 
						  ev);

	/* FIX: Check for exceptions */
}

void
task_mc_reposition_task (TaskManagerClient *tmc,
			 GM_Id              task_id,
			 GM_Id              sibling_id,
			 GM_TaskOrderType   type,
			 CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	
	g_return_if_fail (tmc != NULL);
	g_return_if_fail (IS_TASK_MANAGER_CLIENT (tmc));
	
	priv = tmc->priv;
	
	GNOME_MrProject_TaskManager_repositionTask (priv->manager,
						    task_id,
						    sibling_id, 
						    type, 
						    ev);

	/* FIX: Check for exceptions */
}

GM_Task *
task_mc_get_task (TaskManagerClient *tmc,
		  GM_Id              task_id,
		  CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	GM_Task               *task;
	
	g_return_val_if_fail (tmc != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MANAGER_CLIENT (tmc), NULL);
	
	priv = tmc->priv;

	task = GNOME_MrProject_TaskManager_getTask (priv->manager,
						    task_id,
						    ev);
	/* FIX: Check for exceptions */

	return task;
}

GSList *        
task_mc_get_all_tasks (TaskManagerClient *tmc, CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	GM_TaskSeq            *tasks;
	GSList                *list;
	
	g_return_val_if_fail (tmc != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MANAGER_CLIENT (tmc), NULL);
	
	priv = tmc->priv;

	tasks = GNOME_MrProject_TaskManager_getAllTasks (priv->manager, ev);

	if (BONOBO_EX (ev) || !tasks) {
		g_warning ("Couldn't get all tasks");
		return NULL;
	}
	
	list = corba_util_task_seq_to_list (tasks);
	
	return list;
}

void
task_mc_remove_tasks (TaskManagerClient *tmc,
		      GSList            *task_ids,
		      CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	GM_IdSeq              *tasks;
	
	g_return_if_fail (tmc != NULL);
	g_return_if_fail (IS_TASK_MANAGER_CLIENT (tmc));
	g_return_if_fail (task_ids != NULL);
	
	priv = tmc->priv;

	tasks = corba_util_id_seq_from_list (task_ids);

	GNOME_MrProject_TaskManager_removeTasks (priv->manager,
						 tasks,
						 ev);
	
	/* FIX: Check for exceptions */
	
	CORBA_free (tasks);
}

GM_Id
task_mc_link_tasks (TaskManagerClient *tmc,
		    GM_Id              task_id,
		    GM_Id              predecessor_id,
		    GM_DependencyType  type,
		    CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	GM_Id                  link_id;
	
	g_return_val_if_fail (tmc != NULL, -1);
	g_return_val_if_fail (IS_TASK_MANAGER_CLIENT (tmc), -1);
	
	priv = tmc->priv;

	link_id = GNOME_MrProject_TaskManager_linkTasks (priv->manager,
							 task_id,
							 predecessor_id,
							 type,
							 ev);

	if (BONOBO_EX (ev)) {
		g_warning ("Exception when linking tasks");
		return -1;
	}

	return link_id;
}

void
task_mc_unlink_tasks (TaskManagerClient *tmc,
		      GM_Id              task_id,
		      GM_Id              predecessor_id,
		      CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	
	g_return_if_fail (tmc != NULL);
	g_return_if_fail (IS_TASK_MANAGER_CLIENT (tmc));
	
	priv = tmc->priv;

	GNOME_MrProject_TaskManager_unlinkTasks (priv->manager,
						 task_id,
						 predecessor_id,
						 ev);
	/* FIX: Check for exceptions */
}

void
task_mc_remove_dependency (TaskManagerClient *tmc,
			   GM_Id              link_id,
			   CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	
	g_return_if_fail (tmc != NULL);
	g_return_if_fail (IS_TASK_MANAGER_CLIENT (tmc));
	
	priv = tmc->priv;

	GNOME_MrProject_TaskManager_removeDependency (priv->manager,
						      link_id,
						      ev);
	/* FIX: Check for exceptions */
}

GM_Dependency * 
task_mc_get_dependency (TaskManagerClient *tmc,
			GM_Id              dependency_id,
			CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	GM_Dependency         *dependency;
	
	g_return_val_if_fail (tmc != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MANAGER_CLIENT (tmc), NULL);
	
	priv = tmc->priv;

	dependency = GNOME_MrProject_TaskManager_getDependency (priv->manager,
								dependency_id,
								ev);

	if (BONOBO_EX (ev) || !dependency) {
		g_warning ("Exception when getting dependency");
		return NULL;
	}

	return dependency;
}

GSList *        
task_mc_get_predecessors (TaskManagerClient *tmc,
			  GM_Id              task_id,
			  CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	GM_DependencySeq      *predecessors;
	GSList                *list;
	
	g_return_val_if_fail (tmc != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MANAGER_CLIENT (tmc), NULL);
	
	priv = tmc->priv;

	predecessors = GNOME_MrProject_TaskManager_getPredecessors (priv->manager,
								    task_id,
								    ev);
	
	if (BONOBO_EX (ev) || !predecessors) {
		g_warning ("Exception when getting predecessor");
		return NULL;
	}

	list = corba_util_dependency_seq_to_list (predecessors);
	
	return list;
}

GSList *
task_mc_get_sucessors (TaskManagerClient *tmc,
		       GM_Id              task_id,
		       CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	GM_DependencySeq      *successors;
	GSList                *list;
	
	g_return_val_if_fail (tmc != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MANAGER_CLIENT (tmc), NULL);
	
	priv = tmc->priv;

	successors = GNOME_MrProject_TaskManager_getSuccessors (priv->manager,
								task_id,
								ev);
	
	if (BONOBO_EX (ev) || !successors) {
		g_warning ("Exception when getting successors");
		return NULL;
	}

	list = corba_util_dependency_seq_to_list (successors);
	
	return list;
}

gchar *
task_mc_get_note (TaskManagerClient *tmc, 
		  GM_Id              task_id, 
		  CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	gchar                 *note;
	
	g_return_val_if_fail (tmc != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MANAGER_CLIENT (tmc), NULL);
	
	priv = tmc->priv;

	note = GNOME_MrProject_TaskManager_getNote (priv->manager,
						    task_id,
						    ev);
	
	if (BONOBO_EX (ev)) {
		g_warning ("Exception when getting note");
		return NULL;
	}

	return note;
}

void
task_mc_set_note (TaskManagerClient *tmc,
		  GM_Id              task_id,
		  gchar             *note,
		  CORBA_Environment *ev)
{
	TaskManagerClientPriv *priv;
	
	g_return_if_fail (tmc != NULL);
	g_return_if_fail (IS_TASK_MANAGER_CLIENT (tmc));
	
	priv = tmc->priv;

	GNOME_MrProject_TaskManager_setNote (priv->manager, task_id, note, ev);
	
	if (BONOBO_EX (ev)) {
		g_warning ("Exception when setting note");
	}
}

GM_TaskManager
task_mc_get_manager (TaskManagerClient *tmc)
{
	return tmc->priv->manager;
}
		     
