/*
 * Copyright (C) 2001  Richard Hult
 * Copyright (C) 2001  CodeFactory AB
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult <rhult@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <math.h>
#include <gtk/gtk.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-canvas.h>
#include <libgnomeui/gnome-canvas-rect-ellipse.h>
#include <libgnomeui/gnome-canvas-text.h>
#include <libgnomeui/gnome-canvas-line.h>
#include <libgnomeui/gnome-canvas-text.h>
#include <libgnomeui/gnome-canvas-util.h>
#include <gal/widgets/e-unicode.h>

#include "util/eel-gdk-font-extensions.h"
#include "util/type-utils.h"
#include "task-box.h"
#include "network-canvas.h"
#include "network-item.h"

#define DEBUG 0
#include "util/debug.h"

static void task_box_init		(TaskBox	 *task_box);
static void task_box_class_init	        (TaskBoxClass    *klass);
static void tb_set_task			(NetworkItem     *item);
static void tb_layout                   (NetworkItem     *item);
static void tb_update_task              (NetworkItem     *item);

static void tb_setup_task_items         (NetworkItem *item);
static void tb_setup_summary_items      (NetworkItem *item);

GNOME_CLASS_BOILERPLATE (TaskBox,
			 task_box,
			 NetworkItem,
			 network_item);

typedef struct {
	gdouble  width;     /* X position of this column. */
	gdouble  current_y; /* Where to put the next item in this column. */
} Column;

struct _TaskBoxPriv {
	GList            *children;       /* Contained boxes. */

	gdouble           width, height;
	
	GnomeCanvasItem  *frame_item;
	guint             reflow_idle_id;

	/* Task info canvas items. */
	GnomeCanvasItem  *name_item;
	GnomeCanvasItem  *id_item;
	GnomeCanvasItem  *duration_item;
	GnomeCanvasItem  *complete_item;
	GnomeCanvasItem  *line_item;
};

static GdkBitmap *stipple;
static gchar      stipple_pattern[] = { 0x02, 0x01 };

static void
tb_destroy (GtkObject *object)
{
	TaskBox *box;
	
	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_TASK_BOX (object));

	box = TASK_BOX (object);

	if (box->priv->reflow_idle_id) {
		g_source_remove (box->priv->reflow_idle_id);
		box->priv->reflow_idle_id = 0;
	}
	
	g_list_free (box->priv->children);
	g_free (box->priv);

	GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static void
tb_realize (GnomeCanvasItem *item)
{
	TaskBoxPriv *priv;

	priv = TASK_BOX (item)->priv;

	GNOME_CALL_PARENT_HANDLER (GNOME_CANVAS_ITEM_CLASS, realize, (item));
}

static void
task_box_class_init (TaskBoxClass *klass)
{
	GtkObjectClass       *object_class;
	NetworkItemClass     *network_item_class;
	GnomeCanvasItemClass *gnome_canvas_item_class;
	
	object_class = (GtkObjectClass*) klass;
	network_item_class = (NetworkItemClass*) klass;
	gnome_canvas_item_class = (GnomeCanvasItemClass*) klass;

	object_class->destroy = tb_destroy;

	gnome_canvas_item_class->realize = tb_realize;

	network_item_class->set_task = tb_set_task;
	network_item_class->update_task = tb_update_task;

	network_item_class->layout = tb_layout;
	
	stipple = gdk_bitmap_create_from_data (NULL, stipple_pattern, 2, 2);
}

static void
task_box_init (TaskBox *box)
{
	TaskBoxPriv *priv;

	priv = g_new0 (TaskBoxPriv, 1);
	box->priv = priv;

	priv->width = 0;
	priv->height = 0;
}

void
task_box_add_child (TaskBox     *parent_box,
		    NetworkItem *child,
		    gint         index)
{
	TaskBoxPriv      *priv;
	NetworkItem      *parent_item;
	GnomeCanvasGroup *group;

	g_return_if_fail (parent_box != NULL);
	g_return_if_fail (IS_TASK_BOX (parent_box));
	g_return_if_fail (child != NULL);
	g_return_if_fail (IS_NETWORK_ITEM (child));

	parent_item = NETWORK_ITEM (parent_box);
	priv = parent_box->priv;
	
	if (child->parent_item != NULL) {
		g_warning ("Item already has a parent.");
		return;
	}

	/* Make sure we look like a task box (but don't bother with the root box). */
	if (parent_item->task) {
		tb_setup_summary_items (parent_item);
	}
	
	child->parent_item = parent_item;

	priv->children = g_list_insert (priv->children, child, index);

	group = GNOME_CANVAS_GROUP (parent_box);

	gnome_canvas_item_reparent (GNOME_CANVAS_ITEM (child), group);
	gnome_canvas_item_raise_to_top (GNOME_CANVAS_ITEM (child));

	/* Put some padding around the child box. */
	network_item_move (child, PAD, PAD);
}

/* Remove child, returns the old position. */
gint
task_box_remove_child (TaskBox     *parent_box,
		       NetworkItem *child)
{
	TaskBoxPriv *priv;
	NetworkItem *parent_item;
	gint         i;
	
	g_return_val_if_fail (parent_box != NULL, 0);
	g_return_val_if_fail (IS_TASK_BOX (parent_box), 0);
	g_return_val_if_fail (child != NULL, 0);
	g_return_val_if_fail (IS_NETWORK_ITEM (child), 0);

	priv = parent_box->priv;
	parent_item = NETWORK_ITEM (parent_box);
	
	if (child->parent_item != parent_item) {
		g_warning ("Task box is not item's parent.");
		return 0;
	}

	child->parent_item = NULL;

	i = g_list_index (priv->children, child);
	
	priv->children = g_list_remove (priv->children, child);

	if (!priv->children && parent_item->task) {
		/* No longer a task, make it a task box (but not if it's the root). */
		tb_setup_task_items (parent_item);
	}

	return i;
}

void
task_box_reparent (TaskBox *parent_box, TaskBox *box)
{
	NetworkItem      *item, *parent_item;
	
	g_return_if_fail (parent_box != NULL);
	g_return_if_fail (IS_TASK_BOX (parent_box));
	g_return_if_fail (box != NULL);
	g_return_if_fail (IS_TASK_BOX (box));

	item = NETWORK_ITEM (box);
	parent_item = NETWORK_ITEM (parent_box);
	
	if (item->parent_item == NULL) {
		g_warning ("Item don't have a parent.");
		return;
	}

	task_box_remove_child (TASK_BOX (item->parent_item), item);
	task_box_add_child (parent_box, item, -1);
}

static void
tb_setup_summary_items (NetworkItem *item)
{
	TaskBox     *box;
	TaskBoxPriv *priv;
	GdkFont     *font;
	
	box = TASK_BOX (item);
	priv = box->priv;

	font = GTK_WIDGET (GNOME_CANVAS_ITEM (item)->canvas)->style->font;

	if (priv->frame_item) {
		gnome_canvas_item_set (priv->frame_item,
				       "outline_stipple", stipple,
				       "fill_color", NULL,
				       NULL);
	}

	if (priv->line_item) {
		gtk_object_destroy (GTK_OBJECT (priv->line_item));
		priv->line_item = NULL;
	}

	if (priv->id_item) {
		gtk_object_destroy (GTK_OBJECT (priv->id_item));
		priv->id_item = NULL;
	}

	if (priv->complete_item) {
		gtk_object_destroy (GTK_OBJECT (priv->complete_item));
		priv->complete_item = NULL;
	}
	
	gnome_canvas_item_set (priv->name_item,
			       "x", SMALL_PAD,
			       "y", SMALL_PAD,
			       "anchor", GTK_ANCHOR_NW,
			       NULL);

	tb_update_task (item);
}

static void
tb_setup_task_items (NetworkItem *item)
{
	TaskBox           *box;
	TaskBoxPriv       *priv;
	GnomeCanvasPoints *points;
	GdkFont           *font;
	
	box = TASK_BOX (item);
	priv = box->priv;

	font = GTK_WIDGET (GNOME_CANVAS_ITEM (item)->canvas)->style->font;
	
	if (priv->frame_item) {
		gnome_canvas_item_set (priv->frame_item,
				       "x1", 0.0,
				       "y1", 0.0,
				       "x2", 200.0,
				       "y2", 100.0,
				       "outline_color", "black",
				       "outline_stipple", NULL,
				       "fill_color", "grey81",
				       NULL);
	}
	
	gnome_canvas_item_set (priv->name_item,
			       "x", 5.0,
			       "y", 5.0,
			       "anchor", GTK_ANCHOR_NW,
			       "fill_color", "black",
			       NULL);
	
	if (!priv->line_item) {
		points = gnome_canvas_points_new (2);
		points->coords[0] = 5;
		points->coords[1] = 20;
		points->coords[2] = 190;
		points->coords[3] = 20;

		priv->line_item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (box),
							 GNOME_TYPE_CANVAS_LINE,
							 "points", points,
							 "fill_color", "black",
							 NULL);
		gnome_canvas_points_unref (points);
	}
	
	if (!priv->complete_item) {
		priv->complete_item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (box),
							     GNOME_TYPE_CANVAS_TEXT,
							     "font_gdk", font,
							     "x", 5.0,
							     "y", 100.0 - SMALL_PAD + 2,
							     "anchor", GTK_ANCHOR_SW,
							     "fill_color", "black",
							     NULL);
	}

	if (!priv->id_item) {
		priv->id_item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (box),
						       GNOME_TYPE_CANVAS_TEXT,
						       "font_gdk", font,
						       "x", 5.0,
						       "y", 25.0,
						       "anchor", GTK_ANCHOR_NW,
						       "fill_color", "black",
						       NULL);
	}

	tb_update_task (item);
}

static void
tb_set_task (NetworkItem *item)
{
	GM_Task     *task;
	TaskBox     *box;
	TaskBoxPriv *priv;
	GdkFont     *font;

	g_return_if_fail (item != NULL);
	g_return_if_fail (IS_TASK_BOX (item));
	
	font = GTK_WIDGET (GNOME_CANVAS_ITEM (item)->canvas)->style->font;
	
	if (!item->task) {
		g_warning ("Task box has no task");
		return;
	}

	box = TASK_BOX (item);
	priv = box->priv;

	task = item->task;

	if (!priv->frame_item) {
		priv->frame_item = gnome_canvas_item_new (
			GNOME_CANVAS_GROUP (item),
			GNOME_TYPE_CANVAS_RECT,
			"x1", 0.0,
			"y1", 0.0,
			"x2", 100.0+priv->width,
			"y2", 100.0+priv->height,
			"outline_color", "black",
			NULL);
	}

	if (!priv->name_item) {
		priv->name_item = 
			gnome_canvas_item_new (
				GNOME_CANVAS_GROUP (item),
				GNOME_TYPE_CANVAS_TEXT,
				"x", SMALL_PAD,
				"y", SMALL_PAD,
				"anchor", GTK_ANCHOR_NW,
				"fill_color", "black",
				"text", "",
				"font_gdk", font,
				NULL);
	}

	if (!priv->duration_item) {
		priv->duration_item =
			gnome_canvas_item_new (GNOME_CANVAS_GROUP (item),
					       GNOME_TYPE_CANVAS_TEXT,
					       "anchor", GTK_ANCHOR_SE,
					       "fill_color", "black",
					       "text", "",
					       "font_gdk", font,
					       NULL);
	}

	if (priv->children) {
		tb_setup_summary_items (item);
	} else {
		tb_setup_task_items (item);
	}
}

static void
tb_layout (NetworkItem *item)
{
	TaskBox     *box;
	TaskBoxPriv *priv;
	GList       *l;
	NetworkItem *child_item;
	gdouble      x, y;
	gdouble      x1, y1, x2, y2;
	gdouble      min_x, max_x, min_y, max_y;
	gdouble      width, height;
	GArray      *columns;

	g_return_if_fail (item != NULL);
	g_return_if_fail (IS_TASK_BOX (item));

	box = TASK_BOX (item);
	priv = box->priv;
	
	width = 0;
	height = 0;
	y = PAD;

	min_x = G_MAXDOUBLE;
	max_x = 0;
	min_y = G_MAXDOUBLE;
	max_y = 0;

	columns = g_array_new (FALSE, TRUE, sizeof (Column*));

	if (priv->children) {
		/* Summary task. */

		/* First calculate the column widths. */
		for (l = priv->children; l; l = l->next) {
			Column  *column;
			
			child_item = l->data;
			
			gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (child_item),
						      &x1, &y1,
						      &x2, &y2);
			
			if (child_item->col >= columns->len) {
				/* Expand the array to make room. */
				g_array_set_size (columns, child_item->col + 1);
			}
			
			column = g_array_index (columns, Column*, child_item->col);
			if (!column) {
				column = g_new (Column, 1);
				column->width = 0.0;
				column->current_y = PAD;
				
				g_array_insert_val (columns, child_item->col, column);
			}
			
			if (column->width < (x2 - x1)) {
				column->width = x2 - x1;
			}
		}

		/* Position the boxes. */
		for (l = priv->children; l; l = l->next) {
			Column *column;
			gdouble tmp;
			
			child_item = l->data;

			gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (child_item),
						      &x1, &y1,
						      &x2, &y2);
			
			/* Move the box to the right of the previous column. */
			column = g_array_index (columns, Column*, child_item->col);
			if (!column) {
				g_warning ("Tried to access column (%d) that does not exist.", child_item->col);
				continue;
			}
			
			y = column->current_y;
			if (child_item->col > 0) {
				Column *prev_column;
				guint   i;
				
				x = PAD;
				for (i = 0; i < child_item->col; i++) {
					prev_column = g_array_index (columns, Column*, i);
					if (!prev_column) {
						g_warning ("Tried to access column (%d) that does not exist.", i);
						continue;
					}
					
					x += prev_column->width + PAD;
				}
			}
			else {
				x = PAD;
			}
			
			y = column->current_y;
			column->current_y += y2 - y1 + PAD;
			
			network_item_move (child_item, x, y);

			tmp = x1;
			x1 = x;
			x2 = x + (x2 - tmp);

			tmp = y1;
			y1 = y;
			y2 = y + (y2 - tmp);
			
			if (x1 < min_x) {
				min_x = x1;
			}
			if (x2 > max_x) {
				max_x = x2;
			}
			if (y1 < min_y) {
				min_y = y1;
			}
			if (y2 > max_y) {
				max_y = y2;
			}
		}

		height = max_y - min_y;
		width = max_x - min_x;

		/* Set the right size for the task frame. */
		priv->width = width + 2 * PAD;
		priv->height = height + 2 * PAD;
	}
	else {
		/* Non-summary task. */

		/* Task box has fixed size (for now at least). */
		width = priv->width = 200;
		height = priv->height = 100;
	}
		
	if (priv->frame_item) {
		gtk_object_get (GTK_OBJECT (priv->frame_item),
				"x1", &x1,
				"y1", &y1,
				NULL);
	
		gnome_canvas_item_set (priv->frame_item,
				       "x2", x1 + priv->width,
				       "y2", y1 + priv->height,
				       NULL);
	}

	/* And reflow the duration text. */
	if (priv->duration_item) {
		gnome_canvas_item_set (priv->duration_item,
				       "y", priv->height - SMALL_PAD + 2,
				       "x", priv->width - SMALL_PAD,
				       NULL);
	}

	g_array_free (columns, TRUE);
}

static void
tb_update_task (NetworkItem *item) 
{
	TaskBox     *box;
	TaskBoxPriv *priv;
	GM_Task     *task;
	gchar       *name, *str, *tmp;
	gint         duration;
	GdkFont     *font;
	GtkWidget   *canvas_widget;
	
	g_return_if_fail (item != NULL);
	g_return_if_fail (IS_TASK_BOX (item));

	canvas_widget = GTK_WIDGET (GNOME_CANVAS_ITEM (item)->canvas);
	
	font = canvas_widget->style->font;

	box = TASK_BOX (item);
	priv = box->priv;
	task = item->task;
	
	name = g_strdup_printf (_("Name: %s"), task->name);
	tmp = e_utf8_to_gtk_string (canvas_widget, name);
	str = eel_string_ellipsize_end (tmp,
					font, 
					180);
	gnome_canvas_item_set (priv->name_item,
			       "text", str,
			       NULL);
	g_free (name);
	g_free (str);
	g_free (tmp);

	if (priv->complete_item) {
		str = g_strdup_printf (_("Complete: %d%%"), task->percentComplete);
		gnome_canvas_item_set (priv->complete_item,
				       "text", str,
				       NULL);
		g_free (str);
	}

	if (priv->id_item) {
		str = g_strdup_printf ("Id: %d", task->taskId);
		gnome_canvas_item_set (priv->id_item,
				       "text", str,
				       NULL);
		g_free (str);
	}

	duration = ceil ((task->end - task->start) / (60*60*24));
	str = g_strdup_printf (_("Duration: %d days"), duration);
	gnome_canvas_item_set (priv->duration_item,
			       "text", str,
			       NULL);
	g_free (str);
}



