# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Authors: Olivier Tilloy <olivier@fluendo.com>
#          Florian Boucault <florian@fluendo.com>

"""
Various widgets used to display the results of a search.
"""

from elisa.core.utils.i18n import install_translation
from elisa.core.utils import notifying_list

from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.list_vertical import ListVertical
from elisa.plugins.pigment.widgets.list_horizontal import ListHorizontal
from elisa.plugins.pigment.widgets.box import VBox

from elisa.plugins.poblesec.widgets.button import PanelButton
from elisa.plugins.poblesec.widgets.background import WidgetWithBackground


_, _n = install_translation('poblesec', True)


class SearcherResultModel(object):
    """
    Represent one search result category, which contains a list of actual
    search results and other information useful to display such results.
    """

    def __init__(self, searcher, result_type, result, label=None):
        """
        Constructor.

        @param searcher:      the searcher entry point
        @type searcher:       C{str}
        @param result_type:   the type of results of this category
        @type result_type:    C{str}
        @param result:       the list of results for this category
        @type result:        C{list} of L{elisa.plugins.base.models}.*
        @param label:       the label to use in the header of this category
        @type label:        C{str}
        """
        self.searcher = searcher
        self.result_type = result_type
        self.result = result
        self.label = label if not label is None else _("Unknown")

class SearchResultWidget(PanelButton):
    """
    A widget used to display the contents of a L{SearchResultModel}.
    """

    def __init__(self):
        super(SearchResultWidget, self).__init__()
        self.panel.set_name("darker")

    def _create_widgets(self):
        super(SearchResultWidget, self)._create_widgets()
        self.label = Text()
        self.add(self.label, forward_signals=False)
        self.label.visible = True

        self.total = Text()
        self.add(self.total, forward_signals=False)
        self.total.visible = True

    def clean(self):
        self.label = None
        self.total = None
        return super(SearchResultWidget, self).clean()

class SearcherModel(object):
    """
    Represents a searcher's results.
    
    DOCME
    """

    def __init__(self, searcher_entry, mediatype):
        """
        Constructor.

        @param searcher_entry: a searcher entry
        @type searcher_entry:  L{elisa.plugins.poblesec.search_controller.SearcherEntry}
        @param mediatype: DOCME
        @type mediatype: str
        """
        self.searcher_entry = searcher_entry
        self.mediatype = mediatype
        self.results = notifying_list.List()

    def set_results(self, result_model):
        """
        Take the results from the Searcher, then for each subtype that it
        supports build one of these ugly intermediate SearcherResultModel
        instances.
        This contains the searcher name, the subtype, the label and the
        actual results.
        Then put each of them into the SearcherModel's results.

        TODO: Most of this stuff should be refactored, it looks that there are
        too many intermediate classes.
        """
        self.results[:] = []

        for subtype in result_model.supported_subtypes:
            results = result_model.results_for_subtype(subtype)
            label = result_model.label_for_subtype(subtype)
            item = SearcherResultModel(self.searcher_entry.searcher, subtype, results, label=label)
            self.results.append(item)

class SearcherWidget(Widget):
    """
    A widget used to display the contents of a L{SearcherModel}.

    @cvar item_widget_class: the class of the widget used to display one search
                             result node (category)
    @type item_widget_class: L{elisa.plugins.pigment.widgets.widget.Widget}
    """

    item_widget_class = SearchResultWidget

    class ListWidget(ListVertical):
        widget_signals = {'activated': 'item-activated'}
        render_empty_items = True

    def __init__(self):
        super(SearcherWidget, self).__init__()
        self._create_widgets()
        self.update_style_properties(self.style.get_items())

    def _create_widgets(self):
        self.title = Text()
        self.add(self.title)
        self.title.visible = True

        # vertical list of results
        self.list = SearcherWidget.ListWidget(widget_class=self.item_widget_class,
                                              visible_range_size=3)
        self.add(self.list)
        self.set_focus_proxy(self.list)
        self.list.set_renderer(self._item_renderer)
        self.list.visible = True

    def clean(self):
        self.title = None
        self.list = None
        return super(SearcherWidget, self).clean()

    def _item_renderer(self, item, widget):
        label = _("%(type)s Results") % {'type': item.label}
        widget.label.label = label
        total = _n("<b>%(count)d</b> Found", "<b>%(count)d</b> Found", len(item.result)) % \
            {'count': len(item.result)}
        widget.total.markup = total



class SearchResultsWidget(VBox):
    """
    A widget that displays the search results for all the searchers of a
    search.
    """
    
    class ListWidget(ListHorizontal):
        widget_signals = {}
        render_empty_items = True

        def _layout_widget(self, widget, position):
            widget.position = (self.compute_x(position), 0.0, 0.0)

    def __init__(self, mediatype, frontend=None):
        """
        Constructor.

        @param mediatype: the type of media searched
        @type mediatype:  C{str}
        """
        super(SearchResultsWidget, self).__init__()
        self.mediatype = mediatype
        self.searchers_results = {}
        # FIXME: passing the frontend around to a widget is a hack; an action
        # could be passed instead executed when a result is clicked
        self._frontend = frontend

    def create_widgets(self):
        self.navigable = True

        self.title = WidgetWithBackground(foreground=Text())
        self.title.visible = True
        self.pack_start(self.title)

        self.list = SearchResultsWidget.ListWidget(SearcherWidget,
                                                   visible_range_size=2)
        self.list.visible = True
        self.list.set_renderer(self._item_renderer)
        self.pack_start(self.list, expand=True)
        self.set_focus_proxy(self.list)        

    def set_searchers(self, searcher_entries):
        """
        Set the list of searchers used for the search and for which results
        should be displayed.

        @param searchers: a list of searcher entries
        @type searchers:  C{list} of
                          L{elisa.plugins.poblesec.search_controller.SearcherEntry}
        """
        for searcher_entry in searcher_entries:
            searcher_model = SearcherModel(searcher_entry, self.mediatype)
            self.searchers_results[searcher_entry] = searcher_model

        self.list.set_model(notifying_list.List(self.searchers_results.values()))

    def set_results(self, searcher_entry, result_model):
        """
        Update the search results for a given searcher entry.

        @param searcher_entry: a searcher entry
        @type searcher_entry:  L{elisa.plugins.poblesec.search_controller.SearcherEntry}
        @param result_model:   the result of a new search
        @type result_model:    one of L{elisa.plugins.search.models}.*
        """
        searcher_model = self.searchers_results[searcher_entry]
        searcher_model.set_results(result_model)

    def _item_renderer(self, item, widget):
        widget.title.label = item.searcher_entry.title
        widget.list.set_model(item.results)

        widget.list.connect('item-activated', self._result_activated_handler)
        # FIXME: disconnect on cleanup

    def _result_activated_handler(self, searcher_widget, result_model):
        browser = self._frontend.retrieve_controllers('/poblesec/browser')[0]
        path = '/poblesec/search/results/%s/%s'
        path = path % (result_model.searcher, result_model.result_type)
        args = {result_model.result_type: result_model.result}
        label = result_model.label
        label = _("%(result_subtype)s Results") % {'result_subtype': label}
        dfr = browser.history.append_controller(path, label, **args)

    @classmethod
    def _demo_widget(cls):
        widget = cls('music')
        widget.title.foreground.label = "EXAMPLE SEARCH RESULTS"
        widget.size = (400.0, 250.0)
        return widget

if __name__ == '__main__':
    from elisa.plugins.poblesec.widgets.search_results import SearchResultsWidget
    SearchResultsWidget.demo()
    try:
        __IPYTHON__
    except NameError:
        import pgm
        pgm.main()
