#ifndef LIBBTUTIL_TYPES_BT_METAINFO_H
#define LIBBTUTIL_TYPES_BT_METAINFO_H

#include <apr.h>
#include <apr_pools.h>
#include <apr_network_io.h>
#include <stdint.h>
#include <time.h>

#include <libbtutil/const.h>

#define BT_METAINFO_MAX_FILES   1024

typedef struct bt_metainfo_finfo_s {
    uint64_t    length;
    char       name[BT_SHORT_STRING];
} bt_metainfo_finfo;

typedef struct bt_metainfo_s {
    /* Path to torrent */
    char               torrent[BT_SHORT_STRING];

    /* General info */
    uint8_t             hash[BT_INFOHASH_LEN];
    char               name[BT_SHORT_STRING];
    time_t              creation_date;

    /* Tracker info */
    char               tracker_address[BT_SHORT_STRING];
    apr_port_t          tracker_port;
    char               tracker_announce[BT_SHORT_STRING];

    /* Pieces info */
    unsigned int        piece_size;
    unsigned int        piece_count;
    uint64_t            total_size;
    uint8_t*            pieces;

    /* Files info */
    int                 file_count;
    bt_metainfo_finfo*  files;
} bt_metainfo;


bt_metainfo* bt_metainfo_parse(apr_pool_t *, const char *);
extern int bt_metainfo_pos2file(
    bt_metainfo* info, bt_off_t pos, bt_off_t* offset
);

extern apr_status_t bt_metainfo_save(
    bt_metainfo* metainfo, apr_pool_t* p, const char* path
);

extern apr_status_t bt_metainfo_init_files(
    bt_metainfo* info, apr_pool_t* p, const char* path
);

static inline uint64_t bt_metainfo_last_piece_size(bt_metainfo* i) {
    return i->total_size - ((i->piece_count-1) * i->piece_size);
}

static inline uint64_t bt_metainfo_piece_size(bt_metainfo* i, int p) {
    return (
        (i->piece_count - 1 > p) ?
            i->piece_size :
            bt_metainfo_last_piece_size(i)
    );
}

static inline void bt_metainfo_set_piece_hash (
    bt_metainfo* metainfo, int piece, uint8_t* hash
) {
    memcpy(
        metainfo->pieces + (piece * BT_SHA1_OUTPUT_SIZE),
        hash,
        BT_SHA1_OUTPUT_SIZE
    );
}

#endif /* LIBBTUTIL_TYPES_BT_METAINFO_H */
