#include "udm_config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <dirent.h>
#include <errno.h>


#if (WIN32|WINNT)
#include <winsock.h>
#include <time.h>
#include <direct.h>
#endif

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#ifdef HAVE_SELECT_H
#include <select.h>
#endif
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

#ifdef USE_HTTPS
#include <openssl/crypto.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#endif

#include "udm_common.h"
#include "udm_proto.h"
#include "udm_utils.h"
#include "udm_mutex.h"
#include "udm_socket.h"
#include "udm_http.h"
#include "udm_ftp.h"
#include "udm_xmalloc.h"
#include "udm_host.h"
#include "udm_mimetype.h"


/*************** Mirroring stuff ************************/

int   UdmMirrorGET(UDM_AGENT * Indexer,char * MirrorRoot, char * MirrorHeadersRoot,char *schema, char *hostname, 
		char *path,char *filename, char *buf,int days, char *errstr) {

	int size = 0, l;
	int fbody, fheader;
	char str[UDMSTRSIZ]="";
	char *content_type = NULL;
	struct stat sb;
	time_t nowtime;
	int have_headers=0;

	nowtime = time(NULL);

	if (days <= 0)return(0);
	
	/* MirrorRoot is not specified, nothing to do */
	if(!MirrorRoot)return(0);

	strcpy(str, MirrorRoot);strcat(str,"/");
	strcat(str, schema);strcat(str,"/");
	strcat(str, hostname);
	strcat(str, path);
	strcat(str, filename && strlen(filename) ? filename : "index.html");

	if ((fbody = open(str, O_RDONLY|UDM_BINARY)) == -1){
		sprintf(errstr,"Mirror file %s not found", str);
		return UDM_MIRROR_NOT_FOUND;
	}

	/* Check on file mtime > days ? return */
	if (fstat(fbody, &sb)) {
		/* error handler */
	}
	if (nowtime > sb.st_mtime + days ) {
		close(fbody);
		sprintf(errstr,"%s is older then %d secs, retrieving", str, days);
		return UDM_MIRROR_EXPIRED;
	}

	if(MirrorHeadersRoot){
		strcpy(str, MirrorHeadersRoot);strcat(str,"/");
		strcat(str, schema);strcat(str,"/");
		strcat(str, hostname);
		strcat(str, path);
		strcat(str, filename && strlen(filename) ? filename : "index.html");
		strcat(str,".header");

		if ((fheader = open(str, O_RDONLY|UDM_BINARY))>=0) {
			size = read(fheader, buf, Indexer->Conf->max_doc_size);
			close(fheader);
			strcpy(buf + size, "\r\n\r\n");
			have_headers=1;
		}
	}
	if(!have_headers){
		/* header file not found   */
		/* get body as file method */
		sprintf(buf,"HTTP/1.0 200 OK\r\n");
		if((content_type=UdmContentType(Indexer->Conf,filename && strlen(filename) ? filename : "index.html")))
			sprintf(UDM_STREND(buf),"Content-Type: %s\r\n",content_type);
		sprintf(UDM_STREND(buf),"\r\n");
	}

	l = strlen(buf);
	size = read(fbody, buf + l, Indexer->Conf->max_doc_size - l) + l;
	close(fbody);
	return size;
}

int   UdmMirrorPUT(UDM_AGENT * Indexer,
		char * MirrorRoot, char * MirrorHeadersRoot,
		char *schema, char *hostname, char *path, char *filename, 
		char *header, char *content, int size, char *errstr) {

	int fd;
	char str[UDMSTRSIZ]="";

	/* Put Content if MirrorRoot is specified */
	if(MirrorRoot){
		strcpy(str, MirrorRoot);strcat(str,UDMSLASHSTR);
		strcat(str, schema);strcat(str,UDMSLASHSTR);
		strcat(str, hostname);
		strcat(str, path);

#if (WIN32|WINNT)
		/* Replace slashes to Win style */
		{char *p; p=str;while(*p){if(*p=='/')*p='\\'; p++;}}
#endif
		if(UdmBuild(str, 0755) != 0){
			sprintf(errstr, "Can't create dir %s", str);
			return UDM_MIRROR_CANT_BUILD;
		}

		strcat(str,UDMSLASHSTR);
		strcat(str, filename && strlen(filename) ? filename : "index.html");
		if ((fd = open(str, O_CREAT|O_WRONLY|UDM_BINARY,UDM_IWRITE)) == -1){
			sprintf(errstr,"Can't open mirror file %s\n", str);
			return UDM_MIRROR_CANT_OPEN;
		}
		size = write(fd, content, (size_t)size);
		close(fd);
	}

	/* Put Headers if MirrorHeadersRoot is specified */
	if(MirrorHeadersRoot){
		strcpy(str, MirrorHeadersRoot);strcat(str,UDMSLASHSTR);
		strcat(str, schema);strcat(str,UDMSLASHSTR);
		strcat(str, hostname);
		strcat(str, path);

#if (WIN32|WINNT)
		/* Replace slashes to Win style */
		{char *p; p=str;while(*p){if(*p=='/')*p='\\'; p++;}}
#endif
		if(UdmBuild(str, 0755) != 0){
			sprintf(errstr, "Can't create mirror dir %s", str);
			return UDM_MIRROR_CANT_BUILD;
		}

		strcat(str,UDMSLASHSTR);
		strcat(str, filename && strlen(filename) ? filename : "index.html");
		strcat(str,".header");

		if ((fd = open(str, O_CREAT|O_WRONLY|UDM_BINARY,UDM_IWRITE)) == -1){
			sprintf(errstr,"Can't open mirror file %s\n", str);
			return UDM_MIRROR_CANT_OPEN;
		}
		size = write(fd, header, strlen(header));
		close(fd);
	}
	return(0);
}

/************** connect with timeout **********/

static int connect_tm(int s, const struct sockaddr *name, unsigned int namelen, unsigned int to){
#if (WIN32|WINNT)
	return connect(s, name, namelen);
#else
	int flags, res, s_err;
	int s_err_size = sizeof(uint);
	fd_set sfds;
	struct timeval tv;

	if (!to)return(connect(s,name, namelen));

	flags = fcntl(s, F_GETFL, 0);	 	/* Set socket to non-blocked */
#ifdef O_NONBLOCK
	fcntl(s, F_SETFL, flags | O_NONBLOCK);  /* and save the flags        */
#endif

	res = connect(s, name, namelen);
	s_err = errno;			/* Save errno */
	fcntl(s, F_SETFL, flags);
	if ((res != 0) && (s_err != EINPROGRESS)){
		errno = s_err;		/* Restore errno              */
		return(-1);		/* Unknown error, not timeout */
	}
	if(!res)return(0);		/* Quickly connected */

	FD_ZERO(&sfds);
	FD_SET(s,&sfds);

	tv.tv_sec = (long) to;
	tv.tv_usec = 0;
	
	while(1){
		res = select(s+1, NULL, &sfds, NULL, &tv);
		if(res==0)return(-1); /* Timeout */
		if(res<0){
			if (errno == EINTR) /* Signal */
				continue;
			else
				return(-1); /* Error */
		}
		break;
	}

	s_err=0;
	if (getsockopt(s, SOL_SOCKET, SO_ERROR, (char*) &s_err, &s_err_size) != 0)
		return(-1);		/* Can't get sock options */

	if (s_err){
		errno = s_err;
		return(-1);
	}
	return(0);			/* We have a connection! */
#endif
}


/************** Open TCP Connect **************/

static int open_host(UDM_AGENT *Indexer, char *hostname,int port, int timeout)
{
	int net;
	size_t len;

	bzero((char*)&Indexer->connp->sin, sizeof(struct sockaddr_in));
	if (!port)
		return(UDM_NET_ERROR);
	Indexer->connp->port = port;
	Indexer->connp->sin.sin_family=AF_INET;

	len = strlen(hostname);
	Indexer->connp->hostname = UdmXrealloc(Indexer->connp->hostname, len+1);
	udm_snprintf(Indexer->connp->hostname, len+1, "%s", hostname);

	if (UdmHostLookup(Indexer->Conf,Indexer->connp)==-1)
		return Indexer->connp->err;

	net=socket(AF_INET, SOCK_STREAM, 0);

	if(connect_tm(net, (struct sockaddr *)&Indexer->connp->sin, sizeof (struct sockaddr_in),(unsigned int)timeout)){
		closesocket(net);
		return(UDM_NET_CANT_CONNECT);
	}
	return(net);
}


/**************** HTTP codes and messages ************/

const char *UdmHTTPErrMsg(int code){
	switch(code){
	case 0:   return("Not indexed yet");
	case 200: return("OK");
	case 206: return("Partial OK");
	case 301: return("Moved Permanently");
	case 302: return("Moved Temporarily");
	case 303: return("See Other");
	case 304: return("Not Modified");
	case 300: return("Multiple Choices");
	case 305: return("Use Proxy (proxy redirect)");
	case 400: return("Bad Request");
	case 401: return("Unauthorized");
	case 402: return("Payment Required");
	case 403: return("Forbidden");
	case 404: return("Not found");
	case 405: return("Method Not Allowed");
	case 406: return("Not Acceptable");
	case 407: return("Proxy Authentication Required");
	case 408: return("Request Timeout");
	case 409: return("Conflict");
	case 410: return("Gone");
	case 411: return("Length Required");
	case 412: return("Precondition Failed");
	case 413: return("Request Entity Too Large");
	case 414: return("Request-URI Too Long");
	case 415: return("Unsupported Media Type");
	case 500: return("Internal Server Error");
	case 501: return("Not Implemented");
	case 502: return("Bad Gateway");
	case 505: return("Protocol Version Not Supported");
	case 503: return("Service Unavailable");
	case 504: return("Gateway Timeout");
	default:  return("Unknown status");
	}
}

int UdmHTTPResponseType(int status){

	switch(status){
	case 1:   return(1);
	case 200: return(UDM_HTTP_STATUS_OK);
	case 206: return(UDM_HTTP_STATUS_PARTIAL_OK);

	case 301:  /* Moved Permanently */
	case 302:  /* Moved Temporarily */
	case 303:  /* See Other */ return(UDM_HTTP_STATUS_REDIRECT);

	case 304:  /* Not Modified*/ return(UDM_HTTP_STATUS_NOT_MODIFIED);

	case 300:  /* Multiple Choices              */
	case 305:  /* Use Proxy (proxy redirect)    */
	case 400:  /* Bad Request ??? We tried ...  */
	case 401:  /* Unauthorized                   */
	case 402:  /* Payment Required              */
	case 403:  /* Forbidden                     */
	case 404:  /* Not found                     */
	case 405:  /* Method Not Allowed            */
	case 406:  /* Not Acceptable                */
	case 407:  /* Proxy Authentication Required */
	case 408:  /* Request Timeout               */
	case 409:  /* Conflict                      */
	case 410:  /* Gone                          */
	case 411:  /* Length Required               */
	case 412:  /* Precondition Failed           */
	case 413:  /* Request Entity Too Large      */
	case 414:  /* Request-URI Too Long          */
	case 415:  /* Unsupported Media Type        */
	case 500:  /* Internal Server Error         */
	case 501:  /* Not Implemented               */
	case 502:  /* Bad Gateway                   */
	case 505:  /* Protocol Version Not Supported */
		return(UDM_HTTP_STATUS_DELETE);

	case 503: /* Service Unavailable */
	case 504: /* Gateway Timeout */
		return(UDM_HTTP_STATUS_RETRY);

	default: return(UDM_HTTP_STATUS_UNKNOWN);
	}
}


/*******************************************************/
#ifdef USE_HTTP

int UdmHTTPGet(UDM_AGENT *Indexer,char * header,char *host,int port)
{
    int fd;
    fd_set sfds;
    time_t start_time;
    int status, nread=0;
    size_t buf_size=NET_BUF_SIZE;

    struct timeval tv;

    /* Connect to HTTP or PROXY server */
    if( (fd=open_host(Indexer,host,port,Indexer->read_timeout)) < 0 )
       return(fd);

    /* Send HTTP request */
    if( send(fd,header,strlen(header),0) < 0 )
       return -1;

    /* Retrieve response */
    tv.tv_sec = (long) Indexer->read_timeout;
    tv.tv_usec = 0;

    start_time=time(NULL);
    while( 1 ){
       FD_ZERO( &sfds );
       FD_SET( fd, &sfds );

       status=select(FD_SETSIZE, &sfds, NULL, NULL, &tv);
       if( status == -1 ){
               nread=UDM_NET_ERROR;
               break;
       }else
       if( status == 0 ){
               nread=UDM_NET_TIMEOUT;
               break;
       }else{
           if(FD_ISSET(fd,&sfds))
           {
		if(nread+buf_size-Indexer->Conf->max_doc_size >=0)
			buf_size = Indexer->Conf->max_doc_size - nread;
		else
			buf_size = NET_BUF_SIZE;
		
               status = recv(fd, Indexer->buf+nread, buf_size, 0);
               if( status < 0 ){
                   nread = status;
                   break;
               }else 
               if( status == 0 ){
                   break;
               }else{
                   nread += status;
                   if(Indexer->doc_timeout<(time(NULL)-start_time)){
                   	nread=UDM_NET_TIMEOUT;
                   	break;
                   }
                   if( nread == Indexer->Conf->max_doc_size )
                       break;
               }
           }else
               break;
       }
    }
    closesocket(fd);
    return(nread);
}
#endif

#ifdef USE_HTTPS

#define sslcleanup closesocket(fd); SSL_free (ssl); SSL_CTX_free (ctx)

int UdmHTTPSGet(UDM_AGENT *Indexer,char * header,char *host,int port)
{
    int fd;
    int nread=0;
    SSL_CTX* ctx;
    SSL*     ssl;
    SSL_METHOD *meth;

    /* Connect to HTTPS server */
    if( (fd=open_host(Indexer,host,port,Indexer->read_timeout)) < 0 )
       return(fd);

    SSLeay_add_ssl_algorithms();
    meth = SSLv2_client_method();
    SSL_load_error_strings();
    ctx = SSL_CTX_new (meth);

    if ((ctx = SSL_CTX_new (meth))==NULL){
      sslcleanup;
      return UDM_NET_ERROR;
    }

    /* -------------------------------------------------- */
    /* Now we have TCP connection. Start SSL negotiation. */

    if ((ssl=SSL_new(ctx))==NULL){
      sslcleanup;
      closesocket(fd);
      return UDM_NET_ERROR;
    }

    SSL_set_fd (ssl, fd);

    if (SSL_connect(ssl)==-1){
      sslcleanup;
      return UDM_NET_ERROR;
    }

    /* Send HTTP request */
    if ((SSL_write(ssl, header,(int)strlen(header)))==-1){
      sslcleanup;
      return UDM_NET_ERROR;
    }

    /* Get result */
    nread = SSL_read(ssl, Indexer->buf, (int)(Indexer->Conf->max_doc_size - 1));

    if(nread == -1){
      sslcleanup;
      return UDM_NET_ERROR;
    }

    Indexer->buf[nread] = '\0';
    SSL_shutdown (ssl);  /* send SSL/TLS close_notify */

    /* Clean up. */
    sslcleanup;
    return(nread);
}

#endif

#ifdef USE_FTP
int UdmURLGetFTP(UDM_AGENT * Indexer,char *host,int port, 
	char *path, char *filename,
	char *user, char *passwd,
	int last_mod_time, int checkonly)
{
	int size=0, last_mod_tm, code;
	char *con_type, buf[256], *full_path=NULL;
	size_t len,buf_size;
	
	/* Check for changing host */
	if (!Indexer->connp->hostname || 
	    strcmp(Indexer->connp->hostname, host) || 
	    (Indexer->connp->connected == NET_NOTCONNECTED)){
		code=UdmFTPConnect(Indexer->Conf,Indexer->connp, host, port, user, passwd, Indexer->read_timeout);
		if (code == -1){
			if (Indexer->connp->err >0){
				sprintf(Indexer->buf, "HTTP/1.1 401 OK\r\n\r\n  ");
				size = strlen(Indexer->buf);
			}else{
				size = Indexer->connp->err;
			}
		}
	}
	
	if (Indexer->connp->connected == NET_CONNECTED){
		/* Make listing */
		if (!filename){
			code = UdmFTPCwd(Indexer->connp, path);
			if (code != -1){
				code = UdmFTPList(Indexer->connp, Indexer->connp->connp, path, NULL, Indexer->Conf->max_doc_size);
				if (code == -1){
					if (Indexer->connp->err >0){
						sprintf(Indexer->buf, "HTTP/1.1 403 OK\r\n\r\n");
						size = strlen(Indexer->buf);
					}else{
						size = Indexer->connp->err;
					}
				}else{
					udm_snprintf(Indexer->buf, Indexer->Conf->max_doc_size, "HTTP/1.1 200 OK\r\nContent-Type: text/html\r\n\r\n<html><body>%s</body></html>", 
						Indexer->connp->connp->buf);
					size = strlen(Indexer->buf);
				}
			}else{
				if (Indexer->connp->err >0){
					sprintf(Indexer->buf, "HTTP/1.1 403 OK\r\n\r\n");
					size = strlen(Indexer->buf);
				}else{
					size = Indexer->connp->err;
				}
			}
		}else{
			con_type = UdmContentType(Indexer->Conf,filename);
			len = strlen(path)+strlen(filename);
			full_path = UdmXmalloc(len+1);
			udm_snprintf(full_path, len+1, "%s%s", path, filename);
			last_mod_tm = UdmFTPMdtm(Indexer->connp,full_path);
			if(last_mod_tm == -1 && Indexer->connp->err){
				if (Indexer->connp->err >0){
					sprintf(Indexer->buf, "HTTP/1.1 404 OK\r\n\r\n");
					size = strlen(Indexer->buf);
				}else
					size = Indexer->connp->err;

			}else if (last_mod_tm == last_mod_time){
				sprintf(Indexer->buf, "HTTP/1.1 304 OK\r\n\r\n");
				size = strlen(Indexer->buf);
			}else{
				UdmTime_t2HttpStr(last_mod_tm, buf);
				if (!checkonly){
					if (!UdmFTPGet(Indexer->connp, Indexer->connp->connp, full_path,Indexer->Conf->max_doc_size)){
						udm_snprintf(Indexer->buf, Indexer->Conf->max_doc_size, "HTTP/1.1 200 OK\r\nContent-Type: %s\n\rLast-Modified: %s\r\n\r\n", 
							con_type,buf);
						size = strlen(Indexer->buf);
						if (Indexer->connp->connp->buf_len+size >= Indexer->Conf->max_doc_size)
							buf_size = Indexer->Conf->max_doc_size - size;
						else
							buf_size = Indexer->connp->connp->buf_len;
						
						memcpy(Indexer->buf+size, Indexer->connp->connp->buf, buf_size);
						size +=buf_size; 
					}else{
						if (Indexer->connp->err >0){
							sprintf(Indexer->buf, "HTTP/1.1 403 OK\r\n\r\n");
							size = strlen(Indexer->buf);
						}else{
							size = Indexer->connp->err;
			
						}
					}
				}else{
				    	sprintf(Indexer->buf, "HTTP/1.1 200 OK\r\nContent-Type: text/html\r\nLast-Modified: %s\r\n\r\n", buf);
					size = strlen(Indexer->buf);
				}
			}
		}
	}

	UDM_FREE(full_path);
	UDM_FREE(Indexer->connp->buf);
	UDM_FREE(Indexer->connp->connp->buf);
	return size;
}
#endif  

#ifdef USE_NEWS
int UdmNEWSGet(UDM_AGENT * Indexer,char *header,char *host,int port){
int fd,size,status;
char *s,*c,*e;
char str[UDMSTRSIZ];
FILE *f,*f1;
char filename[UDMSTRSIZ];
char command[32]="";
char proto[32]="";
int a=0,b=0,from=0,to=0,headers=0;
char *email=NULL;
char *msg_id=NULL;
char *subj=NULL;
int has_content=0;
int has_if_modified=0;
char *lt;

	sscanf(header,"%s%s%s",command,str,proto);
	if(str[0]=='/')strcpy(filename,str+1);
	else	strcpy(filename,str);
	s=UdmGetToken(header,"\r\n",&lt);
	while(s){
		if(!UDM_STRNCASECMP(s,"If-Modified-Since: "))
			has_if_modified=1;
		s=UdmGetToken(NULL,"\r\n",&lt);
	}

	/* Connect to NEWS server */
	fd=open_host(Indexer,host,port,Indexer->read_timeout);	/* Connect */
	if(fd<0)return(fd);			/* Could not connect */

	size=0;s=Indexer->buf;
	f=fdopen(fd,"r");f1=fdopen(fd,"w");
	fgets(str,sizeof(str),f); /* Read hello string */
	fprintf(f1,"mode reader\r\n");fflush(f1);
	fgets(str,sizeof(str),f); /* Read 2nd hello string*/ 
	
	/* All news groups on the server */
	if((!strcmp(filename,"/"))||(!strcmp(filename,""))){
		fprintf(f1,"list\r\n");fflush(f1);
		fgets(str,sizeof(str),f);
		sprintf(Indexer->buf,"HTTP/1.0 200 OK\r\nContent-Type: text/html\r\n\r\n<HTML><BODY>\n");
		while(fgets(str,sizeof(str),f)){
			if(str[0]=='.')break;
			if((c=strchr(str,' ')))*c=0;
			if(str[0] && ((strlen(Indexer->buf)+strlen(host)+strlen(str)) < Indexer->Conf->max_doc_size - 25))
			  sprintf(UDM_STREND(Indexer->buf),"<A HREF=\"news://%s/%s\"></A>\n",host,str);
		}
		sprintf(UDM_STREND(Indexer->buf),"</BODY></HTML>\n");
	}else /* List of the articles in one news group */
	if(!strchr(filename,'@')){
		if(filename[strlen(filename)-1]=='/')
			filename[strlen(filename)-1]='\0';
		fprintf(f1,"group %s\r\n",filename);fflush(f1);
		fgets(str,sizeof(str),f);
		sscanf(str,"%d%d%d%d",&a,&b,&from,&to);
		if(a != NNTP_GROUP_OK){
			/* Newsgroup does not exist */
			status=404;
			sprintf(Indexer->buf,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
		}else{
			fprintf(f1,"xover %d-%d\r\n",from,to);fflush(f1);
			fgets(str,sizeof(str),f);
			sprintf(Indexer->buf,"HTTP/1.0 200 OK\r\nContent-Type: text/html\r\n\r\n<HTML><BODY>\n");
			while(fgets(str,sizeof(str),f)){
				if(str[0]=='.')break;
				if((e=UdmGetToken(str,"\t\r\n",&lt)))
				if((subj=UdmGetToken(NULL,"\t\r\n",&lt)))
				if((email=UdmGetToken(NULL,"\t\r\n",&lt)))
				if((e=UdmGetToken(NULL,"\t\r\n",&lt)))
				if((msg_id=UdmGetToken(NULL,"\t\r\n",&lt)));
				str[0]=0;
				if(msg_id)sscanf(msg_id,"<%[^>]s>",str);
				if(str[0] && ((strlen(Indexer->buf)+strlen(host)+strlen(str)) < Indexer->Conf->max_doc_size - 25))
					sprintf(UDM_STREND(Indexer->buf),"<A HREF=\"news://%s/%s\"></A>\n",host,str);
				/* in the line above, I left 20 bytes of free */
				/* space at the end of the string for the     */
				/* closing string: */
			}
		}
		sprintf(UDM_STREND(Indexer->buf),"</BODY></HTML>\n");

	}else{ /* One article */
		if(has_if_modified){
			fprintf(f1,"stat <%s>\r\n",filename);fflush(f1);
			fgets(str,sizeof(str),f);sscanf(str,"%d",&a);
			if(a==223){
				status=304; /* Not modified */
				sprintf(Indexer->buf,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			}else{
				status=404; /* Not found */
				sprintf(Indexer->buf,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			}
		}else{
			fprintf(f1,"article <%s>\r\n",filename);fflush(f1);
			fgets(str,sizeof(str),f);sscanf(str,"%d",&a);
			if(a==220){
				headers=1;has_content=0;
				sprintf(Indexer->buf,"HTTP/1.0 200 OK\r\n");
				while(fgets(str,sizeof(str),f)){
					if(str[0]=='.')break;
					if(headers){
						if(!UDM_STRNCASECMP(str,"Content-Type:"))has_content=1;
						if((!strcmp(str,"\r\n"))||(!strcmp(str,"\n"))){
							headers=0;
							if(!has_content)strcat(Indexer->buf,"Content-Type: text/plain\r\n");
						}
					}
					if((int)(strlen(Indexer->buf)+strlen(str))<Indexer->Conf->max_doc_size)
						strcat(Indexer->buf,str);
				}
			}else{
				status=404;
				sprintf(Indexer->buf,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			}
		}
	}
	fprintf(f1,"quit\r\n");fflush(f1);
	fclose(f);/* Close NEWS connection */
	fclose(f1);
	return(strlen(Indexer->buf));
}


int UdmNNTPGet(UDM_AGENT * Indexer,char *header,char *host,int port){
int fd,size,status;
char *s,*c,*e;
char str[UDMSTRSIZ];
FILE *f,*f1;
char filename[UDMSTRSIZ];
char command[32]="";
char proto[32]="";
int a=0,b=0,from=0,to=0,headers=0;
int has_content=0;
int has_if_modified=0;
char *lt;

	sscanf(header,"%s%s%s",command,str,proto);
	if(str[0]=='/')strcpy(filename,str+1);
	else	strcpy(filename,str);

	if(filename[0])
		if(filename[strlen(filename)-1]=='/')
			filename[strlen(filename)-1]='\0';

	s=UdmGetToken(header,"\r\n",&lt);
	while(s){
		if(!UDM_STRNCASECMP(s,"If-Modified-Since: "))has_if_modified=1;
		s=UdmGetToken(NULL,"\r\n",&lt);
	}

	/* Connect to NNTP server */
	fd=open_host(Indexer,host,port,Indexer->read_timeout);	/* Connect */
	if(fd<0)return(fd);			/* Could not connect */

	size=0;s=Indexer->buf;
	f=fdopen(fd,"r");f1=fdopen(fd,"w");
	fgets(str,sizeof(str),f); /* Read hello string */
	fprintf(f1,"mode reader\r\n");fflush(f1);
	fgets(str,sizeof(str),f); /* Read 2nd hello string*/ 

	/* All news groups on the server */
	if((!strcmp(filename,"/"))||(!strcmp(filename,""))){
		fprintf(f1,"list\r\n");fflush(f1);
		fgets(str,sizeof(str),f);
		sprintf(Indexer->buf,"HTTP/1.0 200 OK\r\nContent-Type: text/html\r\n\r\n<HTML><BODY>\n");
		while(fgets(str,sizeof(str),f)){
			if(str[0]=='.')break;
			if((c=strchr(str,' ')))*c=0;
			sprintf(UDM_STREND(Indexer->buf),"<A HREF=\"nntp://%s/%s\"></A>\n",host,str);
		}
		sprintf(UDM_STREND(Indexer->buf),"</BODY></HTML>\n");
	}else /* List of the articles in one news group */
	if(!strchr(filename,'/')){
		fprintf(f1,"group %s\r\n",filename);fflush(f1);
		fgets(str,sizeof(str),f);
		sscanf(str,"%d%d%d%d",&a,&b,&from,&to);
		fprintf(f1,"xover %d-%d\r\n",from,to);fflush(f1);
		fgets(str,sizeof(str),f);
		sprintf(Indexer->buf,"HTTP/1.0 200 OK\r\nContent-Type: text/html\r\n\r\n<HTML><BODY>\n");
		while(fgets(str,sizeof(str),f)){
			if(str[0]=='.')break;
			if((e=UdmGetToken(str,"\t\r\n",&lt)))
				sprintf(UDM_STREND(Indexer->buf),
				"<A HREF=\"nntp://%s/%s/%s\"></A>\n",
				host,filename,e);
		}
		sprintf(UDM_STREND(Indexer->buf),"</BODY></HTML>\n");
	}else{ /* One article */
		c = strchr(filename,'/');
		*c = '\0'; c++;
		fprintf(f1, "group %s\r\n", filename); fflush(f1);
		fgets(str, sizeof(str),f);
		if(has_if_modified){
			fprintf(f1,"stat %s\r\n",c);fflush(f1);
			fgets(str,sizeof(str),f);sscanf(str,"%d",&a);
			if(a==223){
				status=304; /* Not modified */
				sprintf(Indexer->buf,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			}else{
				status=404; /* Not found */
				sprintf(Indexer->buf,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			}
		}else{
			fprintf(f1,"article %s\r\n",c);fflush(f1);
			fgets(str,sizeof(str),f);sscanf(str,"%d",&a);
			if(a==220){
				headers=1;has_content=0;
				sprintf(Indexer->buf,"HTTP/1.0 200 OK\r\n");
				while(fgets(str,sizeof(str),f)){
					if(str[0]=='.')break;
					if(headers){
						if(!UDM_STRNCASECMP(str,"Content-Type:"))has_content=1;
						if((!strcmp(str,"\r\n"))||(!strcmp(str,"\n"))){
							headers=0;
							if(!has_content)strcat(Indexer->buf,"Content-Type: text/plain\r\n");
						}
					}
					if((int)(strlen(Indexer->buf)+strlen(str))<Indexer->Conf->max_doc_size)
						strcat(Indexer->buf,str);
				}
			}else{
				status=404;
				sprintf(Indexer->buf,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			}
		}
	}
	fprintf(f1,"quit\r\n");fflush(f1);
	fclose(f);/* Close NNTP connection */
	fclose(f1);
	return(strlen(Indexer->buf));
}
#endif


#ifdef USE_FILE

#if (WIN32|WINNT)

static time_t FileTimeToStamp(FILETIME *ft){
	SYSTEMTIME st;
	struct tm t;

	FileTimeToSystemTime(ft, &st);
	t.tm_year	= st.wYear-1900;
	t.tm_mon	= st.wMonth;
	t.tm_mday	= st.wDay;
	t.tm_hour	= st.wHour;
	t.tm_min	= st.wMinute;
	t.tm_sec	= st.wSecond;

	return(mktime(&t));
}

int UdmFILEGet(UDM_AGENT * Indexer,char *header,char* user,char* passwd){
WIN32_FIND_DATA wfd;
HANDLE hFind;

WIN32_FILE_ATTRIBUTE_DATA wfad;
int is_dir;
int status=0;
int fd,size,l;
char filename[UDMSTRSIZ];
char newfilename[UDMSTRSIZ];
char openname[UDMSTRSIZ];
char command[32]="";
char proto[32]="";
char host[256]="";
char *content_type=NULL,*lt,*s;
time_t ims=0;

	sscanf(header,"%s%s%s",command,filename,proto);

	/* Remember If-Modified-Since timestamp and Host */
	s=UdmGetToken(header,"\r\n",&lt);
	while(s){
		if(!UDM_STRNCASECMP(s,"If-Modified-Since: ")){
			ims=UdmHttpDate2Time_t(s+19);
		}

		if(!UDM_STRNCASECMP(s,"Host: ")){
			strcpy(host,s+6);
		}
		s=UdmGetToken(NULL,"\r\n",&lt);
	}

	if(host[0]){
		strcpy(newfilename,"\\\\");
		strcat(newfilename,host);
		strcat(newfilename,filename);
	}else{
		strcpy(newfilename,filename);
	}
	UdmUnescapeCGIQuery(openname,newfilename);

	s=openname;
	while(*s){
		if(*s=='/')*s='\\';
		s++;
	}

	if(!GetFileAttributesEx(openname,GetFileExInfoStandard,&wfad)){
		int err=GetLastError();

		if(err==ERROR_LOGON_FAILURE){
			char cname[1024]="\\\\";
			NETRESOURCE nr;
			
			nr.dwType		= RESOURCETYPE_DISK;
			nr.lpLocalName	= NULL;
			strcat(cname,host);
			nr.lpRemoteName	= cname;
			nr.lpProvider	= NULL;

			err=WNetAddConnection2(&nr, passwd, user, 0);

			switch(err){
				case NO_ERROR: break;
				case ERROR_LOGON_FAILURE:
				default: 
					status=403; 
					break;
			}
			if(!GetFileAttributesEx(openname,GetFileExInfoStandard,&wfad)) status=403;
		}else{
			switch(err){
				case ERROR_FILE_NOT_FOUND:
				case ERROR_PATH_NOT_FOUND:	status=404; break;
				case ERROR_ACCESS_DENIED:	status=403; break;
				default:					status=500; break;
			}
		}
		if(status){
			sprintf(Indexer->buf,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			return(strlen(Indexer->buf));
		}
	}


	is_dir=wfad.dwFileAttributes&FILE_ATTRIBUTE_DIRECTORY;

	if(is_dir){
		if(filename[strlen(filename)-1]!='/') strcat(filename,"/");
		strcat(openname,"\\*");
		hFind=FindFirstFile(openname, &wfd);
		if(hFind==INVALID_HANDLE_VALUE){
			status=500;
			sprintf(Indexer->buf,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			return(strlen(Indexer->buf));
		}
		strcpy(Indexer->buf,"HTTP/1.0 200 OK\r\nContent-type: text/html\r\n\r\n<HTML><BODY>\n");
		do{
			char escaped_name[UDMSTRSIZ]="";
			char *src,*e;

			if(!strcmp(wfd.cFileName,"..") || !strcmp(wfd.cFileName,".")) continue;

			e=escaped_name;
			for(src=wfd.cFileName;*src;src++){
				if(strchr(" %&<>+[](){}/?#'\"\\;,",*src)){
					sprintf(e,"%%%X",(int)*src);
					e+=3;
				}else{
					*e=*src;
					e++;
				}
			}

			*e=0;
			is_dir=wfd.dwFileAttributes&FILE_ATTRIBUTE_DIRECTORY;
			if(host[0]){
				sprintf(UDM_STREND(Indexer->buf),"<A HREF=\"file://%s%s%s%s\">%s</A>\n",
					host,filename,escaped_name,is_dir?"/":"",escaped_name);
			}else{
				sprintf(UDM_STREND(Indexer->buf),"<A HREF=\"file:%s%s%s\">%s</A>\n",
					filename,escaped_name,is_dir?"/":"",escaped_name);
			}
		}while(FindNextFile(hFind, &wfd));

		strcpy(UDM_STREND(Indexer->buf),"</BODY><HTML>\n");
		FindClose(hFind);
		return(strlen(Indexer->buf));
	}else{
		/* Lets compare last modification date */

		if(ims>=FileTimeToStamp(&wfad.ftLastWriteTime)){
			status=304;
			sprintf(Indexer->buf,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			size=strlen(Indexer->buf);
			return(size);
		}

		if((fd=open(openname,O_RDONLY|UDM_BINARY))<0){
			int err=errno;
			switch(err){
				case ENOENT: status=404;break;
				case EACCES: status=403;break;
				default: status=1;
			}
			sprintf(Indexer->buf,"HTTP/1.0 %d %s %s\r\n\r\n",status,UdmHTTPErrMsg(status),strerror(err));
			return(strlen(Indexer->buf));
		}
		sprintf(Indexer->buf,"HTTP/1.0 200 OK\r\n");
		if((content_type=UdmContentType(Indexer->Conf,filename)))
			sprintf(UDM_STREND(Indexer->buf),"Content-Type: %s\r\n",content_type);

		
		strcpy(UDM_STREND(Indexer->buf),"Last-Modified: ");
		UdmTime_t2HttpStr(FileTimeToStamp(&wfad.ftLastWriteTime), UDM_STREND(Indexer->buf));
		strcpy(UDM_STREND(Indexer->buf),"\r\n");

		sprintf(UDM_STREND(Indexer->buf),"\r\n");
		l=strlen(Indexer->buf);
		size = read(fd, Indexer->buf + l, Indexer->Conf->max_doc_size - l) + l;
		close(fd);
		return(size);
	}
}

#else

int UdmFILEGet(UDM_AGENT * Indexer,char *header){
int fd,size,l,status=0,is_dir;
struct stat sb,sb1;
DIR * dir;
struct dirent *rec;
char filename[UDMSTRSIZ];
char newfilename[UDMSTRSIZ];
char openname[UDMSTRSIZ];
char mystatname[UDMSTRSIZ];
char command[32]="";
char proto[32]="";
char *content_type=NULL,*lt,*s;
time_t ims=0;

	sscanf(header,"%s%s%s",command,filename,proto);
	strcpy(newfilename,filename);
	UdmUnescapeCGIQuery(openname,newfilename);

	/* Remember If-Modified-Since timestamp */
	s=UdmGetToken(header,"\r\n",&lt);
	while(s){
		if(!UDM_STRNCASECMP(s,"If-Modified-Since: ")){
			ims=UdmHttpDate2Time_t(s+19);
		}
		s=UdmGetToken(NULL,"\r\n",&lt);
	}

	strcpy(mystatname,openname);
	
	if(stat(mystatname,&sb)){
		switch(errno){
			case ENOENT: status=404;break; /* Not found */
			case EACCES: status=403;break; /* Forbidden*/
			default: status=500;
		}
		sprintf(Indexer->buf,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
		return(strlen(Indexer->buf));
	}

	/* If directory is given without ending "/"   */
	/* we must redirect to the same URL with "/"  */
	if((sb.st_mode&S_IFDIR)&&(filename[strlen(filename)-1]!='/')){
		status=301;
		sprintf(Indexer->buf,"HTTP/1.0 %d %s\r\nLocation: file:%s/\r\n\r\n",status,UdmHTTPErrMsg(status),filename);
		return(strlen(Indexer->buf));
	}
	
	if(sb.st_mode&S_IFDIR){
		if((dir=opendir(openname))){
			strcpy(Indexer->buf,"HTTP/1.0 200 OK\r\nContent-type: text/html\r\n\r\n<HTML><BODY>\n");
			while((rec = readdir (dir))){
				char escaped_name[UDMSTRSIZ]="";
				char *src,*e;
				/*
				This does not work on Solaris
				is_dir=(rec->d_type==DT_DIR);
				*/
				sprintf(newfilename,"%s%s",openname,rec->d_name);
				if(stat(newfilename,&sb1)){
					/* FIXME!!! Handler should be here */
				}
				is_dir=((sb1.st_mode&S_IFDIR)>0);
				
				e=escaped_name;
				for(src=rec->d_name;*src;src++){
					if(strchr(" %&<>+[](){}/?#'\"\\;,",*src)){
						sprintf(e,"%%%X",(int)*src);
						e+=3;
					}else{
						*e=*src;
						e++;
					}
				}
				*e=0;
				sprintf(UDM_STREND(Indexer->buf),"<A HREF=\"%s%s\">%s%s</A>\n",
					escaped_name,is_dir?"/":"",
					escaped_name,is_dir?"/":"");
			}
			closedir(dir);
			strcpy(UDM_STREND(Indexer->buf),"</BODY><HTML>\n");
			return(strlen(Indexer->buf));
		}else{
			switch(errno){
				case ENOENT: status=404;break; /* Not found */
				case EACCES: status=403;break; /* Forbidden*/
				default: status=500;
			}
			sprintf(Indexer->buf,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			return(strlen(Indexer->buf));
		}
	}else{
		/* Lets compare last modification date */
		if(ims>=sb.st_mtime){
			/* Document seems to be unchanged */
			status=304;
			sprintf(Indexer->buf,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			size=strlen(Indexer->buf);
			return(size);
		}

		if((fd=open(openname,O_RDONLY|UDM_BINARY))<0){
			switch(errno){
				case ENOENT: status=404;break;
				case EACCES: status=403;break;
				default: status=1;
			}
			sprintf(Indexer->buf,"HTTP/1.0 %d %s\r\n\r\n",status,UdmHTTPErrMsg(status));
			return(strlen(Indexer->buf));
		}
		sprintf(Indexer->buf,"HTTP/1.0 200 OK\r\n");
		if((content_type=UdmContentType(Indexer->Conf,filename)))
			sprintf(UDM_STREND(Indexer->buf),"Content-Type: %s\r\n",content_type);

		strcpy(UDM_STREND(Indexer->buf),"Last-Modified: ");
		UdmTime_t2HttpStr(sb.st_mtime, UDM_STREND(Indexer->buf));
		strcpy(UDM_STREND(Indexer->buf),"\r\n");

		sprintf(UDM_STREND(Indexer->buf),"\r\n");
		l=strlen(Indexer->buf);
		size = read(fd, Indexer->buf + l, Indexer->Conf->max_doc_size - l) + l;
		close(fd);
		return(size);
	}
}

#endif

#endif
