# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Dialog with internal calibration.

# load libs
import wx

# load modules
import count
from dlg_calibration import dlgCalibration
from nucleus import mwx
from nucleus import commfce


class dlgInterCalib(dlgCalibration):
    """ Internal calibration. """

    # ----
    def __init__(self, parent, peaklist, config):

        # init base class
        self.digits = config.cfg['common']['digits']
        self.grayColour = config.cfg['colours']['grayed']
        dlgCalibration.__init__(self, parent, title="Internal Calibration", calibType='internal', grayColour=self.grayColour)

        self.availableReferences = config.ref
        self.currentReferences = []

        # calculate peaklist
        for peak in peaklist:
            self.peaklist.append([False, peak[0], None, None, None])

        # make items
        self.makePeakList()
        references = self.makeReferenceList()
        edit = self.makeEditBox()
        calibration = self.makeCalibrationBox()
        results = self.makeResultsBox()
        zoom = self.makeZoomBox()
        buttons = self.makeButtonBox()

        # pack main frame
        col1Sizer = wx.BoxSizer(wx.VERTICAL)
        col2Sizer = wx.BoxSizer(wx.VERTICAL)
        rowSizer = wx.BoxSizer(wx.HORIZONTAL)
        mainSizer = wx.BoxSizer(wx.VERTICAL)

        if wx.Platform == '__WXMAC__':
            col1Sizer.Add(edit, 0, wx.EXPAND|wx.BOTTOM, 5)
            col1Sizer.Add(calibration, 0, wx.EXPAND|wx.BOTTOM, 5)
            col1Sizer.Add(results, 0, wx.EXPAND|wx.BOTTOM, 5)
            col1Sizer.Add(zoom, 0, wx.EXPAND)
            col2Sizer.Add(self.peaklist_list, 1, wx.EXPAND, 0)
            col2Sizer.Add(references, 1, wx.EXPAND)
            rowSizer.Add(col1Sizer, 0, wx.EXPAND|wx.RIGHT, 20)
            rowSizer.Add(col2Sizer, 1, wx.EXPAND)
            mainSizer.Add(rowSizer, 1, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 20)
            mainSizer.Add(buttons, 0, wx.ALIGN_CENTER|wx.ALL, 20)
        else:
            col1Sizer.Add(edit, 0, wx.EXPAND|wx.BOTTOM, 5)
            col1Sizer.Add(calibration, 0, wx.EXPAND|wx.BOTTOM, 5)
            col1Sizer.Add(results, 0, wx.EXPAND|wx.BOTTOM, 5)
            col1Sizer.Add(zoom, 0, wx.EXPAND)
            col2Sizer.Add(self.peaklist_list, 1, wx.EXPAND, 0)
            col2Sizer.Add(references, 1, wx.EXPAND)
            rowSizer.Add(col1Sizer, 0, wx.EXPAND|wx.RIGHT, 10)
            rowSizer.Add(col2Sizer, 1, wx.EXPAND)
            mainSizer.Add(rowSizer, 1, wx.EXPAND|wx.ALL, 5)
            mainSizer.Add(buttons, 0, wx.ALIGN_CENTER|wx.ALL, 5)

        # update fields
        self.updatePeaklist()
        self.updatePeaksCount()
        self.onReferenceListChanged()

        # fit layout
        mainSizer.Fit(self)
        self.SetSizer(mainSizer)
        self.SetMinSize(self.GetSize())
        self.Centre()
    # ----


    # ----
    def makePeakList(self):
        """ Make box with peaks. """

        # set style
        if wx.Platform == '__WXMAC__':
            style = wx.LC_REPORT|wx.LC_VRULES|wx.LC_HRULES|wx.LC_SINGLE_SEL|wx.SIMPLE_BORDER
        else:
            style = wx.LC_REPORT|wx.LC_VRULES|wx.LC_HRULES|wx.LC_SINGLE_SEL

        # make items
        self.peaklist_list = mwx.ListCtrl(self, -1, size=(310, 200), style=style)
        self.peaklist_list.InsertColumn(0, "#", wx.LIST_FORMAT_RIGHT)
        self.peaklist_list.InsertColumn(1, "Status", wx.LIST_FORMAT_CENTER)
        self.peaklist_list.InsertColumn(2, "m/z", wx.LIST_FORMAT_RIGHT)
        self.peaklist_list.InsertColumn(3, "Ref.", wx.LIST_FORMAT_RIGHT)
        self.peaklist_list.InsertColumn(4, "Calib.", wx.LIST_FORMAT_RIGHT)
        self.peaklist_list.InsertColumn(5, "Error [Da]", wx.LIST_FORMAT_RIGHT)

        # set events
        self.peaklist_list.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onPeakSelected)
        self.peaklist_list.Bind(wx.EVT_LIST_ITEM_DESELECTED, self.onPeakDeselected)
        self.peaklist_list.Bind(wx.EVT_LIST_ITEM_ACTIVATED, self.onPeakActivated)

        # set columns width
        for col in range(6):
            self.peaklist_list.SetColumnWidth(col, wx.LIST_AUTOSIZE)
    # ----


    # ----
    def makeReferenceList(self):
        """ Make box with references. """

        # get available lists of references
        list_choices = []
        for item in self.availableReferences:
            list_choices.append(item)
        list_choices.sort()

        # set style
        if wx.Platform == '__WXMAC__':
            style = wx.LC_REPORT|wx.LC_VRULES|wx.LC_HRULES|wx.LC_SINGLE_SEL|wx.SIMPLE_BORDER
        else:
            style = wx.LC_REPORT|wx.LC_VRULES|wx.LC_HRULES|wx.LC_SINGLE_SEL

        # make items
        reference_label = wx.StaticText(self, -1, "Reference lists: ")
        self.reference_combo = wx.ComboBox(self, -1, size=(100, -1), choices=list_choices, style=wx.CB_READONLY)

        self.reference_list = mwx.ListCtrl(self, -1, size=(310, 150), style=style)
        self.reference_list.InsertColumn(0, "Reference name", wx.LIST_FORMAT_LEFT)
        self.reference_list.InsertColumn(1, "m/z", wx.LIST_FORMAT_RIGHT)

        # pack items
        listBox = wx.BoxSizer(wx.HORIZONTAL)
        listBox.Add(reference_label, 0, wx.ALIGN_CENTER_VERTICAL|wx.RIGHT, 5)
        listBox.Add(self.reference_combo, 1)

        mainBox = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainBox.Add(listBox, 0, wx.EXPAND|wx.TOP|wx.LEFT|wx.BOTTOM, 20)
        else:    
            mainBox.Add(listBox, 0, wx.EXPAND|wx.TOP|wx.LEFT|wx.BOTTOM, 5)
        mainBox.Add(self.reference_list, 1, wx.EXPAND)

        # set defaults
        if list_choices:
            self.reference_combo.Select(0)

        # set events
        self.reference_combo.Bind(wx.EVT_COMBOBOX, self.onReferenceListChanged)
        self.reference_list.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onReferenceSelected)

        # set columns width
        for col in range(2):
            self.reference_list.SetColumnWidth(col, wx.LIST_AUTOSIZE)

        return mainBox
    # ----


    # ----
    def makeEditBox(self):
        """ Make box for peaks' editing. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Edit Reference Mass"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        editMeasured_label = wx.StaticText(self, -1, "Measured: ")
        self.editMeasured_value = wx.TextCtrl(self, -1, size=(85, -1), style=wx.TE_READONLY|wx.TE_RIGHT)
        self.editMeasured_value.Enable(False)

        editReference_label = wx.StaticText(self, -1, "Reference: ")
        self.editReference_value = wx.TextCtrl(self, -1, size=(85, -1), style=wx.TE_RIGHT, validator=mwx.txtValidator('float'))

        self.Edit_button = wx.Button(self, -1, "Update", size=(85, -1))
        self.Edit_button.Enable(False)

        # pack items
        grid.Add(editMeasured_label, (0, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.editMeasured_value, (0, 1))
        grid.Add(editReference_label, (1, 0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.editReference_value, (1, 1))
        grid.Add(self.Edit_button, (2, 1))

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 0, wx.EXPAND, 0)
        else:
            mainBox.Add(grid, 0, wx.EXPAND|wx.ALL, 5)

        # set events
        self.Edit_button.Bind(wx.EVT_BUTTON, self.onPeakEdited)

        return mainBox
    # ----


    # ----
    def updatePeaklist(self):
        """ Update list of loaded peaks. """

        # clear list
        self.peaklist_list.DeleteAllItems()

        # paste new data
        digitsFormat = '%0.' + `self.digits` + 'f'
        for x, peak in enumerate(self.peaklist):

            # format data
            use = "skip"
            measured = digitsFormat % peak[1]
            reference = ''
            calibrated = ''
            error = ''
            if peak[0]:
                use = "use"
            if peak[2] != None:
                reference = digitsFormat % peak[2]
            if peak[3] != None:
                calibrated = digitsFormat % peak[3]
            if peak[4] != None:
                error = digitsFormat % peak[4]

            # show data
            self.peaklist_list.InsertStringItem(x, str(x+1))
            self.peaklist_list.SetStringItem(x, 1, use)
            self.peaklist_list.SetStringItem(x, 2, measured)
            self.peaklist_list.SetStringItem(x, 3, reference)
            self.peaklist_list.SetStringItem(x, 4, calibrated)
            self.peaklist_list.SetStringItem(x, 5, error)

            # set colour
            if not peak[0]:
                self.peaklist_list.SetItemTextColour(x, self.grayColour)

        # set columns width
        self.peaklist_list.SetColumnWidth(0, wx.LIST_AUTOSIZE)
        self.peaklist_list.SetColumnWidth(1, wx.LIST_AUTOSIZE_USEHEADER)
        self.peaklist_list.SetColumnWidth(2, wx.LIST_AUTOSIZE)
        self.peaklist_list.SetColumnWidth(3, wx.LIST_AUTOSIZE)
        self.peaklist_list.SetColumnWidth(4, wx.LIST_AUTOSIZE)
        self.peaklist_list.SetColumnWidth(5, wx.LIST_AUTOSIZE)
        self.peaklist_list.updateLastCol()
    # ----


    # ----
    def updateEdit(self):
        """ Show selected peak. """

        mass = ''
        reference = ''

        # get data
        if self.selected != -1:
            digitsFormat = '%0.' + `self.digits` + 'f'
            mass = digitsFormat % self.peaklist[self.selected][1]
            reference = self.peaklist[self.selected][2]
            if reference:
                reference = str(reference)
            else:
                reference = ''

        # update fields
        self.editMeasured_value.SetValue(mass)
        self.editReference_value.SetValue(reference)
    # ----


    # ----
    def updateReferenceList(self):
        """ Update list of references. """

        # clear list
        self.reference_list.DeleteAllItems()

        # paste new data
        digitsFormat = '%0.' + `self.digits` + 'f'
        for x, reference in enumerate(self.currentReferences):
            mz = digitsFormat % reference[1]
            self.reference_list.InsertStringItem(x, reference[0])
            self.reference_list.SetStringItem(x, 1, mz)

        # set columns width
        self.reference_list.SetColumnWidth(0, wx.LIST_AUTOSIZE)
        self.reference_list.SetColumnWidth(1, wx.LIST_AUTOSIZE)
        self.reference_list.updateLastCol()
    # ----


    # ----
    def onPeakSelected(self, evt):
        """ Select peak for editting. """

        # get selected peak and update editbox
        self.selected = evt.m_itemIndex
        self.updateEdit()
        self.Edit_button.Enable(True)

        # show selected peak
        peak = self.peaklist[self.selected][1]
        self.highlightPoint(peak)
    # ----


    # ----
    def onPeakDeselected(self, evt):
        """ Deselect peak for editting. """

        self.selected = -1
        self.updateEdit()
        self.Edit_button.Enable(False)
    # ----


    # ----
    def onPeakEdited(self, evt):
        """ Edit reference of selected peak. """

        if self.selected == -1:
            return

        # get reference and error
        reference = self.editReference_value.GetValue()
        reference = reference.replace(',', '.')
        if reference == '':
            reference = None
            error = None
        else:
            try:
                reference = float(reference)
            except ValueError:
                return
            error = self.peaklist[self.selected][1] - reference

        # update peaklist values
        self.peaklist[self.selected][2] = reference
        self.peaklist[self.selected][4] = error

        # update peaklist
        digitsFormat = '%0.' + `self.digits` + 'f'
        if reference != None:
            self.peaklist[self.selected][0] = True
            self.peaklist_list.SetStringItem(self.selected, 1, "use")
            self.peaklist_list.SetItemTextColour(self.selected, (0, 0, 0))
            self.peaklist_list.SetStringItem(self.selected, 3, digitsFormat % reference)
            self.peaklist_list.SetStringItem(self.selected, 5, digitsFormat % error)
        else:
            self.peaklist[self.selected][0] = False
            self.peaklist_list.SetStringItem(self.selected, 1, "skip")
            self.peaklist_list.SetItemTextColour(self.selected, self.grayColour)
            self.peaklist_list.SetStringItem(self.selected, 3, '')
            self.peaklist_list.SetStringItem(self.selected, 5, '')

        self.peaklist_list.SetColumnWidth(3, wx.LIST_AUTOSIZE)
        self.peaklist_list.SetColumnWidth(4, wx.LIST_AUTOSIZE)
        self.peaklist_list.SetColumnWidth(5, wx.LIST_AUTOSIZE)
        self.peaklist_list.updateLastCol()

        # update peaks' counter
        self.updatePeaksCount()
    # ----


    # ----
    def onReferenceListChanged(self, evt=None):
        """ Update reference list. """

        # get references
        refList = self.reference_combo.GetValue()
        self.currentReferences = []
        if self.availableReferences.has_key(refList):
            for item in self.availableReferences[refList]:
                self.currentReferences.append((item, float(self.availableReferences[refList][item])))

        # sort references by mass
        self.currentReferences = commfce.sortMultiList(self.currentReferences, 1)
        
        # update references list
        self.updateReferenceList()
    # ----


    # ----
    def onReferenceSelected(self, evt):
        """ Set selected reference to selected peak. """

        # get reference value
        reference = self.currentReferences[evt.m_itemIndex][1]

        # update editbox
        self.editReference_value.SetValue(str(reference))

        # show selected mass in the spectrum
        self.highlightPoint(reference)
    # ----


    # ----
    def onClear(self, evt):
        """ Clear calibration data. """

        self.fit = []
        self.Clear_button.Enable(False)
        self.OK_button.Enable(False)
        self.selected = -1

        # clear peaklist data
        for x, peak in enumerate(self.peaklist):
            peak[3] = None
            if peak[2]:
                peak[4] = peak[1] - peak[2]

        # update peaklist and results
        self.updatePeaklist()
        self.updateEdit()
        self.updateResults()
    # ----
