/*
 * Written by Bastien Chevreux (BaCh)
 *
 * Copyright (C) 2009 and later by Bastien Chevreux
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 *
 *
 */


#include "maf_parse.H"
#include <stdlib.h>
#include <string.h>

#include "errorhandling/errorhandling.H"
#include "util/progressindic.H"



// 	$Id$	

#ifndef lint
static char vcid[] = "$Id$";
#endif /* lint */



MAFParse::MAFParse(ReadPool &aPool, list<Contig> &aContiglist, vector<MIRAParameters> * mp) {
  MAF_readpool    = &aPool;
  MAF_contiglist = &aContiglist;
  MAF_miraparams = mp;
}


MAFParse::~MAFParse() {
  //discard();
}



//void MAFParse::discard() {
//}


size_t MAFParse::countReadsBeforeLoad(const string & fileName, size_t & maxlinelen) 
{
  FUNCSTART("void MAFParse::countReadsBeforeLoad()");

  ifstream mafin;

  size_t numseqsloaded=0;
  maxlinelen=0;

  mafin.open(fileName.c_str(), ios::in|ios::ate);
  if(!mafin) {
    throw Notify(Notify::FATAL, fileName.c_str(),"MAF file not found for loading."); 
  }
  if(!mafin.tellg() ) {
    throw Notify(Notify::FATAL, fileName.c_str(),"MAF file is empty.");
  }

  ProgressIndicator<streamsize> P(0, mafin.tellg(),5000);

  mafin.seekg(0, ios::beg);

  string actline;
  string acttoken;

  actline.reserve(10000);

  while(!mafin.eof()){
    if(mafin.eof()) break;
    getline(mafin,actline);

    if(actline.size()>=2
       && actline[0]=='R'
       && actline[1]=='D') {
      numseqsloaded++;
    }
    if(P.delaytrigger()) P.progress(mafin.tellg());
  }
  P.finishAtOnce();

  mafin.close();

  maxlinelen=actline.capacity();

  FUNCEND();

  return numseqsloaded;
}



void MAFParse::cleanupHeaderData()
{
  MAF_vmajor=-1;
  MAF_vminor=-1;
}

void MAFParse::cleanupContigData()
{
  MAF_contig_assembledfrom.clear();
  MAF_contig_sequence.clear();
  MAF_contig_qualities.clear();
  MAF_contig_taglist.clear();

  MAF_contig_name.clear();

  MAF_contig_numreads=0;
  MAF_contig_len=0;
  MAF_contig_rawlen=0;

}

void MAFParse::cleanupReadData()
{
  MAF_read_sequence.clear();
  MAF_read_qualities.clear();
  MAF_read_align_origin.clear();
  MAF_read_taglist.clear();

  MAF_read_name.clear();
  MAF_read_scf_file.clear();
  MAF_read_template.clear();
  MAF_read_base_caller.clear();
  MAF_read_sequencing_vector.clear();
  MAF_read_strain.clear();
  MAF_read_machinetype.clear();

  MAF_read_len=-1;
  MAF_read_insert_size_min=-1;
  MAF_read_insert_size_max=-1;

  MAF_read_ql=-1;
  MAF_read_qr=-1;
  MAF_read_cl=-1;
  MAF_read_cr=-1;
  MAF_read_sl=-1;
  MAF_read_sr=-1;

  MAF_read_strand_given='N';
  MAF_read_seqtype=Read::SEQTYPE_SANGER;

  MAF_read_isbackbone=false;
  MAF_read_israil=false;
  MAF_read_isCER=false;

  MAF_readpoolid=-1;
}


/*
  seqtype = default seqtype of sequences if not encoded in the CAF
  loadaction:
    //  0 = count only
    //  1 = count, adjust readpool capacity and load
    //  2 = load only
  lrperseqtype = longest read per seqtype

  returns:
    1) number of sequences loaded
    2) a) if loadaction = 0 and lrperseqtype empty: nothing more
       b) else size of longest read per seqtype in lrperseqtype
 */
size_t MAFParse::load(const string & fileName, const uint8 seqtype, const uint8 loadaction, vector<uint32> & lrperseqtype, bool recalcconsensus, void  (*callback)(list<Contig> &, ReadPool &), bool isVerbose) 
{
  FUNCSTART("void MAFParse::load()");

  MAF_callbackfunc=callback;
  MAF_recalcconsensus=recalcconsensus;

  MAF_simulateloading=false;

  size_t numseqsloaded=0;
  size_t maxlinelen=0;
  if(loadaction<2){
    cout << "First counting reads:\n";
    numseqsloaded=countReadsBeforeLoad(fileName,maxlinelen);

    if(loadaction==0){
      if(lrperseqtype.empty()) return numseqsloaded;
      MAF_simulateloading=true;
    }

    if(loadaction==1) MAF_readpool->reserve(MAF_readpool->size()+numseqsloaded+10);
  }

  numseqsloaded=0;

  if(MAF_simulateloading){
    cout << "\nNow simulating loading of data:\n";
  }else{
    cout << "\nNow loading and processing data:\n";
  }

  MAF_lrperseqtype.clear();
  MAF_lrperseqtype.resize(Read::SEQTYPE_END,0);

  ifstream mafin;

  mafin.open(fileName.c_str(), ios::in|ios::ate);
  if(!mafin) {
    throw Notify(Notify::FATAL, fileName.c_str(),"CAF file not found for loading."); 
  }
  if(!mafin.tellg() ) {
    throw Notify(Notify::FATAL, fileName.c_str(),"CAF file is empty.");
  }

  ProgressIndicator<streamsize> P(0, mafin.tellg(),5000);

  mafin.seekg(0, ios::beg);
  
  string acttoken;
  string actline;

  size_t linenumber=0;

  if(maxlinelen>10000) {
    actline.reserve(maxlinelen);
  }else{
    actline.reserve(10000);
  }

  MAF_isinread=false;
  MAF_isincontig=false;

  cleanupHeaderData();
  cleanupReadData();
  cleanupContigData();

  while(true){
    linenumber++;
    mafin >> acttoken;
    if(mafin.eof()) break;

    CEBUG("l: " << linenumber << "\tt: ###" << acttoken << "###" << endl);

    if(acttoken.empty()) continue;

    if(acttoken=="FV"){
      // file version
      parseLineFV(mafin,acttoken,actline);

/* here for read*/

    }else if(acttoken=="RS"){
      // Read Sequence
      parseLineRS(mafin,acttoken,actline);
    }else if(acttoken=="RQ"){
      // Read Qualities
      parseLineRQ(mafin,acttoken,actline);
    }else if(acttoken=="RD"){
      // read name
      parseLineRD(mafin,acttoken,actline);
    }else if(acttoken=="LR"){
      // length read
      parseLineLR(mafin,acttoken,actline);
    }else if(acttoken=="SV"){
      // sequencing vector
      parseLineSV(mafin,acttoken,actline);
    }else if(acttoken=="TN"){
      // template name
      parseLineTN(mafin,acttoken,actline);
    }else if(acttoken=="DI"){
      // Direction (strand)
      parseLineDI(mafin,acttoken,actline);
    }else if(acttoken=="TF"){
      // template insize from
      parseLineTF(mafin,acttoken,actline);
    }else if(acttoken=="TT"){
      // template insize to
      parseLineTT(mafin,acttoken,actline);
    }else if(acttoken=="SF"){
      // Sequencing File
      parseLineSF(mafin,acttoken,actline);
    }else if(acttoken=="BC"){
      // base caller
      parseLineBC(mafin,acttoken,actline);
    }else if(acttoken=="SL"){
      //
      parseLineSL(mafin,acttoken,actline);
    }else if(acttoken=="SR"){
      //
      parseLineSR(mafin,acttoken,actline);
    }else if(acttoken=="QL"){
      //
      parseLineQL(mafin,acttoken,actline);
    }else if(acttoken=="QR"){
      //
      parseLineQR(mafin,acttoken,actline);
    }else if(acttoken=="CL"){
      //
      parseLineCL(mafin,acttoken,actline);
    }else if(acttoken=="CR"){
      //
      parseLineCR(mafin,acttoken,actline);
    }else if(acttoken=="AO"){
      //
      parseLineAO(mafin,acttoken,actline);
    }else if(acttoken=="RT"){
      //
      parseLineRT(mafin,acttoken,actline);
    }else if(acttoken=="ST"){
      //
      parseLineST(mafin,acttoken,actline);
    }else if(acttoken=="SN"){
      //
      parseLineSN(mafin,acttoken,actline);
    }else if(acttoken=="MT"){
      //
      parseLineMT(mafin,acttoken,actline);
    }else if(acttoken=="IB"){
      //
      parseLineIB(mafin,acttoken,actline);
    }else if(acttoken=="IC"){
      //
      parseLineIC(mafin,acttoken,actline);
    }else if(acttoken=="IR"){
      //
      parseLineIR(mafin,acttoken,actline);
    }else if(acttoken=="ER"){
      //
      parseLineER(mafin,acttoken,actline);

/* here for contig*/
    }else if(acttoken=="CS"){
      // Consensus Sequence
      parseLineCS(mafin,acttoken,actline);
    }else if(acttoken=="CQ"){
      // Consensus Qualities
      parseLineCQ(mafin,acttoken,actline);
    }else if(acttoken=="CO"){
      // COntig name
      parseLineCO(mafin,acttoken,actline);
    }else if(acttoken=="NR"){
      // Num Reads
      parseLineNR(mafin,acttoken,actline);
    }else if(acttoken=="LC"){
      // Length Contig
      parseLineLC(mafin,acttoken,actline);
    }else if(acttoken=="CT"){
      // Contig Tag
      parseLineCT(mafin,acttoken,actline);
    }else if(acttoken=="//"){
      // start of contig reads

    }else if(acttoken=="\\\\"){
      // end of contig reads

    }else if(acttoken=="AT"){
      // Assembled From
      parseLineAT(mafin,acttoken,actline);
    }else if(acttoken=="EC"){
      // end contig
      parseLineEC(mafin,acttoken,actline);
    }else{
      cout << "File " << fileName << ": around line " << linenumber
	   << "\ndid not recognize token " << acttoken << '\n';
      MIRANOTIFY(Notify::FATAL, "Error while reading MAF file.");
    }

    if(P.delaytrigger()) P.progress(mafin.tellg());
  }

  P.finishAtOnce();
  cout << endl;

  if(MAF_isincontig){
    MIRANOTIFY(Notify::FATAL, "MAF file ends without closing and open contig. File truncated?");
  }
  if(MAF_isinread){
    MIRANOTIFY(Notify::FATAL, "MAF file ends without closing and open read. File truncated?");
  }

  lrperseqtype=MAF_lrperseqtype;

  FUNCEND();

  return numseqsloaded;
}


void MAFParse::checkParseIsInRead(string & acttoken)
{
  FUNCSTART("void MAFParse::checkParseIsInRead(string & acttoken)");
  if(!MAF_isinread) {
    MIRANOTIFY(Notify::FATAL,"Encountered " << acttoken << " line while not in read (RD line missing?)");
  }
  FUNCEND();
}

void MAFParse::checkParseIsNotInRead(string & acttoken)
{
  FUNCSTART("void MAFParse::checkParseIsNotInRead(string & acttoken)");
  if(MAF_isinread) {
    MIRANOTIFY(Notify::FATAL,"Encountered " << acttoken << " line while bein in read (ER line missing?)");
  }
  FUNCEND();
}

void MAFParse::checkParseIsInContig(string & acttoken)
{
  FUNCSTART("void MAFParse::checkParseIsInContig(string & acttoken)");
  if(!MAF_isincontig) {
    MIRANOTIFY(Notify::FATAL,"Encountered " << acttoken << " line while not in contig (CO line missing?)");
  }
  if(MAF_isinread) {
    MIRANOTIFY(Notify::FATAL,"Encountered " << acttoken << " line while being in read (RD line not closed by ER?)");
  }
  FUNCEND();
}

void MAFParse::parseLineFV(ifstream & mafin, string & acttoken, string & actline)
{
  mafin >> MAF_vmajor >> MAF_vminor;
}



void MAFParse::parseLineRD(ifstream & mafin, string & acttoken, string & actline)
{
  FUNCSTART("void MAFParse::parseLineRD(ifstream & mafin, string & acttoken, string & actline)");
  if(MAF_isinread) {
    MIRANOTIFY(Notify::FATAL,"Encountered new " << acttoken << " line when the previous read " << MAF_read_name << " was not closed with 'ER'");
  }
  cleanupReadData();
  mafin >> MAF_read_name;
  MAF_isinread=true;

  FUNCEND();
}

void MAFParse::parseLineRS(ifstream & mafin, string & acttoken, string & actline)
{
  FUNCSTART("void MAFParse::parseLineRS(ifstream & mafin, string & acttoken, string & actline)");

  checkParseIsInRead(acttoken);

  if(!MAF_read_sequence.empty()){
    MIRANOTIFY(Notify::FATAL,"Encountered RS line when there already was one for read " << MAF_read_name);
  }

  mafin >> actline;

  MAF_read_sequence.reserve(actline.size());
  const char * seq=actline.c_str();
  for(; *seq; ++seq){
    MAF_read_sequence.push_back(*seq);
  }

  FUNCEND();
}

void MAFParse::parseLineRQ(ifstream & mafin, string & acttoken, string & actline)
{
  FUNCSTART("void MAFParse::parseLineRQ(ifstream & mafin, string & acttoken, string & actline)");

  checkParseIsInRead(acttoken);

  if(!MAF_read_qualities.empty()){
    MIRANOTIFY(Notify::FATAL,"Encountered RQ line when there already was one for read " << MAF_read_name);
  }

  mafin >> actline;

  MAF_read_qualities.reserve(actline.size());
  const char * seq=actline.c_str();
  for(; *seq; ++seq){
    MAF_read_qualities.push_back(*seq-33);
  }

  FUNCEND();
}

void MAFParse::parseLineLR(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_len;
  if(MAF_read_len>actline.capacity()) actline.reserve(MAF_read_len+10);
}

void MAFParse::parseLineSV(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_sequencing_vector;
}

void MAFParse::parseLineTN(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_template;
}

void MAFParse::parseLineDI(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_strand_given;
}

void MAFParse::parseLineTF(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_insert_size_min;
}

void MAFParse::parseLineTT(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_insert_size_max;
}

void MAFParse::parseLineSF(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_scf_file;
}

void MAFParse::parseLineBC(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_base_caller;
}

void MAFParse::parseLineSL(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_sl;
  MAF_read_sl--;
}

void MAFParse::parseLineSR(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_sr;
}

void MAFParse::parseLineQL(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_ql;
  MAF_read_ql--;
}

void MAFParse::parseLineQR(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_qr;
}

void MAFParse::parseLineCL(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_cl;
  MAF_read_cl--;
}

void MAFParse::parseLineCR(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_cr;
}

void MAFParse::parseLineAO(ifstream & mafin, string & acttoken, string & actline)
{
  FUNCSTART("void MAFParse::parseLineAO(ifstream & mafin, string & acttoken, string & actline)");

  checkParseIsInRead(acttoken);

  if(MAF_read_sequence.empty()){
    MIRANOTIFY(Notify::FATAL,"While reading AO line for read " << MAF_read_name << ": sequence (SQ line) must be defined before AO line");
  }
  if(MAF_read_align_origin.empty()){
    MAF_read_align_origin.resize(MAF_read_sequence.size(),-1);
  }

  int32 seqfrom, seqto, origfrom, origto;
  mafin >> seqfrom;
  mafin >> seqto;
  mafin >> origfrom;
  mafin >> origto;

  //cout << "xxx " << seqfrom << " " << seqto << " " << origfrom << " " << origto << "\n";

  if(seqfrom<1
     || seqto<1
     || origfrom<1
     || origto<1){
    MIRANOTIFY(Notify::FATAL,"While reading AO line for read " << MAF_read_name << ":  values may not be <1");
  }

  int32 seqinc=1;
  int32 originc=1;
  if(seqto<seqfrom) seqinc=-1;
  if(origto<origfrom) originc=-1;


  if (abs(seqto - seqfrom) != abs(origto - origfrom)) { 
    MIRANOTIFY(Notify::FATAL,"While reading AO line for read " << MAF_read_name << ":  distance between seqfrom/to (" << seqfrom << " " << seqto << ") is unequal to originalfrom/to (" << origfrom << " " << origto << ")");
  }
  if (max(seqfrom, seqto) > static_cast<int32>(MAF_read_align_origin.size())) {
    MIRANOTIFY(Notify::FATAL,"While reading AO line for read " << MAF_read_name << ":  seqfrom/to (" << seqfrom << " " << seqto << ") is larger than size of read (" << MAF_read_align_origin.size() << ")");
  }

  int32 seqi=seqfrom;
  int32 origi=origfrom;
  for(int32 loopi=0; loopi < abs(seqto-seqfrom)+1; loopi++){
    if(seqi-1 <0 || seqi-1 >= MAF_read_align_origin.size()) {
      MIRANOTIFY(Notify::FATAL,"While reading AO line for read: " << MAF_read_name << " with AO values " << seqfrom << " " << seqto << " " <<origfrom << " " << origto << "\nThis makes for an illegal alignment of the sequence to the original, wrong values in this line?\n");
    }

    MAF_read_align_origin[seqi-1] = origi - 1;
    seqi += seqinc;
    origi += originc;
  }

  FUNCEND();
}

void MAFParse::parseLineRT(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);

  MAF_read_taglist.resize(MAF_read_taglist.size()+1);
  parseTagData(mafin,acttoken,MAF_read_taglist.back());

  //cout << MAF_read_taglist.back();
}

void MAFParse::parseTagData(ifstream & mafin, string & acttoken, tag_t & tag)
{
  FUNCSTART("void MAFParse::parseTagData(ifstream & mafin, string & acttoken, tag_t & tag)");
  mafin >> tag.identifier;
  mafin >> tag.from;
  mafin >> tag.to;

  if(tag.from<1){
    MIRANOTIFY(Notify::FATAL, "Error in " << MAF_read_name << " in tag line " << acttoken << ": (" << tag.from << " " << tag.to << ") -> " << tag.from << " is <1, not allowed.");
  }
  if(tag.to<1){
    MIRANOTIFY(Notify::FATAL, "Error in " << MAF_read_name << " in tag line " << acttoken << ": (" << tag.from << " " << tag.to << ") -> " << tag.to << " is <1, not allowed.");
  }

  tag.from-=1;
  tag.to-=1;

  if(tag.from<=tag.to){
    tag.strand='+';
  }else{
    tag.strand='-';
    swap(tag.from, tag.to);
  }

  // comment may be present or not
  char nextchar;
  mafin.get(nextchar);
  if(nextchar=='\n') return;
  if(nextchar=='\r') {
    // also eat \n
    mafin.get(nextchar);
    return;
  }
  getline(mafin,tag.comment);

  FUNCEND();
}

void MAFParse::parseLineST(ifstream & mafin, string & acttoken, string & actline)
{
  FUNCSTART("void MAFParse::parseLineST(ifstream & mafin, string & acttoken, string & actline)");
  checkParseIsInRead(acttoken);
  mafin >> actline;

  MAF_read_seqtype=Read::stringToSeqType(actline);

  if(MAF_read_seqtype==Read::SEQTYPE_END){
    MIRANOTIFY(Notify::FATAL, "Error in " << MAF_read_name << " in tag " << acttoken << ": unkown sequencing type '" << actline << "'?");
  }
  FUNCEND();
}

void MAFParse::parseLineSN(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_strain;
}

void MAFParse::parseLineMT(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_machinetype;
}

void MAFParse::parseLineIB(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_isbackbone;
}

void MAFParse::parseLineIC(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_isCER;
}

void MAFParse::parseLineIR(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  mafin >> MAF_read_israil;
}

void MAFParse::parseLineER(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInRead(acttoken);
  addReadToReadPool();
  MAF_isinread=false;
  if(!MAF_isincontig  && MAF_callbackfunc!=NULL) {
    (*MAF_callbackfunc)(*MAF_contiglist, *MAF_readpool);
  }
}


void MAFParse::parseLineCO(ifstream & mafin, string & acttoken, string & actline)
{
  FUNCSTART("void MAFParse::parseLineCO(ifstream & mafin, string & acttoken, string & actline)");
  if(MAF_isincontig){
    MIRANOTIFY(Notify::FATAL, "Seen new CO line while previous CO was not closed by EC");
  }

  cleanupContigData();
  mafin >> MAF_contig_name;
  MAF_isincontig=true;
  FUNCEND();
}

void MAFParse::parseLineCS(ifstream & mafin, string & acttoken, string & actline)
{
  FUNCSTART("void MAFParse::parseLineSQ(ifstream & mafin, string & acttoken, string & actline)");

  checkParseIsInContig(acttoken);

  if(!MAF_contig_sequence.empty()){
    MIRANOTIFY(Notify::FATAL,"Encountered CS line when there already was one for contig " << MAF_contig_name);
  }

  mafin >> actline;

  MAF_contig_sequence.reserve(actline.size());
  const char * seq=actline.c_str();
  for(; *seq; ++seq){
    MAF_contig_sequence.push_back(*seq);
  }

  FUNCEND();
}

void MAFParse::parseLineCQ(ifstream & mafin, string & acttoken, string & actline)
{
  FUNCSTART("void MAFParse::parseLineCQ(ifstream & mafin, string & acttoken, string & actline)");

  checkParseIsInContig(acttoken);

  if(!MAF_contig_qualities.empty()){
    MIRANOTIFY(Notify::FATAL,"Encountered CQ line when there already was one for contig " << MAF_contig_name);
  }
  
  mafin >> actline;

  MAF_contig_qualities.reserve(actline.size());
  const char * seq=actline.c_str();
  for(; *seq; ++seq){
    MAF_contig_qualities.push_back(*seq-33);
  }

  FUNCEND();
}

void MAFParse::parseLineNR(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInContig(acttoken);
  mafin >> MAF_contig_numreads;
}

void MAFParse::parseLineLC(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInContig(acttoken);
  mafin >> MAF_contig_len;

  if(MAF_contig_len>actline.capacity()) actline.reserve(MAF_contig_len+10);
}

void MAFParse::parseLineCT(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInContig(acttoken);

  MAF_contig_taglist.resize(MAF_contig_taglist.size()+1);
  parseTagData(mafin,acttoken,MAF_contig_taglist.back());
  // for consensus tags, change strand to '='
  MAF_contig_taglist.back().strand='=';

  //cout << MAF_contig_taglist.back();
}

void MAFParse::parseLineAT(ifstream & mafin, string & acttoken, string & actline)
{
  FUNCSTART("void MAFParse::parseLineAT(ifstream & mafin, string & acttoken, string & actline)");

  checkParseIsInContig(acttoken);
  checkParseIsNotInRead(acttoken);

  if(MAF_readpoolid<0){
    MIRANOTIFY(Notify::FATAL, "Seen AT line but no read in contig defined before? (RD/ER block in a CO block)");
  }

  int32 cfrom,cto,rfrom,rto;
  Contig::contig_init_read_t tmpcr;
  int8 direction=1;

  mafin >> cfrom;
  mafin >> cto;
  mafin >> rfrom;
  mafin >> rto;

  if(cfrom > cto){
    direction=-1;
    tmpcr.offset_in_contig=cto-1;
    if (cfrom > MAF_contig_rawlen) {
      MAF_contig_rawlen = cfrom;
    }
  }else{
    tmpcr.offset_in_contig=cfrom-1;
    if (cto > MAF_contig_rawlen) {
      MAF_contig_rawlen = cto;
    }
  }

  if(rfrom>rto){
    tmpcr.read_rclip=rfrom;
    tmpcr.read_lclip=rto-1;
    tmpcr.direction= -direction;
  }else{
    tmpcr.read_rclip=rto;
    tmpcr.read_lclip=rfrom-1;
    tmpcr.direction= direction;
  }

  tmpcr.id=MAF_readpoolid;
  MAF_contig_assembledfrom.push_back(tmpcr);

  FUNCEND();
}

void MAFParse::parseLineEC(ifstream & mafin, string & acttoken, string & actline)
{
  checkParseIsInContig(acttoken);
  MAF_isincontig=false;

  {
    Contig dummy(MAF_miraparams, *MAF_readpool);
    MAF_contiglist->push_back(dummy);
  }


  if(MAF_recalcconsensus){
    string dummy1;
    vector<base_quality_t> dummy2;
    MAF_contiglist->back().initialiseContig(MAF_contig_rawlen, 
					    MAF_contig_assembledfrom, 
					    MAF_contig_taglist,
					    MAF_contig_name,
					    dummy1,dummy2);
  }else{
    string dummy1;
    dummy1.reserve(MAF_contig_sequence.size()+2);
    {
      vector<char>::const_iterator dnaI=MAF_contig_sequence.begin();
      for(; dnaI != MAF_contig_sequence.end(); dnaI++){
	dummy1+=*dnaI;
      }
    }
    MAF_contiglist->back().initialiseContig(MAF_contig_rawlen, 
					    MAF_contig_assembledfrom, 
					    MAF_contig_taglist,
					    MAF_contig_name,
					    dummy1,
					    MAF_contig_qualities);
  }

  if(MAF_callbackfunc!=NULL) {
    (*MAF_callbackfunc)(*MAF_contiglist, *MAF_readpool);
  }
}



void MAFParse::checkReadData()
{
  FUNCSTART("void MAFParse::checkReadData()");

  BUGIFTHROW(MAF_read_seqtype>=Read::SEQTYPE_END, "Illegal seqtype in checkReadData()???");

  if(MAF_read_len>=0
     && MAF_read_sequence.size() != MAF_read_len){
    MIRANOTIFY(Notify::FATAL,"Read " << MAF_read_name << ": size of sequence (" << MAF_read_sequence.size() << ") is not equal to size given in LR line (" << MAF_read_len << ")");
  }
  if(!MAF_read_qualities.empty()){
    if(MAF_read_sequence.size() != MAF_read_qualities.size()){
      MIRANOTIFY(Notify::FATAL,"Read " << MAF_read_name << ": size of sequence (" << MAF_read_sequence.size() << ") is not equal to size of qualities (" << MAF_read_qualities.size() << ")");
    }
  }else{
    // sequence but no qualities ... then fake some according to the sequencing type

    // if miraparams is larger than seqtype ... bad luck, just give a qual of 10
    if(MAF_miraparams->size()>=MAF_read_seqtype){
      MAF_read_qualities.resize(MAF_read_sequence.size(),10);
    }else{
      // everything ok, give them the standard qual for this sequencing type
      MAF_read_qualities.resize(
	MAF_read_sequence.size(),
	(*MAF_miraparams)[MAF_read_seqtype].getAssemblyParams().as_basedefaultqual
	);
    }
  }

  if(MAF_read_align_origin.empty()){
    MAF_read_align_origin.resize(MAF_read_sequence.size());
    for(size_t ii=0; ii<MAF_read_align_origin.size(); ++ii) MAF_read_align_origin[ii]=ii;
  }

  if(MAF_read_align_origin.size() != MAF_read_sequence.size()){
    MIRANOTIFY(Notify::FATAL,"Read " << MAF_read_name << ": the align to origin (AO) data led to a larger or smaller array that the length of the sequence?");
  }

  if(MAF_read_ql < 0) MAF_read_ql = 0;
  if(MAF_read_sl < 0) MAF_read_sl = 0;
  if(MAF_read_cl < 0) MAF_read_cl = 0;

  if(MAF_read_qr < 0) MAF_read_qr = MAF_read_sequence.size();
  if(MAF_read_sr < 0) MAF_read_sr = MAF_read_sequence.size();
  if(MAF_read_cr < 0) MAF_read_cr = MAF_read_sequence.size();

  FUNCEND();
}


void MAFParse::addReadToReadPool()
{
  checkReadData();

  MAF_readpoolid=MAF_readpool->size();
  Read & newread = MAF_readpool->addNewEmptyRead(); 

  newread.initialiseRead(false, 
			 false, 
			 true,     // always padded
			 MAF_read_sequence,
			 MAF_read_qualities,
			 MAF_read_align_origin,
			 MAF_read_taglist,
			 MAF_read_name,
			 MAF_read_scf_file,
			 MAF_read_ql,
			 MAF_read_qr,
			 MAF_read_sl,
			 MAF_read_sr,
			 MAF_read_cl,
			 MAF_read_cr);
  
  
  if (MAF_read_insert_size_max > 0 || MAF_read_insert_size_min > 0) {
    newread.setInsize(MAF_read_insert_size_min, MAF_read_insert_size_max); 
  }

  if(MAF_read_strand_given!='N'){
    newread.setTemplateEnd(MAF_read_strand_given);
  }

  if (!MAF_read_sequencing_vector.empty()) {
    newread.setSeqvecName(MAF_read_sequencing_vector);
  }

  if (!MAF_read_template.empty()) {
    newread.setTemplate(MAF_read_template);
  }
  
  if (!MAF_read_base_caller.empty()) {
    newread.setBasecaller(MAF_read_base_caller);
  }

  if (!MAF_read_strain.empty()) {
    newread.setStrain(MAF_read_strain);
  }

  if (!MAF_read_machinetype.empty()) {
    newread.setMachineType(MAF_read_machinetype);
  }

  newread.setBackbone(MAF_read_isbackbone);
  newread.setRail(MAF_read_israil);
  newread.setCoverageEquivalentRead(MAF_read_isCER);

  newread.setSequencingType(MAF_read_seqtype);
}

