/* ======================================================================
 * utils.cc
 * 
 * This file is part of MeshIO, the general and extensible 3D mesh I/O
 * library.
 * Copyright (c) 1999, 2000 Niklas Elmqvist. All rights reserved.
 *
 * File created 1999-06-02 by Niklas Elmqvist <d97elm@dtek.chalmers.se>.
 * 
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 *
 * ======================================================================
 */

// -- System Includes
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

// -- Local Includes
#include "utils.hh"

// -- Code Segment

void StringReader(FILE *f, char *buf, int len)
{
    int c, i = 0;
    
    // Read individual characters until terminating zero is found
    while ((c = fgetc(f)) != EOF) { // && c != '\0') {
	if (i < len - 1) buf[i] = c;
	if (c == '\0') break;
	i++;
    }
    
    // Terminate string
    buf[MIN(i + 1, len - 1)] = '\0';
}

void StringWriter(FILE *f, char *buf)
{
    if (buf != NULL)
	fprintf(f, "%s", buf);
    fputc('\0', f);
}

bool ReadData(FILE *f, void *val, int nsize, int nelem, bool swap)
{
    char *str = (char *) val;
    short *val16;
    int *val32;
    
    // Repeat for each element
    while (nelem--) {
	
	// Read the value
	if (fread(str, nsize, 1, f) != 1) return false;
	
	// Do we want to byte-swap?
	if (swap == false)
	    continue;
	
	// Assign pointers
	val16 = (short *) str;
	val32 = (int *) str;
	
	// Okay, then swap the bytes
	switch (nsize) {
	case 1: break;
	case 2: *val16 = SWAP16(*val16); break;
	case 4: *val32 = SWAP32(*val32); break;
	default: break;
	}
	
	// Move on to the next item
	str += nsize;
    }
    
    // Return with success
    return true;
}

bool WriteData(FILE *f, void *val, int nsize, int nelem, bool swap)
{
    char *ptr = (char *) malloc(nsize * nelem);
    char *str = ptr;
    short *val16;    
    int *val32;
    
    // Copy data 
    memcpy(str, val, nsize * nelem);
    
    // Repeat for each element
    while (nelem--) {

	// Assign pointers
	val16 = (short *) str;
	val32 = (int *) str;
	
	// Do we need to swap?
	if (swap == true) {
	    
	    // Okay, then swap the bytes
	    switch (nsize) {
	    case 1: break;
	    case 2: *val16 = SWAP16(*val16); break;
	    case 4: *val32 = SWAP32(*val32); break;
	    default: break;
	    }
	}
	
	// Write the value
	if (fwrite(str, nsize, 1, f) != 1) return false;
	
	// Move on to the next item
	str += nsize;
    }

    // Free memory
    free(ptr);
    
    // Return with success
    return true;
}

ListNode *AddNode(ListNode *list, void *data)
{
    // Create and set up the new node
    ListNode *node = (ListNode *) malloc(sizeof(ListNode));
    node->data     = data;
    node->next     = list;

    // Return with the new list head
    return node;
}

ListNode *DelNode(ListNode *list, void *data)
{
    ListNode *node = list;
    
    // Special case for the head of the list
    if (list->data != data) {	
	list = list->next;
    }
    else {
	
	// Search for the desired node
	for ( ; node != NULL; node = node->next)
	    if (node->data == data)
		break;
	
	// Make sure we found it
	if (node == NULL)
	    return list;
    }
    
    // Delete the list node
    free(node);
    
    // Return with list head
    return list;
}

void ClearList(ListNode *list)
{
    // Clear the list
    while (list != NULL) {
	ListNode *temp = list;
	list = list->next;
	free(temp);
    }
}

ListNode *ReverseList(ListNode *node)
{
    ListNode *temp;
    
    // Base case
    if (node == NULL)
	return NULL;
    
    // Call ourselves recursively
    temp = ReverseList(node->next);
    
    // Do we need to change the pointer?
    if (node->next == NULL) 
	return node;
    else {
	node->next->next = node;
	node->next = NULL;
	return temp;
    }
}
