/* sel-onetarget.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "sel-onetarget.h"
#include <libgnomedb/libgnomedb.h>
#include "mg-query.h"
#include "mg-entity.h"
#include "mg-qfield.h"
#include "mg-field.h"
#include "mg-renderer.h"
#include "mg-server-data-type.h"
#include "mg-qf-all.h"
#include "mg-qf-field.h"
#include "mg-qf-value.h"
#include "mg-target.h"

/*
 *
 * Modules for the fields, targets, joins, etc of a given MgQuery
 *
 */
typedef struct {
	ModFlatData  data;
	GdkPixbuf   *field_pixbuf;
	MgTarget    *target;
	gboolean    *manager_is_represented_entity;
} ModOnetargetData;
#define TARGET_DATA(x) ((ModOnetargetData *)(x->mod_data))

static void         module_onetarget_fill_model (Module *module);
static void         module_onetarget_free (Module *module);
static const gchar *module_onetarget_col_name (Module *module, guint colno);
static Module      *module_onetarget_obj_manager (Module *module, GtkTreeIter *iter, GObject *object);
static void         module_onetarget_model_store_data (Module *module, GtkTreeIter *iter);
static GSList      *module_onetarget_get_objects_list (Module *module);
static const gchar *module_onetarget_render_qfield_type (MgQfield *field);


static Module *real_sel_module_onetarget_new (MgSelector *mgsel, gboolean insert_header, 
					      GtkTreeIter *iter, gpointer data, gboolean all);

Module *
sel_module_onetarget_new (MgSelector *mgsel, gboolean insert_header, 
			      GtkTreeIter *iter, gpointer data)
{
	return real_sel_module_onetarget_new (mgsel, insert_header, iter, data, FALSE);
}

Module *
sel_module_onetarget_new_all (MgSelector *mgsel, gboolean insert_header, 
			      GtkTreeIter *iter, gpointer data)
{
	return real_sel_module_onetarget_new (mgsel, insert_header, iter, data, TRUE);
}

static Module *
real_sel_module_onetarget_new (MgSelector *mgsel, gboolean insert_header, 
			       GtkTreeIter *iter, gpointer data, gboolean all)
{
	Module *module;
	MgTarget *target;
	GdkPixbuf *pixbuf_field = NULL;
	MgQuery *query;

	g_assert (data && IS_MG_TARGET (data));
	pixbuf_field = gnome_db_stock_get_icon_pixbuf_file ("gnome-db-field_16x16.png");

	/* module structure */
	target = MG_TARGET (data);
	query = mg_target_get_query (target);
	module = g_new0 (Module, 1);
	module->selector = mgsel;
	module->fill_model = module_onetarget_fill_model;
	module->free = module_onetarget_free;
	module->col_name = module_onetarget_col_name;
	module->obj_manager = NULL;
	module->model_store_data = module_onetarget_model_store_data;
	module->mod_data = NULL;
	module->iter = NULL;
	module->parent_module = NULL;
	module->sub_modules = NULL;

	/* Module's private data */
	module->mod_data = g_new0 (ModOnetargetData, 1);
	if (!all) {
		FLAT_DATA (module)->manager = G_OBJECT (query);
		TARGET_DATA (module)->manager_is_represented_entity = FALSE;
	}
	else {
		FLAT_DATA (module)->manager = mg_target_get_represented_entity (target);
		TARGET_DATA (module)->manager_is_represented_entity = TRUE;
	}
	FLAT_DATA (module)->manager_weak_refed = FALSE;
	FLAT_DATA (module)->pixbuf_hash = NULL;
	g_object_ref (pixbuf_field);
	FLAT_DATA (module)->fallback_obj_pixbuf = pixbuf_field;
	FLAT_DATA (module)->get_objects_list = module_onetarget_get_objects_list;
	TARGET_DATA (module)->field_pixbuf = pixbuf_field;
	TARGET_DATA (module)->target = target;

	/* model settings */
	if (insert_header) {
		GdkPixbuf *pixbuf = NULL;
		GtkTreeModel *model = mgsel->priv->model;

		module->iter = g_new0 (GtkTreeIter, 1);
		gtk_tree_store_append (GTK_TREE_STORE (model), module->iter, iter);
		gtk_tree_store_set (GTK_TREE_STORE (model), module->iter, 
				    NAME_COLUMN, mg_base_get_name (MG_BASE (target)), 
				    PIXBUF_COLUMN, pixbuf, 
				    CONTENTS_COLUMN, CONTENTS_TOP_CATEGORY, 
				    SUB_MODULE_COLUMN, NULL, -1);
	}
	else {
		if (iter)
			module->iter = gtk_tree_iter_copy (iter);
	}

	return module;	
}

static void
module_onetarget_fill_model (Module *module)
{
	GObject *manager;
	GtkTreeModel *model;
	
	manager = FLAT_DATA (module)->manager;

	/* Signals handlers */
	g_signal_connect (manager, "field_added",
			  G_CALLBACK (flat_obj_added_cb), module);
	g_signal_connect (manager, "field_removed",
			  G_CALLBACK (flat_obj_removed_cb), module);
	g_signal_connect (manager, "field_updated",
			  G_CALLBACK (flat_obj_updated_cb), module);

	/* Initial model filling */
	model = module->selector->priv->model;
	flat_init_model_fill (module, model);
}

static GSList *
module_onetarget_get_objects_list (Module *module)
{
	GSList *retval = NULL;

	if (TARGET_DATA (module)->manager_is_represented_entity) 
		retval = mg_entity_get_visible_fields (MG_ENTITY (FLAT_DATA (module)->manager));
	else {
		if ((module->selector->priv->mode & MG_SELECTOR_QVIS_FIELDS) ||
		    (module->selector->priv->mode & MG_SELECTOR_QALL_FIELDS)) {
			MgQuery *query;
			
			query = mg_target_get_query (TARGET_DATA (module)->target);
			retval = mg_query_get_fields_by_target (query, TARGET_DATA (module)->target, 
								module->selector->priv->mode & MG_SELECTOR_QVIS_FIELDS);
		}
	}

	return retval;
}

static void
module_onetarget_free (Module *module)
{
	GObject *manager = FLAT_DATA (module)->manager;
	GSList *list = module->sub_modules;

	g_assert (manager);

	/* free the extra pixbufs */
	if (TARGET_DATA (module)->field_pixbuf)
		g_object_unref (G_OBJECT (TARGET_DATA (module)->field_pixbuf));

	/* Free the sub modules */
	while (list) {
		(MODULE (list->data)->free) (MODULE (list->data));
		g_free (list->data);
		list = g_slist_next (list);
	}
	if (module->sub_modules) {
		g_slist_free (module->sub_modules);
		module->sub_modules = NULL;
	}

	/* free this module */
	g_signal_handlers_disconnect_by_func (manager,
					      G_CALLBACK (flat_obj_added_cb), module);
	g_signal_handlers_disconnect_by_func (manager,
					      G_CALLBACK (flat_obj_removed_cb), module);
	g_signal_handlers_disconnect_by_func (manager,
					      G_CALLBACK (flat_obj_updated_cb), module);

	if (module->iter)
		gtk_tree_iter_free (module->iter);
	flat_free_mod_data (module);
	g_free (module->mod_data);
	module->mod_data = NULL;
}


static const gchar *
module_onetarget_col_name (Module *module, guint colno)
{
	switch (colno) {
	case 0:
		return _("Field");
		break;
	case EXTRA1_COLUMN:
		return _("Type");
		break;
	default:
		return NULL;
		break;
	}
}

static void
module_onetarget_model_store_data (Module *module, GtkTreeIter *iter)
{
	GObject *obj;
	GtkTreeModel *model;

	model = module->selector->priv->model;
	gtk_tree_model_get (model, iter, OBJ_COLUMN, &obj, -1);

	if (obj && IS_MG_QFIELD (obj)) {
		MgQfield *field;
		const gchar *str1, *str2;
		gchar *str3;
		MgServerDataType *type;
		
		field = MG_QFIELD (obj);

		/* data type */
		type = mg_field_get_data_type (MG_FIELD (obj));
		if (type)
			str1 = mg_server_data_type_get_sqlname (type);
		else
			str1 = _("-");
		/* other */
		str2 = module_onetarget_render_qfield_type (MG_QFIELD (obj));
		str3 = mg_renderer_render_as_str (MG_RENDERER (obj), NULL);
		
		gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
				    EXTRA1_COLUMN, str1,
				    EXTRA6_COLUMN, str3,
				    EXTRA7_COLUMN, str2,
				    PIXBUF_COLUMN, TARGET_DATA (module)->field_pixbuf,
				    -1);
		if (str3)
			g_free (str3);

		str1 = mg_base_get_name (MG_BASE (field));
		if (!str1 || !(*str1)) {
			gtk_tree_store_set (GTK_TREE_STORE (model), iter, 
				    NAME_COLUMN, _("Field <no name>"),
				    -1);
		}
	}
}

static const gchar *
module_onetarget_render_qfield_type (MgQfield *field)
{
	GType ftype = G_OBJECT_TYPE (field);

	if (ftype == MG_QF_ALL_TYPE)
		return _("entity.*");
	if (ftype == MG_QF_FIELD_TYPE)
		return _("entity.field");
	if (ftype == MG_QF_VALUE_TYPE) {
		if (mg_qf_value_is_parameter (MG_QF_VALUE (field)))
			return _("parameter");
		else
			return _("value");
	}

	/* for other types */
	TO_IMPLEMENT;
	return "???";
}
