/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* This module defines the data type "value_t", and many
 * operations to build, modify, and print such values.
 * There are six different types of values:
 * symbol, string, list, record, number and index. */

/* constants ================================================================*/

/* some standard symbols */
enum {NIL_SYMBOL, YES_SYMBOL, NO_SYMBOL, 
      SYMBOL_SYMBOL, STRING_SYMBOL, NUMBER_SYMBOL, LIST_SYMBOL, RECORD_SYMBOL};

#define SYMBOL_MAX 8192 /* symbols must be in the range of 0..SYMBOL_MAX-1 */

/* types ====================================================================*/

typedef u_short_t cell_t;
/* A value is stored in one or more cells.
 * Use this type if you want to allocate memory (pools etc.) for values. */ 

typedef cell_t *value_t; /* Reference to a Malaga values by this type. */
typedef cell_t symbol_t;

/* variables ================================================================*/

GLOBAL string_t (*get_symbol_name) (symbol_t symbol);
/* Return the name of "symbol".
 * This is a callback function that must be set externally. */

GLOBAL value_t (*get_atoms) (symbol_t symbol);
/* Return the list of atoms of <symbol>.
 * This is a callback function that must be set externally. */

GLOBAL value_t *value_stack;
/* The value stack contains static values and local values. 
 * The location of the <value_stack>-vector may change when the stack size
 * must increase. */

GLOBAL int_t top; 
/* The index of the first unused item on <value_stack>.
 * You may ONLY READ OR DECREASE this variable. */

GLOBAL enum {INTERNAL_ORDER, 
	     ALPHABETIC_ORDER, DEFINITION_ORDER} attribute_order;
/* The order in which attributes in a record are printed.
 * INTERNAL_ORDER is the order in which attributes are stored internally.
 * ALPHABETIC_ORDER means the alphabetic order of the attribute names.
 * DEFINITION_ORDER is the order in which the attributes are defined in the 
 * symbol file.
 * Used by "value_readable". */

/* module initialisation. ===================================================*/

extern void init_values (void);
/* Initialise this module. */

extern void terminate_values (void);
/* Terminate this module. */

/* value operations =========================================================*/

extern value_t new_value (value_t value);
/* Allocate space for <value> and copy it.
 * Use "free" to free the space occupied by this value. */

extern value_t copy_value_to_pool (pool_t value_pool, 
				   value_t value, 
				   int_t *index);
/* Copy <value> to the pool <value_pool> and store its index in *<index>. */

extern int_t length_of_value (value_t value);
/* Return the length of <value> in cells. */

extern symbol_t get_value_type (value_t value);
/* Return the type of <value>. Depending of the type, the result value may be
 * SYMBOL_SYMBOL, STRING_SYMBOL, NUMBER_SYMBOL, LIST_SYMBOL, RECORD_SYMBOL. */

extern void push_value (value_t value);
/* STACK EFFECTS: (nothing) -> <value>. */

extern void insert_value (int_t n, value_t value);
/* STACK EFFECTS: <value_1>...<value_n> -> <value> <value_1>...<value_n>. */

/* symbol operations ========================================================*/

extern symbol_t value_to_symbol (value_t value);
/* Return <value> as a symbol. It is an error if <value> is no symbol. */

extern void push_symbol_value (symbol_t symbol);
/* STACK EFFECTS: (nothing) -> <new_symbol>.
 * <new_symbol> is <symbol> converted to a Malaga value. */

/* string operations ========================================================*/

extern string_t value_to_string (value_t string);
/* Return the value of <string> as a C style string. */

extern void push_string_value (string_t string_start, string_t string_end);
/* STACK EFFECTS: (nothing) -> <new_string>.
 * <new_string> is the string starting at <string_start> as a Malaga value.
 * If <string_end> != NULL, it marks the end of the string. */

extern void concat_string_values (void);
/* STACK EFFECTS: <string_1> <string_2> -> <new_string>.
 * <new_string> is the concatenation of <string_1> and <string_2>. */

/* record operations ========================================================*/

extern value_t get_attribute (value_t record, symbol_t attribute);
/* Return the value of <attribute> in the record <record> 
 * or NULL if it doesn't exist. */

extern void build_record (int_t n);
/* STACK EFFECTS: <attr_1> <value_1> ... <attr_n> <value_n> -> <new_record>.
 * <new_record> looks like [<attr_1>: <value_1>, ..., <attr_n>: <value_n>]. */

extern void join_records (void);
/* STACK EFFECTS: <record_1> <record_2> -> <new_record>.
 * <new_record> contains all attributes of <record_1> and <record_2>, and 
 * their associated values. If an attribute has different values in <record_1>
 * and <record_2>, the value in <record_2> will be taken. */

extern void select_attribute (symbol_t attribute);
/* STACK EFFECTS: <record> -> <new_record>.
 * <new_record> contains <attribute> and its value in <record>. */

extern void select_attributes (void);
/* STACK EFFECTS: <record> <list> -> <new_record>.
 * <new_record> contains all attribute-value pairs of <record> whose attributes
 * are in <list>. */

extern void remove_attribute (symbol_t attribute);
/* STACK EFFECTS: <record> -> <new_record>.
 * <new_record> contains all attribute-value pairs of <record> but the one with
 * attribute <attribute>. */

extern void remove_attributes (void);
/* STACK EFFECTS: <record> <list> -> <new_record>.
 * <new_record> contains all attribute-value pairs of <record> but the ones
 * whose attributes are in <list>. */

extern void replace_attribute (symbol_t attribute);
/* STACK EFFECTS: <record> <value> -> <new_record>.
 * <new_record> is equal to <record>, only the value of <attribute> is replaced
 * by <value>. <record> must contain <attribute>. */

/* list operations ==========================================================*/

extern int_t get_list_length (value_t list);
/* Return the number of elements in the list <list>. */

extern value_t get_element (value_t list, int_t n);
/* Return the <n>-th element of the list <list>,
 * or NULL, if that element doesn't exist.
 * If <n> is positive, elements will be counted from the left border.
 * If it's negative, elements will be counted from the right border. */

extern void build_list (int_t n);
/* STACK EFFECTS: <value_1> ... <value_n> -> <new_list>.
 * <new_list> looks like < <value_1>, ..., <value_n> >. */

extern void concat_lists (void);
/* STACK EFFECTS: <list_1> <list_2> -> <new_list>.
 * <new_list> is the concatenation of <list_1> and <list_2>. */

extern void get_list_difference (void);
/* STACK EFFECTS: <list_1> <list_2> -> <new_list>.
 * <new_list> contains the list difference of <list_1> and <list_2>:
 * An element that appears <m> times in <list_1> and <n> times in <list_2> 
 * appears <m-n> times in <new_list>. */

extern void get_set_difference (void);
/* STACK EFFECTS: <list_1> <list_2> -> <new_list>.
 * <new_list> contains the set difference of <list_1> and <list_2>.
 * Each element of <list_1> is in <new_list> if it is not in <list2>. */

extern void intersect_lists (void);
/* STACK EFFECTS: <list_1> <list_2> -> <new_list>.
 * <new_list contains the list intersection of <list_1> and <list_2>.
 * Each element that appears <m> times in <list_1> and <n> times in <list_2>
 * appears min(<m>, <n>) times in <new_list>. */

extern void remove_element (int_t n);
/* STACK EFFECTS: <list> -> <new_list>.
 * <new_list> is <list> without element at index <n>.
 * If <n> is positive, the elements will be counted from the left border;
 * if <n> is negative, they will be counted from the right border.
 * If <list> contains less than abs(<n>) elements, then <new_list> = <list>. */

extern void remove_elements (int_t n);
/* STACK EFFECTS: <list> -> <new_list>.
 * <new_list> is <list> without abs(<n>) elements.
 * If <n> is positive, the elements will be cut from the left border,
 * if <n> is negative, they will be cut from the list's right border.
 * If <list> contains less than abs(<n>) elements, then <new_list> = <>. */

extern void replace_element (int_t n);
/* STACK EFFECTS: <list> <value> -> <new_list>.
 * <new_list> is <list>, but its <n>-th element is replaced by <value>.
 * If <n> is negative, count from the right end.
 * <list> must contain at least <n> elements. */

extern void convert_list_to_set (void);
/* STACK EFFECTS: <list> -> <new_list>.
 * <new_list> contains all elements of <list>, but multiple appearances
 * of one value are reduced to a single appearance.
 * That means, <new_list> is <list> converted to a set. */

/* number operations ========================================================*/

extern double value_to_double (value_t value);
/* Return the value of <value> which must be a number value. */

extern int_t value_to_int (value_t value);
/* Return the value of <value> which must be an integral number value. */

extern void push_number_value (double number);
/* STACK EFFECTS: (nothing) -> <new_number>.
 * <new_number> is <number> as a Malaga value. */

/* type dependent Malaga operations =========================================*/

extern void dot_operation (void);
/* STACK EFFECTS: <value_1> <value_2> -> <new_value>.
 * <new_value> is <value_1> "." <value_2>. 
 * The actual operation depends on the type of the values. */

extern void plus_operation (void);
/* STACK EFFECTS: <value_1> <value_2> -> <new_value>.
 * <new_value> is <value_1> "+" <value_2>. 
 * The actual operation depends on the type of the values. */

extern void minus_operation (void);
/* STACK EFFECTS: <value_1> <value_2> -> <new_value>.
 * <new_value> is <value_1> "-" <value_2>. 
 * The actual operation depends on the type of the values. */

extern void asterisk_operation (void);
/* STACK EFFECTS: <value_1> <value_2> -> <new_value>.
 * <new_value> is <value_1> "*" <value_2>. 
 * The actual operation depends on the type of the values. */

extern void slash_operation (void);
/* STACK EFFECTS: <value_1> <value_2> -> <new_value>.
 * <new_value> is <value_1> "/" <value_2>. 
 * The actual operation depends on the type of the values. */

extern void unary_minus_operation (void);
/* STACK EFFECTS: <value> -> <new_value>.
 * <new_value> is "-" <value>.
 * The actual operation depends on the type of the value. */

/* functions for value paths ================================================*/

extern value_t get_value_part (value_t value, value_t path);
/* Return the value part of <value> that is specified by the path
 * <path>. If that value part doesn't exist, return NULL. */

extern void build_path (int_t n);
/* STACK EFFECTS: <value_1> ... <value_n> -> <new_list>.
 * <new_list> is a path which contains <value_1>, ..., <value_n>. 
 * <value_1>, ..., <value_n> must be numbers, symbols or lists of numbers and 
 * symbols. If a value is a list, the elements of this list are inserted into
 * <new_list> instead of the value itself. */

extern void modify_value_part (void (*modifier) (void));
/* STACK EFFECTS: <value> <path> <mod_value> -> <new_value>.
 * <new_value> is <value>, but the part that is described by <path> is 
 * modified. <path> must be a list of symbols and numbers <e1, e2, .. , en>.
 * They will be used as nested attributes and indexes, so the part of <value>
 * that is actually modified is <old_value> := <value>.<e1>.<e2>..<en>. 
 * If this part does not exist, an error will be reported. Else the function 
 * <modifier> will be called on <old_value> and <mod_value>. 
 * The value returned by <modifier> will be entered in <value> in place of
 * <old_value>. */

extern void right_value (void);
/* STACK EFFECTS: <left_value> <right_value> -> <right_value>.
 * A modifier for "modify_value_part". */

/* functions for list/record iteration ======================================*/

extern void get_first_element (void);
/* STACK EFFECTS: <value> -> <new_value>.
 * If <value> is a list, then <new_value> is its first element (or NULL).
 * If <value> is a record, then <new_value> is its first attribute (or NULL).
 * If <value> is a number, then <new_value> is NULL (if <value> == 0),
 * 1 (if <value> > 0) or -1 (if <value> < 0). */

extern void get_next_element (int_t stack_index);
/* STACK EFFECTS: (nothing) -> (nothing).
 * <value_1> is <value_stack>[<index>-1], <value_2> is <value_stack>[<index>].
 * <value_stack>[<index>] will be set to <new_value>.
 * <value_2> must be the result of an application of "get_first_element" or 
 * "get_next_element" on <value_1>.
 * If <value_1> is a list, and <value_2> one of its elements,
 * then <new_value> is the successor of <value_2> (or NULL).
 * If <value_1> is a record, and <value_2> one of its attributes,
 * then <new_value> is the next attribute in <value_1> (or NULL).
 * If <value_1> is a positive number, and <value_2> a number smaller than
 * <value_1>, then <new_value> is <value_2> + 1.
 * If <value_1> is a negative number, and <value_2> a number greater than
 * <value_1>, then <new_value> is <value_2> - 1. */

/* functions to compare values ==============================================*/

extern int_t compare_atom_lists (value_t atoms1, value_t atoms2);
/* Compare atom lists <atoms1> and <atoms2>.
 * Return -1 if <atoms1> < <atoms2>.
 * Return 0 if <atoms1> == <atoms2>.
 * Return 1 if <atoms1> > <atoms2>. */

extern bool_t values_equal (value_t value1, value_t value2);
/* Return a truth value indicating whether <value1> and <value2> are equal.
 * <value1> an <value2> must be of same type.
 * Refer to documentation to see what "equal" in Malaga really means. */

extern bool_t values_congruent (value_t value1, value_t value2);
/* Return a truth value indicating whether <value1> and <value2> have
 * at least one element in common.
 * <value1> and <value2> must both be symbols or lists. */

extern bool_t value_in_value (value_t value1, value_t value2);
/* Return bool value saying if <value1> is element or attribute of <value2>.
 * <value2> must be a list or a record.
 * If <value2> is a record, then <value1> must be a symbol. */

/* functions to convert values to text ======================================*/

extern symbol_t *get_hidden_attributes (void);
/* Get a SYMBOL_MAX-terminated vector of the currently hidden attributes. 
 * The vector must be freed after use. */

extern void add_hidden_attribute (symbol_t attribute);
/* Add <attribute> to the list of currently hidden attributes. */

extern void remove_hidden_attribute (symbol_t attribute);
/* Remove <attribute> from the list of currently hidden attributes. */

extern void clear_hidden_attributes (void);
/* Clear the list of currently hidden attributes. */

extern string_t value_to_readable (value_t value, bool_t full_value);
/* Return <value> in a format readable for humans. 
 * If <full_value> == TRUE, show all attributes, even those that are hidden.
 * The result must be freed with "free" after use. */

/* end of file ==============================================================*/
