/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeViewNode.h
    This file provides TerraLib's tree structure of views
*/
#ifndef  __TERRALIB_INTERNAL_VIEWNODE_H
#define  __TERRALIB_INTERNAL_VIEWNODE_H

#define THEME_GROUP

#include <iostream>
#include <string>
#include <vector>
using namespace std;

//! Type of node
enum TeViewNodeType
{ TeTHEME, TeTREE }; 

//! A class to deal with nodes of a view tree structure
class TeViewNode
{
public:
	//! Constructor
    TeViewNode( const string& name, TeViewNode* parent = 0, int view=0, int id=0, TeViewNodeType type= TeTHEME)
		: myParent_(parent), 
		  id_(id), 
		  name_(name), 
		  view_(view), 
		  type_(type),
		  priority_(0)
	{}

	//! Destructor
	virtual ~TeViewNode()
	{}

	//! Inserts a new child in the tree hierachy. Fails if the object is a leaf
	virtual void add (TeViewNode*) {}


    //!	Remove an existing child identified by its id from the tree hierarchy
    /*!	
		Fails if the object is a leaf
	    Should be used in conjunction with a delete 
	*/	
	virtual TeViewNode* removeID (int /* id */) { return 0; }
 
   //!	Remove an existing child identified by its name from the tree hierarchy
    /*!	
		Fails if the object is a leaf
	    Should be used in conjunction with a delete 
	*/	
	virtual TeViewNode* remove (const string& /*name*/) { return 0; }

	//! Retrieve a node identified by its identifier from the tree structure
	virtual TeViewNode* retrieve (int)  { return 0; }

	//! Returns a pointer to a parent node
	TeViewNode* parent()
	{ return myParent_; }

	//! Returns the identifier of a node parent
	int parentId()
	{
		if (myParent_)
			return myParent_->id(); 
		else
			return 0;
	}

	//! Sets the identification of the parent node
	void parentId(int i) 
	{
		if (myParent_)
			myParent_->id(i);
	}

	//! Sets the parent of node
	void setParent ( TeViewNode* node )
	{ myParent_ = node; }

	//! Returns the identification of a node
	int		id () { return id_; }
	//! Sets the identification of a node
	void	id (int i){ id_ = i; }

	//! Returns the name of a node
	string	name () { return name_; }
	//! Sets the name of a node
	void	name (const string& s) { name_ = s; }

	//! Returns the priority of a node
	int		priority () { return priority_; }
	//! Sets the priority of a node
	void	priority (int i) { priority_ = i; }

	//! Sets the view identification of a node
	void	view (int view) { view_ = view; }
	//! Returns the view identification of a node
	int		view () { return view_ ; }

	//! Returns the node type
	TeViewNodeType type() { return type_; };
	//! Sets the node type
	void type(TeViewNodeType t) { type_ = t; }

	//! Moves a node up in the tree structure
	virtual void moveUp ()
	{
		if (myParent_)
			myParent_->moveUp (this);
	}
	
	//! Moves a node down in the tree structure
	virtual void moveDown ()
	{
		if (myParent_)
			myParent_->moveDown (this);
	}

	//! Swaps nodes 
	virtual void swap ( unsigned int, unsigned int ) {}

	//! Draws a node
	virtual void draw() {}


protected:

	virtual void	moveUp (TeViewNode*) {}
	virtual void	moveDown (TeViewNode*) {}

	TeViewNode* 	myParent_;
	int				id_;	// Node id
	string			name_;	// Node name
	int				view_;	// view id

	TeViewNodeType type_;

// Precedence when stored in a view
	int			priority_;
};


//! A class to deal with branchs in a view tree structure  
class TeViewTree: public TeViewNode
{
public:

	//! Constructor
	TeViewTree(const string& name=""): TeViewNode(name)
	{
		type_ = TeTREE;
	}

	//! Destructor
	~TeViewTree ()
	{
	    // For all my children, delete them
        vector<TeViewNode*>::iterator child = nodes_.begin();
		unsigned int size = nodes_.size();
	    for (unsigned int n=0; n<size; n++ )
	    {
			// Create a temporary pointer to store the child's
			// address
			TeViewNode* pt = (*child);
            // Remove the child from the list

			nodes_.erase(child);

			// Ok, now the child's parent points to NULL and we can
			// safely call the child's destructor
			delete pt;
	    }
	}

	//! Swap nodes
	virtual void swap (unsigned int i, unsigned int j)
	{
		if (i==j || i>=nodes_.size() || j>=nodes_.size())
			return;
		TeViewNode* temp = nodes_[i];
		nodes_[i] = nodes_[j];
		nodes_[j] = temp;
		nodes_[i]->priority (i);
		nodes_[j]->priority (j);
	}

	//! Moves a node up in the tree structure
	virtual void moveUp (TeViewNode* node)
	{
		for (unsigned int i=0 ; i<nodes_.size() ; i++)
		{
			if (node == nodes_[i])
			{
				swap (i, i-1);
				return;
			}
		}
	}

	//! Moves a node down in the tree structure
	virtual void moveDown (TeViewNode* node)
	{
		for (unsigned int i=0 ; i<nodes_.size() ; i++)
		{
			if (node == nodes_[i])
			{
				swap (i, i+1);
				return;
			}
		}
	}

	//! Adds a node to the structure
	virtual void add (TeViewNode* node)
	{
		node->setParent ( this ); // I am the father
		node->priority(static_cast<int>(nodes_.size()));
		nodes_.push_back (node);
	}

    //!	Removes a node identified by its name
	virtual TeViewNode* remove (const string& name)
	{
		vector<TeViewNode*>::iterator child = nodes_.begin();
		while ( child != nodes_.end() )
		{
			TeViewNode* pt = (*child);
			if (pt->name() == name)
			{
				nodes_.erase(child);
				return pt;
			}
			if(pt->type() == TeTREE)
			{
				TeViewNode* result = pt->remove(name);
				if(result) return result;
			}
			++child;
		}
		return 0;
	}

    //!	Removes a node through its identifier
	virtual TeViewNode* removeID (int id) 
	{ 
		for (vector<TeViewNode*>::iterator child = nodes_.begin();
			 child != nodes_.end(); ++child)
		{
			TeViewNode* pt = (*child);
			if(pt->id() == id)
			{
				nodes_.erase(child);
				return pt;
			}
			if(pt->type() == TeTREE)
			{
				TeViewNode* result = pt->removeID(id);
				if(result) return result;
			}
		}
		return 0;
	}

    //!	Retrieves a node through its index
	virtual TeViewNode* retrieve (unsigned int i) 
	{ return nodes_[i]; }

	//! Draw a node
	virtual void draw ()
	{				  
		vector<TeViewNode*>::iterator child = nodes_.begin();
	    while ( child != nodes_.end() )
		{
			TeViewNode* pt = (*child);
			pt->draw();
			child++;
		}
	}

	//! Returns the size of the structure
	unsigned int size()
    { return nodes_.size(); }

	//! Returns the vector of View nodes
	vector<TeViewNode*>& nodes()
	{ return nodes_; }

	//! Sort the vector of nodes according to their priorities
	void sort()
	{
		unsigned int i, j;
		for (i = 0; i < size() - 1; ++i)
		{
			for (j = i+1; j < size(); ++j)
			{
				if (nodes_[i]->priority() > nodes_[j]->priority())
				{
					TeViewNode* temp = nodes_[i];
					nodes_[i] = nodes_[j];
					nodes_[j] = temp;
				}
			}
		}
	}

private:
	vector<TeViewNode*> nodes_;
};

#endif



