/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#ifndef  __TERRALIB_INTERNAL_QUERIER_PARAMS_H
#define  __TERRALIB_INTERNAL_QUERIER_PARAMS_H

#include "TeTheme.h"


//! class to handle the parameters that define a querier 
class TeQuerierParams
{

private:
	
	/* @name Params used to fill STOs */
	//@{
	bool					loadGeometries_;  
	bool					loadAllAttributes_;
	vector<string>			loadAttrNames_;
	vector<string>			loadAttrNamesSeted_;
	TeGroupingAttr			groupingAttr_;
	//@}

	bool					themeFlag_;	//! if the theme is a temporary theme
	TeTheme*				theme_;		//! pointer to a theme
	TeLayer*				layer_;		//! pointer to a layer
	string					fileName_;	//! file name (.shp, etc)
	
	TeChronon				chr_;		//! chronon used to generate the slides 
	string					objId_;		//! object identifier
	
	//TeStatisticType		tsOper_;		//operator used in the construction of temporal series
	
	string					strategy_;		//! strategy name

	/* @name Restrictions */
	//@{
	TeSelectedObjects		selectedObjs_;  
	
	int						spatialRelation_; 
	bool					hasSpatialRes_;
	TeBox					boxRest_;		//box which defines the spatial restriction 
	TeGeometry*				geomRest_;		//geometry which defines the spatial restriction 
	TeGeomRep				geomRepRest_;	//geometry representation of the theme which will be 
											//considered in the spatial restriction  
	//@}

public:

	//! Constructor
	TeQuerierParams(bool loadGeom = false, bool loadAllAttr = true): 
			loadGeometries_(loadGeom),
			loadAllAttributes_(loadAllAttr),
			themeFlag_(false),
			theme_ (0),
			layer_ (0),
			fileName_(""),
			chr_ (TeNOCHRONON),
			objId_(""),
			selectedObjs_(TeAll),
			spatialRelation_(0),
			hasSpatialRes_(false),
			geomRest_(0)
			{ }

	//! Constructor
	TeQuerierParams(bool loadGeom, const vector<string>& loadAttr): 
			loadGeometries_(loadGeom),
			loadAllAttributes_(false),
			themeFlag_(false),
			theme_ (0),
			layer_ (0),
			fileName_(""),
			chr_ (TeNOCHRONON),
			objId_(""),
			selectedObjs_(TeAll),
			spatialRelation_(0),
			hasSpatialRes_(false),
			geomRest_(0)
			{
				loadAttrNamesSeted_.clear();
				loadAttrNamesSeted_ = loadAttr;
			}


	//! Constructor
	TeQuerierParams(bool loadGeom, TeGroupingAttr& groupAttr): 
			loadGeometries_(loadGeom),
			loadAllAttributes_(false),
			themeFlag_(false),
			theme_ (0),
			layer_ (0),
			fileName_(""),
			chr_ (TeNOCHRONON),
			objId_(""),
			selectedObjs_(TeAll),
			spatialRelation_(0),
			hasSpatialRes_(false),
			geomRest_(0)
			{
				groupingAttr_.clear();
				groupingAttr_ = groupAttr;
			}
	
	//! Destructor
	~TeQuerierParams();

	//! Copy Constructor
	TeQuerierParams(const TeQuerierParams& qp);

	TeQuerierParams& operator=(const TeQuerierParams& rhs);

	/* @name Sets params */
	// @{
	void setParams(TeLayer* layer);

	void setParams(TeTheme* theme, TeChronon chr=TeNOCHRONON); 

	void setParams(TeTheme* theme, const string& objId, TeChronon chr=TeNOCHRONON);

	void setParams(const string& fileName, TeChronon chr=TeNOCHRONON);
	// @}

	//! Sets the params to fill the STO
	void setFillParams(bool loadGeom, bool loadAllAttr, vector<string> loadAttr = vector<string>());

	//! Sets the params to fill the STO
	void setFillParams(bool loadGeom, TeGroupingAttr attrG);
	
	//! Set the spatial restriction to be a spatial relation with a box
	void setSpatialRest(TeBox& box, int relation = TeWITHIN, TeGeomRep rep = TeGEOMETRYNONE); 
	//! Set the spatial restriction to be a spatial relation with a geometry
	void setSpatialRest(TeGeometry* geom, int relation = TeWITHIN, TeGeomRep rep = TeGEOMETRYNONE); 

	//! Set selected objects
	void setSelecetObjs(TeSelectedObjects so) { selectedObjs_ = so; }

	//! Sets the attributes that will be loaded
	void setLoadAttrs(const vector<string>& vec) { loadAttrNames_ = vec; }

	//! Returns the identifier of the strategy associated to the querier
	string decName() { return strategy_; }

	//! Returns the theme
	TeTheme* theme() { return theme_; }

	//! Returns the layer
	TeLayer* layer() { return layer_; }

	//! Returns the file name
	string fileName() { return fileName_; }
	
	//! Returns the chronon
	TeChronon chronon()	{ return chr_; }
	
	//! Returns the object identifier
	string objId() { return objId_; }
	
	//! Returns the attributes that will be grouped
	TeGroupingAttr& groupAttr() { return groupingAttr_;}

	//! Returns if it loads geometries
	bool loadGeom ()  { return	loadGeometries_; }
	
	//! Returns if it loads all attributes 
	bool loadAllAttr() { return loadAllAttributes_;}

	//! Returns the attributes that will be loaded
	vector<string>& loadAttrs() { return loadAttrNames_;}

	//! Returns the original attributes that will be loaded
	vector<string>& loadSetedAttrs() { return loadAttrNamesSeted_;}

	TeSelectedObjects selectedObjs() { return selectedObjs_; }
	
	int spatialRelation() { return spatialRelation_; }

	bool hasSpatialRes() { return hasSpatialRes_;}
	
	TeBox boxRest() { return boxRest_; } 

	TeGeometry* geomRest() { return geomRest_; }

	TeGeomRep  geomRepRest() { return geomRepRest_; }
};

#endif
