/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeQuerierDBStr1.h"
#include "TeDatabase.h"
#include "TeTemporalSeries.h"
#include "TeSTInstance.h"

bool 
TeQuerierDBStr1::initPortal(TeRepresentation& rep, TeTSEntry* ent)
{
	string selectClause, fromClause, whereClause, orderByClause, groupByClause, parClause;
	TeDatabase* db = params_.theme()->layer()->database();
	isGroup_ = false;

	//verify if the theme has collection table
	bool hasCollTable = false;
	if(db->tableExist(params_.theme()->collectionAuxTable()))
		hasCollTable = true;
	
	clearVectors();
	params_.setLoadAttrs(params_.loadSetedAttrs()); //the original data
	
	//get the link name 
	string sqlGroup =""; 
	if(hasCollTable && (!params_.hasSpatialRes()) && (!params_.loadGeom()))
	{
		linkName_ = params_.theme()->collectionTable() +".c_object_id";
		sqlGroup = linkName_;
	}
	else  //has spatial restriction  
	{
		linkName_ = rep.tableName_ +".object_id"; 
		sqlGroup = linkName_ +","+ rep.tableName_ +".geom_id";
		geomRepr_.push_back(rep);
	}

	// order by clause
	orderByClause = " ORDER BY "+ sqlGroup;

	//verify if will be group the objects	
	TeGroupingAttr groups = params_.groupAttr();  
	bool groupAttr = !(groups.empty());
	
	// ---------------------------------------------  Mount SQL
	
	// --------- begin attributes

	// load only the attribute that will be grouped
	if(groupAttr)
	{
		if(!params_.loadGeom())
		{
			string sGroup = db->getSQLStatistics(groups);
			isGroup_ = true;
			TeGroupingAttr::iterator it = groups.begin();
			while(it!= groups.end())
			{
				if(it->second != TeNOSTATISTIC)
					isGroup_ = false;
				++it;
			}

			if(isGroup_)
			{
				selectClause  = sGroup;
				orderByClause = "";
			}
			
			//if there is a spatial restriction, group every objects
			if(!params_.hasSpatialRes())
				groupByClause = " GROUP BY "+ linkName_;
		}

		if(!isGroup_)  
		{
			//group attributes in memory
			groupInMemory_ = true;
			//fill select clause from set of attributes
			string lastAttr = "";
			TeGroupingAttr::iterator it = groups.begin(); 
			while(it!= groups.end())
			{
				if(lastAttr != it->first.name_)
				{
					if(it != groups.begin())
						selectClause += ",";

					selectClause += it->first.name_;
				}
				
				lastAttr = it->first.name_;
				++it;
			}
		}
	}
	// load all attributes or the attributes that are in the vector
	else
	{
			//get attribute tables
		TeAttrTableVector atts; 
		params_.theme()->getAttTables(atts); 
		if(atts.empty() && (params_.loadAllAttr() || (!params_.loadAttrs().empty())))
			return false;
			
		// get some information about the attribute tables required
		for(unsigned int i=0; i<atts.size(); i++)
		{
			string initialTime, finalTime;

			if((atts[i].tableType()==TeAttrEvent) || (atts[i].tableType()==TeFixedGeomDynAttr))
			{
				attrTable_ = atts[i]; 
				initialTime = attrTable_.attInitialTime();
				finalTime = attrTable_.attFinalTime();
			}
		
			// fill vector of unique name 
			string unName = TeConvertToUpperCase(atts[i].uniqueName());
			string unNameT = TeConvertToUpperCase(atts[i].name()+"."+atts[i].uniqueName());
			uniqueNames_.push_back(unName);
			uniqueNames_.push_back(unNameT);

			//fill vector of attributes and sql string with all attributes
			if(params_.loadAllAttr())
			{
				TeAttributeList::iterator itAttr = atts[i].attributeList().begin();
				while(itAttr!= atts[i].attributeList().end())
				{
					string attribute = atts[i].name() +"."+ (*itAttr).rep_.name_;
					params_.loadAttrs().push_back(attribute);
					params_.loadAttrs().push_back((*itAttr).rep_.name_);
				
					//select clause
					if(!selectClause.empty())
						selectClause += ", ";
					selectClause += attribute; 

					++itAttr;
				}
			}
			//only fill the sql string with the attributes of the TeSTInstance
			else // make sure the unique and temporal columns are retrieved
			{
				string un = atts[i].name()+"."+ atts[i].uniqueName();
				if (find(params_.loadAttrs().begin(), params_.loadAttrs().end(), un) == params_.loadAttrs().end())
				{
					if(!selectClause.empty())
						selectClause += ", ";
					selectClause += un;
				}

				string ini = atts[i].name()+"."+initialTime;
				if (!initialTime.empty() && (find(params_.loadAttrs().begin(), params_.loadAttrs().end(), ini) == params_.loadAttrs().end()))
				{
					if(!selectClause.empty())
						selectClause += ", ";
					selectClause += ini;
				}

				string fin = atts[i].name()+"."+finalTime;
				if	(!finalTime.empty() &&  initialTime != finalTime && 
					(find(params_.loadAttrs().begin(), params_.loadAttrs().end(), fin) == params_.loadAttrs().end()))
				{
					if(!selectClause.empty())
						selectClause += ", ";
					selectClause += fin;
				}
			}
		}

		if(!params_.loadAllAttr())
		{
			vector<string>  newNames;
			vector<string>::iterator itVec = params_.loadAttrs().begin();
			while(itVec!=params_.loadAttrs().end())
			{
				//insert in select clause
				if(!selectClause.empty())
						selectClause += ", ";
				selectClause += (*itVec);
			
				//insert in the attribute name vector without table name
				size_t pos = (*itVec).find(".", string::npos,1);
				if (pos != string::npos)
				{
					string newNam = (*itVec).substr(pos+1);
					newNames.push_back(newNam);
				}
				++itVec;
			}

			itVec = newNames.begin();
			while(itVec!=newNames.end())
			{
				params_.loadAttrs().push_back(*itVec);
				++itVec;
			}
		}
	}
	
	// selecionar o campo de data tambm
	if(groupInMemory_ && params_.loadGeom() && (!attrTable_.name().empty()))
	{
		selectClause += ", "+ attrTable_.name()+"."+attrTable_.attInitialTime();
		if(attrTable_.attInitialTime()!=attrTable_.attFinalTime())
			selectClause += ", "+ attrTable_.name()+"."+attrTable_.attFinalTime();
	}

	// --------- end attributes


	// --------- begin geometry

	if(!selectClause.empty())
		selectClause += ", ";

	// select com todos os campos da tabela de geometria
	if(params_.loadGeom())
	{
		TeTable table;
		if(!db->loadTable(rep.tableName_, table))
			return false; 

		vector<string> attrs; 
		table.attributeNames(attrs); 
		for(unsigned int i=0; i<attrs.size(); ++i)
		{
			if(i>0)
				selectClause += ",";
			selectClause += rep.tableName_+"."+ attrs[i];
		}

		// order by clause
		if((rep.geomRep_ == TePOLYGONS) && (db->dbmsName() != "OracleSpatial") && (db->dbmsName() != "PostGIS") )
			orderByClause += ", parent_id ASC, num_holes DESC";

	}
	// select apenas com object_id e geom_id
	else if(isGroup_)
	{
		if(params_.hasSpatialRes())
			selectClause += " MIN("+ linkName_ +") AS terraObjId, MIN("+ rep.tableName_ +".geom_id) AS geom_id ";
		else
			selectClause += " MIN("+ linkName_ +") AS terraObjId "; 

		linkName_ = "terraObjId";
	}
	else
		selectClause += sqlGroup;
		
	// --------- end geometry
		
	// --------- begin restrictions
	if(!hasCollTable)
	{
		TeAttrTableVector atts; 
		params_.theme()->getAttTables(atts); 

		if(atts.empty())
			return false;

		fromClause = " FROM " + tableJoin(atts, rep.tableName_, "object_id");
		whereClause = params_.theme()->sqlWhereRestrictions(&rep);
	}
	else
	{
		if(params_.loadGeom() || params_.hasSpatialRes()) 
			fromClause = params_.theme()->sqlGridFrom(rep.tableName_);
		else
			fromClause = params_.theme()->sqlGridFrom();
	}
				

	if(ent)
	{
		if(attrTable_.name().empty())
			params_.theme()->getTemporalTable(attrTable_);

		string iniTime = attrTable_.name()+"."+attrTable_.attInitialTime(); 
		string finTime = attrTable_.name()+"."+attrTable_.attFinalTime(); 
		if(!whereClause.empty())
			whereClause += " AND ";

		if ((params_.chronon()==TeMONTHOFYEAR) || (params_.chronon()==TeDAYOFWEEK))
			whereClause += db->getSQLTemporalWhere(ent->timeInt_, ent->timeInt_, params_.chronon(), TeTIMEDURING, iniTime, finTime);    
		else
		{
			TeTimeInterval interval = ent->time_;
			interval.intervalChronon(params_.chronon());

			whereClause += db->getSQLTemporalWhere(interval, TeTIMEDURING, iniTime, finTime);
		}
	}

	// --------- end restrictions

	//where clause
	if(!objectId().empty())
	{
		if(!whereClause.empty())
			whereClause += " AND ";
		
		whereClause += linkName_ +" = '"+ objectId() +"'";  
	}


	//---------- querier restriction
	string sqlQuerierRest = sqlWhereRestrictions(&rep);
	if(!whereClause.empty())
		whereClause += " AND ";

	whereClause += sqlQuerierRest;

	//----------

	string sql = "SELECT "+ selectClause + fromClause;
	if(!whereClause.empty())
		sql += " WHERE "+ whereClause;
	
	if(!groupByClause.empty())
		sql += " "+ groupByClause;
	else
		sql += orderByClause;
	
	// --------- Submit the query
	portals_.clear();
	TeDatabasePortal* portal = db->getPortal();
	
	if(!portal)
		return false;
	
	if(!portal->query(sql))
	{
		delete (portal);
		flagPortal_ = false;
		return false;
	}

	if(!portal->fetchRow())
	{
		delete (portal);
		flagPortal_ = false;
		return false;
	}

	string id = portal->getData(linkName_);
	if(id.empty())
	{
		delete (portal);
		flagPortal_ = false;
		return false;
	}

	portals_.push_back (portal);
	flagPortal_ = true;
	lastObjId_ ="";
	return true;
}


bool 
TeQuerierDBStr1::initGeomPortal(TeRepresentation& rep, TeTSEntry* ent)
{
	string selectClause, fromClause, whereClause, orderByClause, parClause;
	string initialTime, finalTime;
	string uniqueName;
	
	TeDatabase* db = params_.theme()->layer()->database();

	//------- Get geometry table
	geomRepr_.push_back(rep);
	if(rep.tableName_.empty())
		return false;
	
	selectClause = rep.tableName_ +".* ";
	orderByClause = " ORDER BY "+ rep.tableName_ +".object_id ";

	if((rep.geomRep_ == TePOLYGONS) && (db->dbmsName() != "OracleSpatial") && (db->dbmsName() != "PostGIS") )
		orderByClause += " , parent_id ASC, num_holes DESC";
	
	//------- Get temporal attribute 
	if(!attrTable_.name().empty())
	{
		initialTime = attrTable_.name() +"."+ attrTable_.attInitialTime ();
		finalTime = attrTable_.name() +"."+ attrTable_.attFinalTime ();
			
		// fill vector of unique name 
		uniqueName = TeConvertToUpperCase(attrTable_.name()+"."+attrTable_.uniqueName());
						
		selectClause += ", "+ attrTable_.name()+"."+ attrTable_.uniqueName();
		selectClause += ", "+ attrTable_.name()+"."+initialTime;

		if (initialTime != finalTime)
			selectClause += ", "+ attrTable_.name()+"."+finalTime;
	}
	
	if(!db->tableExist(params_.theme()->collectionAuxTable()))
	{
		TeAttrTableVector atts; 
		params_.theme()->getAttTables(atts); 

		if(atts.empty())
			return false;

		fromClause = " FROM " + tableJoin(atts, rep.tableName_, "object_id");
		whereClause = params_.theme()->sqlWhereRestrictions(&rep);
	}
	else
		fromClause = params_.theme()->sqlGridFrom(rep.tableName_);
	
	if(ent)
	{
		string iniTime = attrTable_.name()+"."+initialTime;
		string finTime = attrTable_.name()+"."+finalTime;
		if(!whereClause.empty())
			whereClause += " AND ";

		if ((params_.chronon()==TeMONTHOFYEAR) || (params_.chronon()==TeDAYOFWEEK))
			whereClause += db->getSQLTemporalWhere(ent->timeInt_, ent->timeInt_, params_.chronon(), TeTIMEDURING, iniTime, finTime);    
		else
		{
			TeTimeInterval interval = ent->time_;
			interval.intervalChronon(params_.chronon());

			whereClause += db->getSQLTemporalWhere(interval, TeTIMEDURING, iniTime, finTime);
		}
	}

	//where clause
	if(!objectId().empty())
	{
		if(!whereClause.empty())
			whereClause += " AND ";
		
		whereClause += rep.tableName_+".object_id = '"+ objectId() +"'";  
	}

	//---------- querier restriction
	string sqlQuerierRest = sqlWhereRestrictions(&rep);
	if(!whereClause.empty())
		whereClause += " AND ";

	whereClause += sqlQuerierRest;

	//----------

	string sql = "SELECT "+ selectClause + fromClause;
	if(!whereClause.empty())
		sql += " WHERE "+ whereClause;
	sql += orderByClause;

	//------ Submit the query
	TeDatabasePortal* portal = db->getPortal();
	if(!portal)
		return false;
	
	if(!portal->query(sql))
	{
		delete portal;
		return false;
	}

	if(!portal->fetchRow())
	{
		delete portal;
		return false;
	}

	portals_.push_back (portal);
	return true;
}


// ---------- final - initPortal

// It is used in two cases:
// 1) when it must return all instances of an object 
// 2) when the attributes was grouped by sql functions - without geometry 

bool
TeQuerierDBStr1::fillSTOGrouped(TeSTInstance& sto, bool fetchInstance)
{
	if(portals_.empty()) 
		return false;

	TeDatabasePortal* portal = portals_[0]; 
	if(!portal) 
		return false; 

	TeRepresentation* rep = 0;
	if(!geomRepr_.empty())
		rep = &(geomRepr_[0]);

	if(!flagPortal_)
	{
		clearVectors();
		return false;
	}

	TeTime  minT1, maxT2;
	
	// builds an ST object instance
	TeTime t1, t2;
	TePropertyVector prop;

	// verify if exist several geometries for an object
	bool flagGeom = true;
	while(flagGeom && fetchInstance)
	{
		sto.objectId (portal->getData(linkName_));
						
		if(rep && (!params_.loadGeom()))
		{
			int geomId = portal->getInt(rep->tableName_ +".geom_id");
			
			if(sto.objectId()==lastObjId_) 
			{
				if (geomId_!=geomId)
				{
					flagGeom = portal->fetchRow();
					continue;
				}
			}
			else
				geomId_ = geomId;
		}
		flagGeom = false;
	}
	
	
	// ------------- begin fill attribute
	// process the records filling the parameters of each instance
	TeAttributeList attrsPortal = portal->AttributeList();
	TeAttributeList::iterator it= attrsPortal.begin();

	if(isGroup_)  // it was grouped by sql functions and without geometry 
	{
		sto.addUniqueId(string(portal->getData(linkName_)));

		unsigned int size = params_.groupAttr().size();
		for(unsigned int s=0; s<size; ++s)
		{
			TeProperty p;
			p.attr_ = attrsPortal[s];   
			p.value_ = portal->getData (s);
			prop.push_back(p);
		}
	}
	else
	{
		while(it!=attrsPortal.end())
		{
			TeProperty p;
			string name = (*it).rep_.name_;
			p.attr_ = (*it);

			if(name==attrTable_.attInitialTime())
			{
				t1 = portal->getDate(attrTable_.attInitialTime());
				if(attrTable_.attInitialTime()==attrTable_.attFinalTime())
				{
					t2 = t1;
					sto.timeInterval(TeTimeInterval(t1,t2));
				}
			}
			else if(name==attrTable_.attFinalTime())
			{
				t2 = portal->getDate(attrTable_.attFinalTime());
				sto.timeInterval(TeTimeInterval(t1,t2));
			}
			else if(find(uniqueNames_.begin(), uniqueNames_.end(), TeConvertToUpperCase(name)) != uniqueNames_.end())
			{
				string uniqueValue = portal->getData(name);
				sto.addUniqueId(uniqueValue);
			}
			
			if(find(params_.loadAttrs().begin(), params_.loadAttrs().end(), name) != params_.loadAttrs().end())
			{
				p.value_ = portal->getData(name);
				prop.push_back(p);
			}
			++it;
		}
	}

	//! Set the property in the stoInstance
	sto.properties(prop);	
	sto.theme(params_.theme());

	if(!fetchInstance) //only to get the attributes
		return true;

	// ------------- end fill attribute

	//total time
	if(t1.isValid() && t1<minT1)
		minT1 = t1;
	if(t2.isValid() && maxT2<t2)
		maxT2 = t2;
		
	// ------------- begin fill geometries

	if(params_.loadGeom())
	{
		if(t1.isValid() && t2.isValid())
			flagPortal_ = addGeometry(portal, rep->geomRep_, sto, rep->tableName_, TeTimeInterval(t1,t2), attrTable_.attInitialTime(), attrTable_.attFinalTime());
		else
			flagPortal_ = addGeometry(portal, rep->geomRep_, sto, rep->tableName_);
	}
	else
		flagPortal_ = portal->fetchRow ();

	// ------------- end fill geometries
	
	lastObjId_ = sto.objectId(); 
	return true;
}

bool 
TeQuerierDBStr1::fillGeomSTO(TeSTInstance& sto, unsigned int index)
{
	if((portals_.size()<(index+1)) || (geomRepr_.size()<(index+1)))
		return false;

	TeDatabasePortal* portal = portals_[index]; 
	TeRepresentation rep = geomRepr_[index];
	if(!portal)
		return false; 

	TeTimeInterval time = sto.timeInterval();
	bool flag = false;
	if(time.isValid())
		flag = addGeometry(portal, rep.geomRep_, sto, rep.tableName_, time, attrTable_.attInitialTime(), attrTable_.attFinalTime());
	else
		flag = addGeometry(portal, rep.geomRep_, sto, rep.tableName_);

	return flag;
}


// It is used in a case:
// 1) when the attributes of the object instances must be grouping in memory - with or without geometry 
bool
TeQuerierDBStr1::fillSTONoGrouped(TeSTInstance& sto)
{
	unsigned int s;
	if(portals_.empty())
		return false;

	TeDatabasePortal* portal = portals_[0]; 
	if(!portal) 
		return false; 

	TeRepresentation* rep = 0; 
	if(!geomRepr_.empty())
		rep = &(geomRepr_[0]);
	
	//verify if the portal points to any STO
	if(!flagPortal_)
	{
		clearVectors();
		return false;
	}

	TePropertyVector prop;
	map<int, vector<double> >	values;
	string lastObj = "";

	TeAttributeList attrsPortal = portal->AttributeList();
	
	if(!params_.loadGeom())
	{
		do
		{
			string objId = portal->getData(linkName_);
			if(!rep)
			{
				lastObj = objId;
				flagPortal_ = portal->fetchRow();
				break;
			}

			int geomId = portal->getInt(rep->tableName_ +".geom_id");

			if(lastObj.empty())
				geomId_ = geomId;  //first read geomId
			
			if((!lastObj.empty()) && (objId != lastObj))
				break;  
					
			//verify this is the geomId  
			if(geomId!=geomId_)
			{
				flagPortal_ = portal->fetchRow();
				lastObj = objId;
				continue;
			}
				
			string val = ""; 
			unsigned int size = attrsPortal.size();
			//portal has the attributes that will be grouped and geometric columns (object_id and geom_id)
			for(unsigned int s=0; s<(size-2); ++s)  
			{
				val = portal->getData(s);
				if(val.empty())
					values[s].push_back (TeMAXFLOAT);  //invalid value
				else
					values[s].push_back (atof(val.c_str()));
			}

			flagPortal_ = portal->fetchRow();
			lastObj = objId;

		} while(flagPortal_);
	}
	else // -- when fill the geometry
	{
		
		//------- first: load the geometry, attributes and time of the first object 
		//object id
		lastObj = portal->getData(rep->tableName_ +".object_id");
		
		//attribute val
		string val = ""; 
		unsigned int size = attrsPortal.size();
		for(s=0; s<(size-1); ++s)
		{
			val = portal->getData(s);
			if(val.empty())
				values[s].push_back (TeMAXFLOAT);  //invalid value
			else
				values[s].push_back (atof(val.c_str()));
		}

		//time
		TeTime t1, t2;
		if(!attrTable_.name().empty())
		{
			t1 = portal->getDate(attrTable_.attInitialTime());
			t2 = portal->getDate(attrTable_.attFinalTime());
		}
		
		if(t1.isValid() && t2.isValid())
			flagPortal_ = addGeometry(portal, rep->geomRep_, sto, rep->tableName_, TeTimeInterval(t1,t2), attrTable_.attInitialTime(), attrTable_.attFinalTime());
		else
			flagPortal_ = addGeometry(portal, rep->geomRep_, sto, rep->tableName_);
		
		//------- second: load the attributes of the other objects
		int cont = 0;
		while (flagPortal_)
		{
			string objId = portal->getData(rep->tableName_ +".object_id");
			int geomId = portal->getInt(rep->tableName_ +".geom_id");

			if(objId != lastObj)
				break;  
			
			if(!cont)
				geomId_ = geomId;  //first geom_id 
			
			//verify the geomId
			if(geomId!=geomId_)
			{
				flagPortal_ = portal->fetchRow();
				lastObj = objId;
				continue;
			}
				
			val = ""; 
			for(s=0; s<(size-1); ++s)
			{
				val = portal->getData(s);
				if(val.empty())
					values[s].push_back (TeMAXFLOAT);  //invalid value
				else
					values[s].push_back (atof(val.c_str()));
			}

			flagPortal_ = portal->fetchRow();
			lastObj = objId; 
			++cont;
		}
	}

	//fill the property from portal 
	//calculate the statistics
	string lastAttr = "";
	unsigned int j=0;
	string colName = "";
	TeGroupingAttr::iterator it = params().groupAttr().begin(); 

	TeStatisticValMap stat;
	while(it!=params().groupAttr().end())
	{
		if(lastAttr != it->first.name_)
		{
			if(j>=(values.size()))
				return false;
			
			stat.clear();
			TeCalculateStatistics(values[j].begin(), values[j].end(), stat);
			colName = attrsPortal[j].rep_.name_;
			++j;
		}

		double result;
		string prefix;
		switch(it->second)
		{
			case TeCOUNT:
				result = stat[TeCOUNT]; 
				prefix = "count";
				break;
			case TeVALIDCOUNT:
				result = stat[TeVALIDCOUNT]; 
				prefix = "valCount";
				break;
			case TeMINVALUE:
				result = stat[TeMINVALUE];
				prefix = "min";
				break;
			case TeMAXVALUE:
				result = stat[TeMAXVALUE];
				prefix = "max";
				break;
			case TeSUM:
				result = stat[TeSUM]; 
				prefix = "sum";
				break;
			case TeMEAN:
				result = stat[TeMEAN]; 
				prefix = "mean";
				break;
			case TeSTANDARDDEVIATION:
				result = stat[TeSTANDARDDEVIATION];
				prefix = "stDeviation";
				break;
			case TeVARIANCE:
				result = stat[TeVARIANCE];
				prefix = "variance";
				break;
			case TeSKEWNESS:
				result = stat[TeSKEWNESS];
				prefix = "skwness";
				break;
			case TeKURTOSIS:
				result = stat[TeKURTOSIS]; 
				prefix = "kurtosis";
				break;
			case TeAMPLITUDE:
				result = stat[TeAMPLITUDE]; 
				prefix = "amplitude";
				break;
			case TeMEDIAN:
				result = stat[TeMEDIAN]; 
				prefix = "median";
				break;
			case TeVARCOEFF:
				result = stat[TeVARCOEFF]; 
				prefix = "varcoeff";
				break;
			case TeMODE: 
				result = stat[TeMODE]; 
				prefix = "mode";
				break;
			default:
				break;
		}
		
		lastAttr = it->first.name_; 
		++it;

		//fill the property 
		TeProperty p;
		p.attr_.rep_.name_ = prefix +"_"+ colName;
		p.attr_.rep_.type_ = TeREAL;
		p.value_ = Te2String(result);

		prop.push_back(p);
	}
	
	//adds the property in the stoInstance
	sto.properties(prop);	
	sto.theme(params_.theme());
	sto.objectId(lastObj);
	sto.addUniqueId(lastObj);

	lastObjId_ = sto.objectId(); 
	return true;
}


bool
TeQuerierDBStr1::loadInstances(TeTSEntry* ent)
{
	if(!params_.theme())
		return false;

	TeRepresPointerVector repres = params_.theme()->layer()->vectRepres();
	TeRepresentation rep;
	
	//fill instances - first fill the geometry representation of the spatial restriction 
	if(params_.theme()->hasSpatialRest())
		rep = **(TeFindRepresentation(repres.begin(), repres.end(), params_.theme()->geomRepRestriction()));
	else
	{
		for(int i=0; i<(int)repres.size(); ++i)
		{
			if(repres[i]->geomRep_==TeTEXT)
				continue; 
			
			rep = *repres[i];
			break;
		}
	}
	
	if(!initPortal(rep, ent))
		return false;

	//get the attribute list
	attrList_.clear();
	if(!groupInMemory_)
	{
		TeSTInstance i;
		fillSTOGrouped(i, false);
		TePropertyVector props = i.getPropertyVector();
		for(int c=0; c<(int)props.size(); ++c)
			attrList_.push_back (props[c].attr_);
	}
	else
	{
		TeGroupingAttr::iterator it = params().groupAttr().begin();
		while(it!=params().groupAttr().end())
		{
			TeAttribute at;
			at.rep_ = it->first;
			attrList_.push_back (at);
			++it;
		}
	}
			
	if(!params_.loadGeom())
		return true;
	
	for(unsigned int i=0; i<repres.size(); ++i)
	{
		if( (repres[i]->geomRep_ == rep.geomRep_) || 
			(repres[i]->geomRep_ == TeTEXT) )
			continue;

		if(!initGeomPortal(*(repres[i]), ent))
		{
			clearVectors();
			return false;
		}
	}

	return true;
}

bool 
TeQuerierDBStr1::fetchInstance(TeSTInstance&  sto)  
{
	sto.clear();
	bool flag = false;
	
	if(groupInMemory_)
		flag = fillSTONoGrouped(sto); 
	else
		flag = fillSTOGrouped(sto); 
	
	if((flag==false) || (!params_.loadGeom()))
		return flag;
	
	for(unsigned int i=1; i<portals_.size(); ++i)
		flag = fillGeomSTO(sto, i);
	
	return flag;
}


int 
TeQuerierDBStr1::numElemInstances()
{
	if((portals_.empty()))
		return 0;

	TeDatabasePortal* portal = portals_[0];
	if(!portal)
		return 0;

	return (portal->numRows());
}




