/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#ifndef  __TERRALIB_INTERNAL_ORACLESPATIAL_H
#define  __TERRALIB_INTERNAL_ORACLESPATIAL_H
#if _MSC_VER > 1000
#pragma once
#endif 

#include "connect.h"
#include "cursor.h"
#include <TeDatabase.h>

#ifndef COMMIT
#define COMMIT 1
#define ROLLBACK 2
#endif

#ifdef AFX_DLL
#define EXPORT_WIN __declspec( dllexport )
#else
#define EXPORT_WIN
#endif

//SDO_GEOMETRY types  (number of dimensions=2 and LRS=0)
enum TeSDOGType  
{ TeSDOUNKNOWN, TeSDOPOINT, TeSDOLINE, TeSDOPOLYGON,TeSDOCOLLECTION, 
  TeSDOMULTIPOINT, TeSDOMULTILINE, TeSDOMULTIPOLYGON};

//! A concrete implementation of a driver to the Oracle Spatial SGDB
class EXPORT_WIN TeOracleSpatial : public TeDatabase
{

friend class TeOracleSpatialPortal;

private:

	//! Inserting geometric table in the metadata table USER_SDO_GEOM_METADATA  
	bool	insertMetadata(const string &table, const string &column, double tolx,double toly,TeBox &box,short srid=0);

	//! Deleting geometric table in the metadata table USER_SDO_GEOM_METADATA  
	bool	DeleteMetadata(const string &table, const string &column);
	
	//! Create/Rebuild/Delete spatial index
	bool	createSpatialIndex(const string &table, const string &column,TeSpatialIndexType type= TeRTREE,short level=0,short tile=0);
	bool	RebuildSpatialIndex(const string &table);
	bool	DeleteSpatialIndex(const string &table);

	bool	createSequence(const string &seqName);
	bool	createAutoIncrementTrigger(const string &tableName, const string &fieldName);
	string	getNameSequence(const string &tableName);
	string  getNameTrigger(const string &tableName);

	//! Connection to ORACLE database   
	OCIConnection  *Conn;  

public:
	
	TeOracleSpatial();

	~TeOracleSpatial();


	string  escapeSequence(const string& from); 

	bool newDatabase(const string& /*database*/, const string& /*user*/, const string& /*password*/, const string& /*host*/, const int& /*port=0*/, bool /*terralibModel=true*/) { return false; }

	bool connect (const string& host, const string& user, const string& password, const string& database, int port = 0);

	void  close();

	bool tableExist(const string& table);

	bool columnExist(const string& table, const string& column, TeAttribute& attr);

	bool createTable (const string& table, TeAttributeList &attr);

	bool addColumn (const string& table,TeAttributeRep &rep);

	bool deleteTable (const string& table);

	bool execute (const string &sql);
	
	TeDatabasePortal* getPortal ();

	TeDBRelationType existRelation(const string& tableName, const string& relName);

	bool createRelation (const string& name, const string& table, const string& fieldName, const string& relatedTable, const string& relatedField, bool cascadeDeletion);
		
	// --- create metadata tables 
	bool createViewTable ();
	
	bool createThemeTable ();  

	bool createGroupingTable();

	bool createThemeTablesTable();

	bool createLayerTable ();

	bool createLayerTableTable ();

	bool createTablesRelationTable();

	bool createRepresentationTable();

	bool createLegendTable();

	bool createVisualRasterTable();

	bool createVisualTable();

	bool createDatabaseTable();
	
	bool createProjectionTable ();
	
	bool createRasterMetadataTable(const string& tableName);
	
	bool createLUTTable(const string& table);
	
	bool createCollectionTable(const string& tableName);
	// ---

	// --- create geometry tables
	bool createCellGeometry (const string& table);
 
	bool createTextGeometry (const string& table);

	bool createArcGeometry (const string& table);

	bool createNodeGeometry (const string& table);

	bool createRasterGeometry (const string& table);

	bool createRasterTable (const string& tableName);

	bool createPolygonGeometry (const string& table);

	bool createLineGeometry (const string& table);

	bool createPointGeometry (const string& table);
	// -----

	// ----- Insert/Update/Delete metadata tables
	bool insertTableInfo (int layerId, TeTable &table, const string& user="");
	bool insertRelationInfo(const int tableId, const string& tField, const string& rName, const string& rField, int& relId);

	bool insertTable	(TeTable &table);
	bool updateTable	(TeTable &table);
	
	bool insertProjection (TeProjection *proj);
	
	bool insertView		(TeView *view);	
	bool insertViewTree (TeViewTree *tree);	

	bool insertTheme		(TeTheme *theme);
	bool insertThemeTable	(int themeId, int tableId, int relationId, int tableOrder);
	bool insertThemeGroup	(TeViewTree* tree);
	bool generateLabelPositions	(TeTheme *theme); 
	
	bool insertLayer	(TeLayer *layer);	
	bool deleteLayer	(int layerId);
	
	bool insertRepresentation (int layerId, TeRepresentation& rep);

	bool insertLegend	(TeLegendEntry *legend);	
	// ----------------

	// ----- Insert/Update/Delete geometry tables
	//polygon
	bool insertPolygon		(const string& table, TePolygon &p);	
	bool updatePolygon		(const string& table, TePolygon &p);
	bool locatePolygon		(const string& table, TeCoord2D &pt, TePolygon &polygon, const double& tol = 0.0);
	
	bool selectPolygonSet	(const string& table, const string& criteria, TePolygonSet &ps); 
	bool loadPolygonSet		(const string& table, const string& geoid, TePolygonSet &ps);  
	bool loadPolygonSet		(const string& table, TeBox &box, TePolygonSet &ps); 
	bool loadPolygonSet		(TeTheme* theme, TePolygonSet &ps); 
	TeDatabasePortal*		loadPolygonSet(const string& table, TeBox &box); 
	
	bool AllocateOrdinatesObject(TePolygon &poly, string& elInfo, OCICursor* cursor=0);
	
	//line
	bool insertLine		(const string& table, TeLine2D &l);		
	bool updateLine		(const string& table, TeLine2D &l);
	bool locateLine		(const string& table, TeCoord2D &pt, TeLine2D &line, const double& tol = 0.0);

	bool loadLineSet	(const string& table, const string& geoid, TeLineSet &ls); 
	bool loadLineSet	(const string& table, TeBox &box, TeLineSet &linSet); 
	TeDatabasePortal*   loadLineSet (const string& table, TeBox &box);
	
	bool AllocateOrdinatesObject(TeLine2D &line, OCICursor* cursor=0);
			
    //point
	bool insertPoint	(const string& table, TePoint &p);	
	bool updatePoint	(const string& table, TePoint &p);
	bool locatePoint	(const string& table, TeCoord2D &pt, TePoint &point, const double& tol = 0.0);
    
	//text
	bool insertText		(const string& table, TeText &t);	

	//arc
	bool insertArc		(const string& table,TeArc &arc);

	//node
	bool insertNode		(const string& table, TeNode &node);	
	bool updateNode		(const string& table, TeNode &node);	

	//cell
	bool insertCell		(const string& table, TeCell &c);
	bool updateCell		(const string& table, TeCell &c);
	bool locateCell		(const string& table, TeCoord2D &pt, TeCell &cell, const double& tol = 0.0);
	
	//raster
	bool insertRasterBlock(const string& table, const string& blockId, const TeCoord2D& ll, const TeCoord2D& ur, unsigned char *buf,unsigned long size, int band=0, unsigned int res=1, unsigned int subband=0);
	// ----------

	// ----- blob
	bool insertBlob (const string& tableName, const string& columnBlob, TeAttributeRep& columnId, const string& valueId, unsigned char* data, int size);
	bool insertBlob (const string& tableName, const string& columnBlob, TeAttributeRep& columnId, const string& valueId, const string& fileName);
	// -----
	
	//---------- SQLs
	string getSQLBoxWhere (TeBox &box, TeGeomRep rep); 
	
	string getSQLBoxWhere (const string& table1, const string& table2, TeGeomRep rep2, TeGeomRep rep1); 
	
	string getSQLBoxSelect (const string& tableName, TeGeomRep rep); 

	string getSQLStatistics (TeGroupingAttr& attrs);

	string  getSQLAutoNumber(const string& table);
	// ------
	
	bool getMBRSelectedObjects(string geomTable,string colGeom, string fromClause, string whereClause, string afterWhereClause, TeGeomRep repType,TeBox &bout, const double& tol = 0.0);

	bool dropConceptualModel();
	
	bool getMBRGeom(string tableGeom, string object_id, TeBox& box, string colGeom);

	//Spatial query
	//spatial relate between a set of geometry of the table actTable and all other geometry 
	//return a portal
	bool spatialRelation(const string& actGeomTable, TeGeomRep actRep, Keys& actIdsIn, TeDatabasePortal *portal, int relate, const string& actCollTable="");
	bool spatialRelation(const string& actGeomTable, TeGeomRep actRep, Keys& actIdsIn, const string& visGeomTable, TeGeomRep visRep, TeDatabasePortal *portal, int relate, const string& visCollTable=""); 
	bool spatialRelation(const string& actGeomTable, TeGeomRep actRep, TeGeometry* geom, TeDatabasePortal *portal, int relate, const string& actCollTable=""); 
	
	//spatial relate between a set of geometry of the table actTable and all other geometry 
	//return a object_id vector
	bool spatialRelation(const string& actGeomTable, TeGeomRep actRep, Keys& actIdsIn, Keys& actIdsOut, int relate, const string& actCollTable="");
	bool spatialRelation(const string& actGeomTable, TeGeomRep actRep, Keys& actIdsIn, const string& visGeomTable, TeGeomRep visRep, Keys& visIdsOut, int relate, const string& visCollTable=""); 
	bool spatialRelation(const string& actGeomTable, TeGeomRep actRep, TeGeometry* geom, Keys& actIdsOut, int relate, const string& actCollTable=""); 
	
	//metric functions
	bool calculateArea(const string& actGeomTable, TeGeomRep actRep, Keys& actIdsOut, double &area);
	bool calculateLength(const string& actGeomTable, TeGeomRep actRep, Keys& actIdsIn, double &length);
	
	bool calculateDistance(const string& actGeomTable, TeGeomRep actRep, Keys& Ids, double& distance);
	bool calculateDistance(const string& actGeomTable, TeGeomRep actRep, const string& objId1, const string& visGeomTable, TeGeomRep visRep, const string& objId2, double& distance);

	//fazer para TeGeometry*
	//bool withinDistance(const string& actGeomTable, TeGeomRep actRep, const TeCoord2D& point, KeysToDist& IdsDistOut, const double& max_distance, const string& actCollTable="");

	// functions that generate new geometry
	bool Buffer(const string& actGeomTable, TeGeomRep actRep, Keys& actIds, TePolygonSet& bufferSet, double dist);
	bool ConvexHull(const string& actGeomTable, TeGeomRep actRep, Keys& actIds, TePolygonSet& convexHullSet);
	bool Centroid(const string&  actGeomTable , TeGeomRep actRep, TePointSet& centroidSet, Keys actIds = vector<string>(), const string& actCollTable = "");

	//return the objects identificators of the nearest neighbors  
	bool nearestNeighbors(const string& actGeomTable, const string& actCollTable, TeGeomRep actRep, const string& objId1, Keys& actIdsOut, int numRes=1);
	bool nearestNeighbors(const string& actGeomTable, TeGeomRep actRep, const string& objId1, const string& visGeomTable, const string& visCollTable, TeGeomRep visRep, Keys& visIdsOut, int numRes=1); 

	//return a portal with the nearest neighbors
	bool nearestNeighbors(const string& actGeomTable, const string& actCollTable, TeGeomRep actRep, const string& objId1, TeDatabasePortal* portal, int numRes=1);
	bool nearestNeighbors(const string& actGeomTable, TeGeomRep actRep, const string& objId1, const string& visGeomTable, const string& visCollTable, TeGeomRep visRep, TeDatabasePortal* portal, int numRes=1); 

	bool geomIntersection(const string& actGeomTable, TeGeomRep actRep, Keys& actIds, TeGeometryVect& geomVect);
	bool geomIntersection(const string& actGeomTable, TeGeomRep actRep, const string& objId1, const string& visGeomTable, TeGeomRep visRep, const string& objId2, TeGeometryVect& geomVect);

	bool geomDifference(const string& actGeomTable, TeGeomRep actRep, const string& objId1, const string& objId2, TeGeometryVect& geomVect);
	bool geomDifference(const string& actGeomTable, TeGeomRep actRep, const string& objId1, const string& visGeomTable, TeGeomRep visRep, const string& objId2, TeGeometryVect& geomVect);

	bool geomUnion(const string& actGeomTable, TeGeomRep actRep, Keys& actIds, TeGeometryVect& geomVect);
	bool geomUnion(const string& actGeomTable, TeGeomRep actRep, const string& objId1, const string& visGeomTable, TeGeomRep visRep, const string& objId2, TeGeometryVect& geomVect);

	bool geomXOr(const string& actGeomTable, TeGeomRep actRep, const string& objId1, const string& objId2, TeGeometryVect& geomVect);
	bool geomXOr(const string& actGeomTable, TeGeomRep actRep, const string& objId1, const string& visGeomTable, TeGeomRep visRep, const string& objId2, TeGeometryVect& geomVect);	

	//! Concat values in a vector using unionString as the join between each value
	string concatValues(vector<string>& values, const string& unionString);

	//! Returns the SQL function for upper case
	string toUpper(const string& value);
	
};

//! A concrete implementation of a portal to a Oracle Spatial database
class EXPORT_WIN TeOracleSpatialPortal : public TeDatabasePortal
{
private:
	OCICursor		*cursor;
	string		Value_;
	long		curRow_;

    bool isConnected ();
    bool isEOF();

	
public:

	TeOracleSpatialPortal(TeOracleSpatial *pDatabase);
	~TeOracleSpatialPortal();

	OCICursor*	getCursor() { return cursor; }

    //! Move operations
	bool moveFirst();
    bool moveNext();

	bool query ( const string &q,TeCursorLocation l = TeSERVERSIDE, TeCursorType t = TeUNIDIRECTIONAL, TeCursorEditType e = TeREADONLY, TeCursorDataType dt = TeTEXTCURSOR );
	bool querySDO (const string &q);
	
	bool fetchRow ();
	bool fetchRow (int i);

	void freeResult ();

	char* getData (int i);
	char* getData (const string& s);

	double getDouble (int i);
	double getDouble (const string& s);
	int	getInt (int i);
	int	getInt (const string& s);

	bool getBool (const string& s);
	bool getBool (int i);

	bool getGeometry (TeGeometry** geom, bool& result);
	bool fetchGeometry (TePolygon& poly);
	bool fetchGeometry (TePolygonSet& polySet);
	bool fetchGeometry (TeLine2D& line);
	bool fetchGeometry (TeLineSet& lineSet);
	bool fetchGeometry (TeNode& n);
	bool fetchGeometry (TePoint& p);
	bool fetchGeometry (TePointSet& pointSet);
	bool fetchGeometry (TeCell& cell);
	
	bool getRasterBlock(unsigned long& size, unsigned char*);
	
	bool getBlob(const string& s, unsigned char* &data, long& size);
		
	TeTime getDate (int i);
	TeTime getDate (const string& s);

	string getDateAsString(int i);
	string getDateAsString(const string& s);

	//! Oci geometry methods: Get SDO_GEOMETRY informations   
	int			GetDimArraySize();
	bool		GetDimElement(int i,int &elem);
	int			NumberOfOrdinates();
	bool		GetCoordinates(int i,TeCoord2D& coord);
	bool		GetGeometryType(TeSDOGType& gType);
	int			GetSpatialReferenceId();
	bool		GetPointXYZ (double& x,double& y);
};

#endif 



