/*
 * libsyncml - A syncml protocol implementation
 * Copyright (C) 2005  Armin Bauer <armin.bauer@opensync.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; version 
 * 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include <libsyncml/syncml.h>
#include "sml_devinf_obj.h"

#include <libsyncml/syncml_internals.h>
#include "sml_devinf_obj_internals.h"
#include <libsyncml/sml_session_internals.h>
#include <libsyncml/sml_elements_internals.h>
#include <libsyncml/sml_command_internals.h>

#include<sys/utsname.h>

/**
 * @defgroup GroupIDPrivate Group Description Internals
 * @ingroup ParentGroupID
 * @brief The private part
 * 
 */
/*@{*/

static void _get_devinf_reply(SmlSession *session, SmlStatus *status, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, status, userdata);
	SmlError *error = NULL;
	SmlDevInfAgent *agent = userdata;
	
	if (smlStatusIsResult(status)) {
		SmlCommand *result = smlStatusGetResult(status);
		
		agent->recvDevInf = smlDevInfFromResult(result, &error);
		if (!agent->recvDevInf)
			goto error;
		
		SmlStatus *reply = smlCommandNewReply(result, SML_NO_ERROR, &error);
		if (!reply)
			goto error;
	
		if (!smlSessionSendReply(session, reply, &error)) {
			smlStatusUnref(reply);
			goto error;
		}
		
		smlStatusUnref(reply);
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;

error:
	smlSessionDispatchEvent(session, SML_SESSION_EVENT_ERROR, NULL, NULL, NULL, error);
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
}

static void _devinf_reply(SmlSession *session, SmlStatus *status, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, status, userdata);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
}

/*@}*/

/**
 * @defgroup GroupID Group Description
 * @ingroup ParentGroupID
 * @brief What does this group do?
 * 
 */
/*@{*/


static SmlBool _send_devinf(SmlDevInfAgent *agent, SmlSession *session, SmlCommand *get, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, agent, session, get, error);
	SmlCommand *result = NULL;
	SmlCommand *cmd = NULL;
	
	if (!agent->devinfSent) {
		if (get) {
			if (smlSessionGetVersion(session) == SML_VERSION_10)
				result = smlDevInfNewResult(get, agent->devinf, SML_DEVINF_VERSION_10, error);
			else if (smlSessionGetVersion(session) == SML_VERSION_12)
				result = smlDevInfNewResult(get, agent->devinf, SML_DEVINF_VERSION_12, error);
			else
				result = smlDevInfNewResult(get, agent->devinf, SML_DEVINF_VERSION_11, error);
			
			if (!result)
				goto error;
			
			if (!smlSessionSendCommand(session, result, NULL, _devinf_reply, agent, error)) {
				smlCommandUnref(result);
				goto error;
			}
			
			smlCommandUnref(result);
			
			SmlStatus *reply = smlCommandNewReply(get, SML_NO_ERROR, error);
			if (!reply)
				goto error;
			
			if (!smlSessionSendReply(session, reply, error)) {
				smlStatusUnref(reply);
				goto error;
			}
			
			smlStatusUnref(reply);
		} else {
			if (smlSessionGetVersion(session) == SML_VERSION_10)
				cmd = smlDevInfNewPut(agent->devinf, SML_DEVINF_VERSION_10, error);
			else if (smlSessionGetVersion(session) == SML_VERSION_12)
				cmd = smlDevInfNewPut(agent->devinf, SML_DEVINF_VERSION_12, error);
			else
				cmd = smlDevInfNewPut(agent->devinf, SML_DEVINF_VERSION_11, error);
			
			if (!cmd)
				goto error;
			
			if (!smlSessionSendCommand(session, cmd, NULL, _devinf_reply, agent, error)) {
				smlCommandUnref(cmd);
				goto error;
			}

			smlCommandUnref(cmd);
		}
		agent->devinfSent = TRUE;
	} else {
		smlTrace(TRACE_INTERNAL, "Already sent the devinf!");
	
		/* We return an generic error if we dont want to send the devinf to a get
		 * request. Hope that all devices like this */
		SmlStatus *reply = smlCommandNewReply(get, SML_ERROR_GENERIC, error);
		if (!reply)
			goto error;
		
		if (!smlSessionSendReply(session, reply, error)) {
			smlStatusUnref(reply);
			goto error;
		}
		
		smlStatusUnref(reply);
	}
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

static void _recv_devinf(SmlSession *session, SmlCommand *cmd, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, cmd, userdata);
	SmlDevInfAgent *agent = userdata;
	SmlError *error = NULL;
	char *data = NULL;
	unsigned int size = 0;
	
	if (!smlItemGetData(cmd->private.access.item, &data, &size, &error))
		goto error;

	agent->recvDevInf = smlDevInfParse(data, size, &error);
	if (!agent->recvDevInf)
		goto error;

	SmlStatus *reply = smlCommandNewReply(cmd, SML_NO_ERROR, &error);
	if (!reply)
		goto error;

	if (!smlSessionSendReply(session, reply, &error)) {
		smlStatusUnref(reply);
		goto error;
	}
	
	smlStatusUnref(reply);
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
	smlErrorDeref(&error);
}

static void _request_devinf(SmlSession *session, SmlCommand *cmd, void *userdata)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, session, cmd, userdata);
	SmlDevInfAgent *agent = userdata;
	SmlError *error = NULL;
	
	if (!_send_devinf(agent, session, cmd, &error))
		goto error;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(&error));
}

SmlDevInfAgent *smlDevInfAgentNew(SmlDevInf *devinf, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p)", __func__, devinf, error);
	smlAssert(devinf);
	
	SmlDevInfAgent *agent = smlTryMalloc0(sizeof(SmlDevInfAgent), error);
	if (!agent)
		goto error;

	agent->devinfSent = FALSE;
	agent->devinf = devinf;
	agent->recvDevInf = NULL;

	if (!smlDevInfGetManufacturer(devinf))
		smlDevInfSetManufacturer(devinf, "OpenSync");
	if (!smlDevInfGetModel(devinf))
		smlDevInfSetModel(devinf, "libsyncml");
	if (!smlDevInfGetOEM(devinf))
	{
		struct utsname *buf = malloc(sizeof(struct utsname));
		if (uname(buf) == 0)
		{
			smlDevInfSetOEM(devinf, buf->sysname);
			smlDevInfSetFirmwareVersion(devinf, buf->release);
		}
		free(buf);
	}
	if (!smlDevInfGetSoftwareVersion(devinf))
		smlDevInfSetSoftwareVersion(devinf, VERSION);
	
	smlTrace(TRACE_EXIT, "%s: %p", __func__, agent);
	return agent;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return NULL;
}


void smlDevInfAgentFree(SmlDevInfAgent *agent)
{
	smlTrace(TRACE_ENTRY, "%s(%p)", __func__, agent);
	smlAssert(agent);
	
	g_free(agent);
	
	smlTrace(TRACE_EXIT, "%s", __func__);	
}

/* Set the devinf of the remote peer. */
void smlDevInfAgentSetDevInf(SmlDevInfAgent *agent, SmlDevInf *devinf)
{
	smlAssert(agent);
	smlAssert(devinf);
	agent->recvDevInf = devinf;
}

/* Get the devinf that was sent in the session. Returns FALSE if no devinf was received yet. */
SmlDevInf *smlDevInfAgentGetDevInf(SmlDevInfAgent *agent)
{
	smlAssert(agent);
	return agent->recvDevInf;
}

/** Issues a put request on the session if needed */
SmlBool smlDevInfAgentSendDevInf(SmlDevInfAgent *agent, SmlSession *session, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, agent, session, error);
	smlAssert(agent);
	
	if (!_send_devinf(agent, session, NULL, error))
		goto error;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

/** Issues a Get request for the devinf on the session if needed */
SmlBool smlDevInfAgentRequestDevInf(SmlDevInfAgent *agent, SmlSession *session, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, agent, session, error);
	smlAssert(agent);
	SmlCommand *get = NULL;
	
	if (agent->recvDevInf) {
		smlTrace(TRACE_EXIT, "%s: Already have the devinf", __func__);
		return TRUE;
	}
	
	if (smlSessionGetVersion(session) == SML_VERSION_10)
		get = smlDevInfNewGet(SML_DEVINF_VERSION_10, error);
	else if (smlSessionGetVersion(session) == SML_VERSION_12)
		get = smlDevInfNewGet(SML_DEVINF_VERSION_12, error);
	else
		get = smlDevInfNewGet(SML_DEVINF_VERSION_11, error);
	
	if (!get)
		goto error;
		
	if (!smlSessionSendCommand(session, get, NULL, _get_devinf_reply, agent, error)) {
		smlCommandUnref(get);
		goto error;
	}
	
	smlCommandUnref(get);
			
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;
	
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlBool smlDevInfAgentRegisterSession(SmlDevInfAgent *agent, SmlManager *manager, SmlSession *session, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p, %p)", __func__, agent, manager, session, error);
	smlAssert(agent);
	smlAssert(manager);

	SmlLocation *devinf12 = smlLocationNew("./devinf12", NULL, error);
	SmlLocation *devinf11 = smlLocationNew("./devinf11", NULL, error);
	SmlLocation *devinf10 = smlLocationNew("./devinf10", NULL, error);

	if (!devinf12 || !devinf11 || !devinf10)
		goto error;

	/* PUT callbacks
	 * 
	 * This command is used to actively send device information without
	 * being asked for. Usually a client do this if it is the first
	 * connection with the server or the client has no problems with
	 * its bandwith.
	 *
	 * Sometimes a server is missing the client's device information.
	 * In this situation the server can send a PUT and a GET to force
	 * the client to send the device information. This can happen if
	 * the client know that server should have its device information
	 * because of an earlier synchronization but the server does not
	 * have the device information. Sometimes server databases (incl.
	 * the device information database) are lost or it is simply a
	 * new server but with the old server address (IP) or name (DNS).
	 *
	 * Please note that a client can ignore a PUT from a server.
	 * Nevertheless it is recommended that a client does not ignore
	 * the device information of a server.
	 */

	smlTrace(TRACE_INTERNAL, "register callbacks for PUT command");
	if (!smlManagerObjectRegister(
			manager, SML_COMMAND_TYPE_PUT, session, 
			NULL, devinf10, NULL, _recv_devinf, NULL, agent, 
			error))
		goto error_free_loc;
	if (!smlManagerObjectRegister(
			manager, SML_COMMAND_TYPE_PUT, session, 
			NULL, devinf11, NULL, _recv_devinf, NULL, agent, 
			error))
		goto error_free_loc;
	if (!smlManagerObjectRegister(
			manager, SML_COMMAND_TYPE_PUT, session, 
			NULL, devinf12, NULL, _recv_devinf, NULL, agent, 
			error))
		goto error_free_loc;

	/* GET callbacks
	 * 
	 * This command is used to request the device information of a
	 * remote peer. Usually the client do this if it is the first
	 * connection with this server or the client does not remember
	 * the server's device information. The most mobiles only do
	 * this for the first successful connection. Data costs usually
	 * high amounts of money in the mobile world and the bandwith is
	 * small in the mobile world.
	 *
	 * Nevertheless if a mobile does not perform the PUT command and
	 * the server does not know the device information of the client
	 * then the server can send a GET command and the client has to
	 * react on it. Please see above why this can happen.
	 *
	 * Please note that client and server must be able to handle the
	 * GET command. A RESULTS command must be send with the local
	 * device information.
	 */

	smlTrace(TRACE_INTERNAL, "register callbacks for GET command");
	if (!smlManagerObjectRegister(
			manager, SML_COMMAND_TYPE_GET, session, 
			devinf10, NULL, NULL, _request_devinf, NULL, agent, 
			error))
		goto error_free_loc;
	if (!smlManagerObjectRegister(
			manager, SML_COMMAND_TYPE_GET, session, 
			devinf11, NULL, NULL, _request_devinf, NULL, agent, 
			error))
		goto error_free_loc;
	if (!smlManagerObjectRegister(
			manager, SML_COMMAND_TYPE_GET, session, 
			devinf12, NULL, NULL, _request_devinf, NULL, agent, 
			error))
		goto error_free_loc;

	/* RESULTS callbacks
	 *
	 * This command is used to send the device information which
	 * was requested with a GET command. This command can be send by
	 * server and clients. Please see above when this happens.
	 *
	 * A server must be able to handle and interpret a RESULTS
	 * command from a client. Especially this is required because a
	 * server must explicitly request such a command with a previous
	 * GET command.
	 *
	 * Please note that a client can ignore a RESULTS from a server.
	 * Nevertheless it is recommended that a client does not ignore
	 * the device information of a server.
	 */

	smlTrace(TRACE_INTERNAL, "register callbacks for RESULTS command");
	if (!smlManagerObjectRegister(
			manager, SML_COMMAND_TYPE_RESULTS, session, 
			devinf10, NULL, NULL, _recv_devinf, NULL, agent, 
			error))
		goto error_free_loc;
	if (!smlManagerObjectRegister(
			manager, SML_COMMAND_TYPE_RESULTS, session,
			devinf11, NULL, NULL, _recv_devinf, NULL, agent,
		 	error))
		goto error_free_loc;
	if (!smlManagerObjectRegister(
			manager, SML_COMMAND_TYPE_RESULTS, session, 
			devinf12, NULL, NULL, _recv_devinf, NULL, agent, 
			error))
		goto error_free_loc;

	smlLocationUnref(devinf10);
	smlLocationUnref(devinf11);
	smlLocationUnref(devinf12);

	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error_free_loc:
	if (devinf10)
		smlLocationUnref(devinf10);
	if (devinf11)
		smlLocationUnref(devinf11);
	if (devinf12)
		smlLocationUnref(devinf12);
error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

SmlBool smlDevInfAgentRegister(SmlDevInfAgent *agent, SmlManager *manager, SmlError **error)
{
	smlTrace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, agent, manager, error);

	SmlBool retval = smlDevInfAgentRegisterSession(agent, manager, NULL, error);
	
	if (!retval)
		goto error;
	
	smlTrace(TRACE_EXIT, "%s", __func__);
	return TRUE;

error:
	smlTrace(TRACE_EXIT_ERROR, "%s: %s", __func__, smlErrorPrint(error));
	return FALSE;
}

/*@}*/
