/* MappedURLStreamHandlerFactory.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.protocol;

import java.net.URL;
import java.net.URLStreamHandler;
import java.net.URLStreamHandlerFactory;
import java.util.HashMap;
import java.util.Map;

/**
 * {@link URLStreamHandlerFactory} which allows run time registration of new handlers.
 * This is mainly useful in those cases where it is not easy to configure new
 * protocols the standard way, i.e., by means of the  
 * 'java.protocol.handler.pkgs' system property.
 * @see URL#URL(String,String,int,String)
 */
public class MappedURLStreamHandlerFactory implements URLStreamHandlerFactory {
    
    //
    private final Map<String,URLStreamHandler> map;

    public MappedURLStreamHandlerFactory() {
        this.map = new HashMap<String,URLStreamHandler> ();
    }

    /**
     * Return the handler associated wit the given protocol, or null
     * if a default handler should be tried
     */
    public URLStreamHandler createURLStreamHandler(String protocol) {
        return map.get(protocol);
    }

    /**
     * Register a handler for a new protocol.
     */
    public void registerHandler (String protocol,URLStreamHandler handler) {
        map.put(protocol, handler);
    }
    
    /**
     * Request that this factory be used before the default factory for resolving
     * URL's.
     */
    public void install () {
        URL.setURLStreamHandlerFactory(this);
    }
}
