#!NOMODULE
/*
	xmlrpc.pmod 

	(c) 1999 Martin Baehr
	(c) 2001 Karl Pitrich

	This program is free software; you can redistribute it and/or modify it *
	under the terms of the GNU General Public License as published by the   *
	Free Software Foundation; either version 2 of the License, or (at your  *
	option) any later version.     
	
	CHANGELOG:
		(pit, 2001-03-30):
			-fixed array type handling
			-class framework
		(pit, 2002-02-27):
			-make really XML-RPC2 compatible 
			-use POST
			-fix empty type handling
			-add base64 type (prefix string with 't_b64:')
			-add isodatetime type (prefix string with 't_iso:')
			-make string default type if not specified
*/

class xmlRPC {
	constant XML_HEADER = "<?xml version=\"1.0\"?>";
	constant cvs_version = "$Id: xmlrpc.pmod,v 1.13 2002/06/12 18:26:43 pit Exp $";

	constant string_replace_entities = ({ "&lt;","&gt;","&amp;", "&quot;" });
	constant string_replace_values = ({ "<",">","&", "\"" });


	public mixed http_call(string url, string f_name, void|mixed ... args) {

		object con = Protocols.HTTP.Query();
		object URI=Standards.URI(url);
		string request;
		if(args) {
			request = compose_call(f_name, @args);
		} else {
			request = compose_call(f_name);
		}

		mapping call_headers=([
				"user-agent" 	: "Mozilla/4.0 compatible (Pike XML-RPC client)",
				"host" 			: URI->host, 
				"content-type"	: "text/xml"
		]);

		string path=URI->path;
		if(path=="") path="/";

		#if constant(SSL.sslfile)
		if(URI->scheme!="http" && URI->scheme!="https")
			throw(({"Invalid protocol. only HTTP and HTTPS supported. ",backtrace()}));

		con->https= (URI->scheme=="https")?1:0;
		#else
		if(URI->scheme!="http")
			throw(({"Invalid protocol. only HTTP supported. ",backtrace()}));
		#endif

		if(URI->user || URI->passwd)
			call_headers->authorization = 
				"Basic " + MIME.encode_base64(URI->user + ":" 
				+ (URI->password || ""));

		object oResult;
		mixed err=catch {
			oResult = con->sync_request(URI->host, URI->port,
									"POST "+path+" HTTP/1.0", call_headers, request);
		};
		if(err) {
			throw(({"Network error. ",backtrace()}));
		}

		if(!objectp(oResult)) 
      	throw(({"XML-RPC request failed ", backtrace()}));
		
		string response = oResult->data();
		mapping mRes = parse(response);

		if (!mappingp(mRes)) {
			throw(({"Invalid XML-RPC response ",backtrace()}));
		}

		if (mRes["fault"]) {
			throw(({sprintf("XML-RPC fault %d - %s",
         		mRes["fault"]["faultCode"],mRes["fault"]["faultString"]),
					backtrace()})
			);
		}
		return mRes["params"][0];
	}

	public string compose_response(mixed m) {
		return (
			XML_HEADER +
			"<methodResponse><params><param>" +
			compose(m) +
			"</param></params></methodResponse>"
		);
	}

	public string compose_call(string f_name, void|mixed ... args) {
		string s_compose;
		int i,sz;
    
		if (!args || !arrayp(args)) {
			s_compose = "";
		} else {
			s_compose = "<params>";
			for (i=0,sz=sizeof(args);i<sz;i++) {
				s_compose += "\n<param>";
				s_compose += compose(args[i]);
				s_compose +="</param>";
			}
			s_compose += "</params>\n";
		}
    
		return (
			XML_HEADER +
			"<methodCall><methodName>" +
			f_name +
			"</methodName>" +
			s_compose +
			"</methodCall>"
		);
	}

	public string compose_fault(int num, string msg) {
		mapping fault;

		fault = ([	"faultCode" : num,
						"faultString" : msg
		]);
		return (
			XML_HEADER +
			"<methodResponse><fault>" +
			compose(fault) +
			"</fault></methodResponse>"
		);
	}


	private mixed process_element(string type, string name, mapping attributes,
                     string|array data, mixed ... extra_args)
	{                        
		mapping out;

		if(type=="error") return 0;

		if(name) {
			out = ([ "name":name ]);
			if(data && data != ({}) ) {
				out += ([ "data":data ]);
			}
			return out;
		} else {
			if( stringp(data) && (data - "\n")-" " != "") {
				return data;
			}
		}
		return 0;
	}

	private mixed flatten(mapping m) {
		if (!mappingp(m)) {
         throw(({"Invalid XML-RPC response ",backtrace()}));
      }

		mapping member;
		string name = m["name"];
	
		if(sizeof(m) == 1) { // empty element fix
			m->data=({""});
		}	

		mixed data = m["data"];
		mixed result;

		if (name == "methodCall" || name == "methodResponse" ) {
			result = ([]);
			foreach (data, mapping r) {
				array(string) data_index;
				data_index = indices(r);
				string rname = r["name"];
				if(rname == "methodName") {
					result[rname] = r["data"][0];
				} else {
					result[rname] = flatten(r);
				}
			}
			return result;
		} 

		switch(name) {
			case "i4":
			case "int":
				return (int)data[0];
			case "dateTime.iso8601":
			case "string":
				return replace( (string)data[0], 
							string_replace_entities,
							string_replace_values);
			case "double":
				return (float)data[0];
			case "base64":
				return MIME.decode_base64(data[0]);
			case "struct":
				mapping struct_map = ([ ]);
				foreach(data, member) {
					result = flatten(member);
					struct_map[result[0]] = result[1];
				}
				return struct_map;
			case "member":
				if (data[0]["name"] == "name") {
					 return ({ data[0]["data"][0], flatten(data[1]["data"][0]) });
				} else {
					return ({ data[1]["data"][0], flatten(data[0]["data"][0]) });
				}
			case "value":
				if(sizeof(data) == 1 && stringp(data[0])) { //default type string
					return flatten( ([ "name" : "string", "data" : data ]) );
				}
			case "array":
				return flatten(data[0]);
			case "data":
			case "params":
				result=({});
				foreach(data, member) {
					result += ({ flatten(member) });
				}
				return result;
			case "param":
			case "fault":
				return flatten(data[0]);
		}
	}

	public mixed parse(string s) {
		s-="\n";
		s-="\t";
		s-="\r";
		array result = spider.XML()->parse(s, process_element);
		mixed res2 = flatten(result[0]);
		return res2;
	}
	
	private string compose_scalar(mixed s) {
		if (intp(s))
			return "<value><int>" + (string)s + "</int></value>\n";
		if (floatp(s))
			return "<value><double>" + s +"</double></value>\n";
		if (stringp(s)) {
				if(s[0..] == "t_b64:") {
					return "<value><base64>" + s + "</base64></value>\n";
				} else if (s[0..] == "t_iso:") {
					return "<value><dateTime.iso8601>" + s + "</dateTime.iso8601></value>\n";

				} else {
					return "<value><string>"  
						+ replace(s, string_replace_values, string_replace_entities)
						+"</string></value>\n";
				}
		}
	}

	private string compose_array(array a) {
		int i, sz;
		string s_compose;
    
		s_compose = "<value><array><data>";
		for (i=0,sz = sizeof(a);i<sz;i++) {
			s_compose += compose(a[i]);
		}
		s_compose += "</data></array></value>";
		return s_compose;
	}

	private string compose_struct(mapping m) {
		int i,sz;
		string s_compose;
		array ind, val;
		ind = indices(m);
		val = values(m);

		s_compose = "<value><struct>";
		for (i=0,sz=sizeof(ind);i<sz;i++) {
			s_compose += "<member>";
			s_compose += "<name>" + (string)ind[i] + "</name>";
			s_compose += compose(val[i]);
			s_compose += "</member>\n";
		}
		s_compose +="</struct></value>";
		return s_compose;
	}

	private string compose(mixed m) {
		if (stringp(m) || intp(m) || floatp(m))
			return compose_scalar(m);
		if (mappingp(m))
			return compose_struct(m);
		if (arrayp(m))
			return compose_array(m);
	}


};



/* vi: set ts=3: */
