#include "osl/pieceStand.h"
#include "osl/ptypeTable.h"
#include "osl/state/numEffectState.h"
#include "osl/misc/random.h"
#include "osl/record/csaRecord.h"
#include "osl/apply_move/applyMove.h"
#include "osl/oslConfig.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

#include <fstream>
#include <iostream>

using namespace osl;

class PieceStandTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(PieceStandTest);
  CPPUNIT_TEST(testCarry);
  CPPUNIT_TEST(testSet);
  CPPUNIT_TEST(testSuperior);
  CPPUNIT_TEST(testAddSubAtmostOnePiece);
  CPPUNIT_TEST(testMax);
  CPPUNIT_TEST(testNextPrev);
  CPPUNIT_TEST_SUITE_END();
public:
  void testCarry();
  void testSet();
  void testSuperior();
  void testAddSubAtmostOnePiece();
  bool isMax(PieceStand l, PieceStand r, PieceStand m)
  {
    for (unsigned int i=0; i<PieceStand::order.size(); ++i)
    {
      const Ptype ptype = PieceStand::order[i];
      if (std::max(l.get(ptype), r.get(ptype)) != m.get(ptype))
	return false;
    }
    return true;
  }
  void testMax();
  void testMaxRecursive(PieceStand, unsigned int ptype_index);
  void testMaxRecursive2(PieceStand, PieceStand, unsigned int ptype_index);
  void testNextPrev();
};

CPPUNIT_TEST_SUITE_REGISTRATION(PieceStandTest);
extern bool isShortTest;

void PieceStandTest::testMax() 
{
  const PieceStand l(12,1,2,3,4,1,0,0);
  const PieceStand r(8,3,0,4,2,0,2,0);
  const PieceStand m = l.max(r);
  CPPUNIT_ASSERT(isMax(l, r, m));

  PieceStand empty_stand;
  testMaxRecursive(empty_stand, 0);
}

void PieceStandTest::
testMaxRecursive(PieceStand stand, unsigned int ptype_index)
{
  using osl::random;
  if (ptype_index < PieceStand::order.size())
  {
    const Ptype ptype = PieceStand::order[ptype_index];
    const int max_pieces
      = Ptype_Table.getIndexLimit(ptype) - Ptype_Table.getIndexMin(ptype);
    for (int i=0; i<max_pieces; ++i)
    {
      PieceStand stand_copy(stand);
      stand_copy.add(ptype, i);
      testMaxRecursive(stand_copy, ptype_index+1);
    }
    return;
  }
  for (unsigned int i=0; i<(isShortTest ? 8 : 16); ++i)
  {
    const PieceStand target(random() % 19, random() % 5, random() % 5,
			    random() % 5, random() % 5, random() % 3, 
			    random() % 3, random() % 3);
    const PieceStand ml = stand.max(target);
    const PieceStand ml2 = stand.max2(target);
    CPPUNIT_ASSERT(isMax(stand, target, ml));
    CPPUNIT_ASSERT(isMax(stand, target, ml2));
    
    const PieceStand mr = target.max(stand);
    const PieceStand mr2 = target.max2(stand);
    CPPUNIT_ASSERT(isMax(stand, target, mr));
    CPPUNIT_ASSERT(isMax(stand, target, mr2));
  }
  // SLOOOW!!
  // testMaxRecursive2(stand, empty_stand, 0);
}

void PieceStandTest::
testMaxRecursive2(PieceStand l, PieceStand r, unsigned int ptype_index)
{
  if (ptype_index < PieceStand::order.size())
  {
    const Ptype ptype = PieceStand::order[ptype_index];
    const int max_pieces
      = Ptype_Table.getIndexLimit(ptype) - Ptype_Table.getIndexMin(ptype);
    for (int i=0; i<max_pieces; ++i)
    {
      PieceStand stand_copy(r);
      stand_copy.add(ptype, i);
      testMaxRecursive2(l, stand_copy, ptype_index+1);
    }
    return;
  }
  const PieceStand m = l.max(r);
  CPPUNIT_ASSERT(isMax(l, r, m));
}

void PieceStandTest::testCarry()
{
  unsigned int flag=0;
  flag |= (1<<30);
  flag |= (1<<27);
  flag |= (1<<23);
  flag |= (1<<17);
  flag |= (1<<13);
  flag |= (1<<9);
  flag |= (1<<5);
  flag |= (1<<2);
  CPPUNIT_ASSERT_EQUAL(flag, PieceStand::carryMask);

  PieceStand pieces;
  pieces.carriesOn();
  for (int i=KING; i<=PTYPE_MAX; ++i)
    CPPUNIT_ASSERT_EQUAL(0u, pieces.get((Ptype)i));
  pieces.carriesOff();
  for (int i=KING; i<=PTYPE_MAX; ++i)
    CPPUNIT_ASSERT_EQUAL(0u, pieces.get((Ptype)i));
}


void PieceStandTest::testSet() 
{
  PieceStand pieces;
  for (int i=KING; i<=PTYPE_MAX; ++i)
  {
    const Ptype pi = (Ptype)i;
    const unsigned int num =	// 持駒になる最大の個数
      Ptype_Table.getIndexLimit(pi) - Ptype_Table.getIndexMin(pi);
    CPPUNIT_ASSERT_EQUAL(0u, pieces.get(pi));
    pieces.add(pi, num);
    for (int j=KING; j<=PTYPE_MAX; ++j)
    {
      const Ptype pj = (Ptype)j;
      if (i==j)
	CPPUNIT_ASSERT_EQUAL(num, pieces.get(pj));
      else
	CPPUNIT_ASSERT_EQUAL(0u, pieces.get(pj));
    }
    pieces.sub(pi, num);
    CPPUNIT_ASSERT_EQUAL(0u, pieces.get(pi));
  }
}

void PieceStandTest::testSuperior() 
{
  PieceStand p1, p2;
  CPPUNIT_ASSERT(p1.isSuperiorOrEqualTo(p2));
  CPPUNIT_ASSERT(p2.isSuperiorOrEqualTo(p1));
  
  p1.add(PAWN, 3);

  CPPUNIT_ASSERT(p1.isSuperiorOrEqualTo(p2));
  CPPUNIT_ASSERT(! p2.isSuperiorOrEqualTo(p1));

  CPPUNIT_ASSERT_EQUAL(3u, p1.get(PAWN));

  p2.add(GOLD, 1);
  
  CPPUNIT_ASSERT(! p1.isSuperiorOrEqualTo(p2));
  CPPUNIT_ASSERT(! p2.isSuperiorOrEqualTo(p1));

  p1.add(GOLD, 1);

  CPPUNIT_ASSERT(p1.isSuperiorOrEqualTo(p2));
  CPPUNIT_ASSERT(! p2.isSuperiorOrEqualTo(p1));
}

void PieceStandTest::testAddSubAtmostOnePiece() 
{
  PieceStand p1;
  PieceStand   pPawn(1,0,0,0,0,0,0,0);
  PieceStand  pLance(0,1,0,0,0,0,0,0);
  PieceStand pKnight(0,0,1,0,0,0,0,0);
  PieceStand pSilver(0,0,0,1,0,0,0,0);
  PieceStand   pGold(0,0,0,0,1,0,0,0);
  PieceStand pBishop(0,0,0,0,0,1,0,0);
  PieceStand   pRook(0,0,0,0,0,0,1,0);
  PieceStand   pKing(0,0,0,0,0,0,0,1);

  p1.addAtmostOnePiece(pPawn);
  p1.addAtmostOnePiece(pKnight);
  p1.addAtmostOnePiece(pGold);
  p1.addAtmostOnePiece(pRook);

  CPPUNIT_ASSERT_EQUAL(PieceStand(1,0,1,0,1,0,1,0),p1);

  p1.addAtmostOnePiece(pPawn);
  p1.addAtmostOnePiece(pLance);
  p1.addAtmostOnePiece(pKnight);
  p1.addAtmostOnePiece(pSilver);

  CPPUNIT_ASSERT_EQUAL(PieceStand(2,1,2,1,1,0,1,0),p1);

  p1.addAtmostOnePiece(pPawn);
  p1.addAtmostOnePiece(pLance);
  p1.addAtmostOnePiece(pGold);
  p1.addAtmostOnePiece(pBishop);

  CPPUNIT_ASSERT_EQUAL(PieceStand(3,2,2,1,2,1,1,0),p1);

  p1.addAtmostOnePiece(pPawn);
  p1.addAtmostOnePiece(pLance);
  p1.addAtmostOnePiece(pKnight);
  p1.addAtmostOnePiece(pSilver);
  p1.addAtmostOnePiece(pGold);
  p1.addAtmostOnePiece(pBishop);
  p1.addAtmostOnePiece(pRook);
  p1.addAtmostOnePiece(pKing);

  CPPUNIT_ASSERT_EQUAL(PieceStand(4,3,3,2,3,2,2,1),p1);

  p1.subAtmostOnePiece(pPawn);
  p1.subAtmostOnePiece(pKnight);
  p1.subAtmostOnePiece(pGold);
  p1.subAtmostOnePiece(pRook);

  CPPUNIT_ASSERT_EQUAL(PieceStand(3,3,2,2,2,2,1,1),p1);

  p1.subAtmostOnePiece(pPawn);
  p1.subAtmostOnePiece(pLance);
  p1.subAtmostOnePiece(pKnight);
  p1.subAtmostOnePiece(pSilver);

  CPPUNIT_ASSERT_EQUAL(PieceStand(2,2,1,1,2,2,1,1),p1);

  p1.subAtmostOnePiece(pPawn);
  p1.subAtmostOnePiece(pLance);
  p1.subAtmostOnePiece(pGold);
  p1.subAtmostOnePiece(pBishop);

  CPPUNIT_ASSERT_EQUAL(PieceStand(1,1,1,1,1,1,1,1),p1);

  p1.subAtmostOnePiece(pPawn);
  p1.subAtmostOnePiece(pLance);
  p1.subAtmostOnePiece(pKnight);
  p1.subAtmostOnePiece(pSilver);
  p1.subAtmostOnePiece(pGold);
  p1.subAtmostOnePiece(pBishop);
  p1.subAtmostOnePiece(pRook);
  p1.subAtmostOnePiece(pKing);

  CPPUNIT_ASSERT_EQUAL(PieceStand(0,0,0,0,0,0,0,0),p1);

}

void PieceStandTest::testNextPrev() 
{
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  std::string file_name;
  for (int i=0;i<(isShortTest ? 10 : 900) && (ifs >> file_name) ; i++)
  {
    if ((i % 100) == 0)
      std::cerr << '.';
    if (file_name == "") 
      break;
    file_name = OslConfig::testCsaFile(file_name);

    const Record record=CsaFile(file_name).getRecord();
    const vector<osl::Move> moves=record.getMoves();

    SimpleState state(record.getInitialState());
    PieceStand black(BLACK, state);
    PieceStand white(WHITE, state);
    
    for (unsigned int i=0; i<moves.size(); i++)
    {
      const Move m = moves[i];
      ApplyMoveOfTurn::doMove(state, m);

      PieceStand black_new(BLACK, state);
      PieceStand white_new(WHITE, state);

      CPPUNIT_ASSERT_EQUAL(black_new, black.nextStand(BLACK, m));
      CPPUNIT_ASSERT_EQUAL(white_new, white.nextStand(WHITE, m));

      CPPUNIT_ASSERT_EQUAL(black, black_new.previousStand(BLACK, m));
      CPPUNIT_ASSERT_EQUAL(white, white_new.previousStand(WHITE, m));

      black = black_new;
      white = white_new;
    }
  }
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
