/* safeFilter.t.cc
 */
#include "osl/move_action/safeFilter.h"
#include "osl/move_action/store.h"
#include "osl/move_generator/allMoves.h"
#include "osl/move_generator/escape_.h"
#include "osl/record/csaString.h"
#include "osl/record/csaRecord.h"
#include "osl/state/numEffectState.h"
#include "osl/container/moveVector.h"
#include "osl/apply_move/applyMove.h"
#include "osl/oslConfig.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <string>
#include <fstream>
#include <iostream>

class SafeFilterTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(SafeFilterTest);
  CPPUNIT_TEST(testSuicide);
  CPPUNIT_TEST(testFile);
  CPPUNIT_TEST_SUITE_END();
public:
  void testFile();
  void testSuicide();
};
CPPUNIT_TEST_SUITE_REGISTRATION(SafeFilterTest);

using namespace osl;
using namespace osl::move_action;
using namespace osl::move_generator;

bool isSuicide(const NumEffectState& state_org, Move move)
{
  const Player turn = move.player();
  NumEffectState state = state_org;
  ApplyMoveOfTurn::doMove(state, move);
  const Position king = state.getKingPosition(turn);
  return state.hasEffectBy(alt(turn), king);
}


void SafeFilterTest::testSuicide()
{
  NumEffectState state=CsaString(
    "P1 *  *  *  *  *  *  * -KE-OU\n"
    "P2 *  *  *  *  *  * -FU-FU-KY\n"
    "P3 *  *  *  *  *  *  * +KE * \n"
    "P4 *  *  *  *  * +HI *  *  * \n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6 *  *  *  *  *  *  *  *  * \n"
    "P7 *  *  *  *  *  *  *  *  * \n"
    "P8 *  *  *  *  *  *  *  *  * \n"
    "P9 *  * +OU *  *  *  *  *  * \n"
    "P+00KE\n"
    "P-00AL\n"
    "-\n").getInitialState();
  const Move m23fu = Move(Position(2,2), Position(2,3), PAWN,
			     KNIGHT, false, WHITE);
  CPPUNIT_ASSERT(state.isValidMove(m23fu));
  CPPUNIT_ASSERT(! isSuicide(state, m23fu));

  NumEffectState state2=CsaString(
    "P1 *  *  *  *  *  *  * -KE-OU\n"
    "P2 *  *  *  *  *  * -FU-FU-KY\n"
    "P3 *  *  *  *  *  *  * +KE * \n"
    "P4 *  *  *  *  * +KA *  *  * \n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6 *  *  *  *  *  *  *  *  * \n"
    "P7 *  *  *  *  *  *  *  *  * \n"
    "P8 *  *  *  *  *  *  *  *  * \n"
    "P9 *  * +OU *  *  *  *  *  * \n"
    "P+00KE\n"
    "P-00AL\n"
    "-\n").getInitialState();
  CPPUNIT_ASSERT(state2.isValidMove(m23fu));
  CPPUNIT_ASSERT(isSuicide(state2, m23fu));
}


static void testFile(const std::string& fileName)
{
  Record rec=CsaFile(fileName).getRecord();
  NumEffectState state=rec.getInitialState();
  const vector<osl::Move> moves=rec.getMoves();
  for(size_t i=0;i<moves.size();i++)
  {
    MoveVector allMoves;

    const Player player = state.getTurn();
    const Piece kingPiece=state.getKingPiece(player);
    const Position kingPosition=kingPiece.position();
    if (state.hasEffectBy(alt(player), kingPosition))
    {
      Store store(allMoves);
      if (player == BLACK)
	move_generator::Escape<Store>::
	  generate<BLACK,true,false>(state, kingPiece, store);
      else
	move_generator::Escape<Store>::
	  generate<WHITE,true,false>(state, kingPiece, store);
    }
    else
    {
      Store store(allMoves);
      AllMoves<Store>::
	generate(state.getTurn(),state,store);
    }
    
    for (size_t j=0; j<allMoves.size(); ++j)
    {
      const bool safe = isSafeMove(state, allMoves[j]);
      const bool suicide = isSuicide(state, allMoves[j]);
      if (suicide == safe)
	std::cerr << state << allMoves[j] << "\n"
		  << "suicide " << suicide << " safe " << safe << "\n";
      CPPUNIT_ASSERT_EQUAL(!suicide, safe);
    }

    Move move=moves[i];
    ApplyMoveOfTurn::doMove(state, move);
  }
}

void SafeFilterTest::testFile()
{
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=100;
  if (isShortTest) 
      count=10;
  std::string fileName;
  while((ifs >> fileName) && (++i<count))
  {
    if (! isShortTest)
      std::cerr << fileName << " ";
    ::testFile(OslConfig::testCsaFile(fileName));
  }
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
