/***************************************************************************
 $RCSfile: file.cpp,v $
 -------------------
 cvs         : $Id: file.cpp,v 1.3 2002/08/02 09:29:30 cstim Exp $
 begin       : Fri Dec 14 2001
 copyright   : (C) 2001 by Martin Preuss
 email       : martin@aquamaniac.de
 */


/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/*
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#ifdef __declspec
# if BUILDING_DLL
#  define DLLIMPORT __declspec (dllexport)
# else /* Not BUILDING_DLL */
#  define DLLIMPORT __declspec (dllimport)
# endif /* Not BUILDING_DLL */
#else
# define DLLIMPORT
#endif


#include "error.h"
#include "file.h"


namespace HBCI {

File::File(string name)
:_handle(FILE_NO_FILEHANDLE)
,_name(_convPath(name))
{
};


File::~File(){
}


void File::operator=(File &f){
  _name=f._name;
  _handle=FILE_NO_FILEHANDLE;
}


string File::_convPath(const string &p) const{
    unsigned int pos=0;
	string pout;

    while(pos<p.length()) {
      if (p.at(pos)=='/')
		  pout+='\\';
	  else
		  pout+=p[pos];
      pos++;
    }
	return pout;
}


string File::_lastErrorString() const{
	LPVOID lpMsgBuf;

	FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | 
				  FORMAT_MESSAGE_FROM_SYSTEM | 
				  FORMAT_MESSAGE_IGNORE_INSERTS,
				  NULL,
				  GetLastError(),
                  MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
                  (LPTSTR)&lpMsgBuf,
				  0,NULL);
	return (LPTSTR)lpMsgBuf;
}


Error File::accessFile(int mode) const {
    int am;
	HANDLE h;
	string fn;

    am=0;
    if (mode & FILE_ACCESS_READ) am|=GENERIC_READ;
    if (mode & FILE_ACCESS_WRITE) am|=GENERIC_WRITE;
    if (mode & FILE_ACCESS_RW) am|=GENERIC_READ | GENERIC_WRITE;
	
	h=CreateFile(_name.c_str(),
		am,
		FILE_SHARE_READ | FILE_SHARE_WRITE,
		NULL,
		OPEN_EXISTING,
		FILE_ATTRIBUTE_NORMAL,
		NULL);
	if (h==FILE_NO_FILEHANDLE)
        return Error("File::accessFile()",
                       ERROR_LEVEL_NORMAL,
                       0,
                       ERROR_ADVISE_DONTKNOW,
                       _lastErrorString(),
                       "error on CreateFile() "+_name);
	if (!CloseHandle(h))
        return Error("File::accessFile()",
                       ERROR_LEVEL_NORMAL,
                       0,
                       ERROR_ADVISE_DONTKNOW,
                       _lastErrorString(),
                       "error on CloseHandle() "+_name);
    return Error();
}


Error File::openFile(int access,
                           int mode){
  int am;
  int om;
  int fatt;

  am=0;
  om=0;
  fatt=0;
  if (access & FILE_AM_RDONLY) om=GENERIC_READ;
  if (access & FILE_AM_WRONLY) om=GENERIC_WRITE;
  if (access & FILE_AM_RDWR) om=GENERIC_READ | GENERIC_WRITE;

  switch (mode & FILE_INTERNAL_AM_MASK) {
    case FILE_AM_CREATE_NEW:
        am|=CREATE_NEW;
        break;
    case FILE_AM_TRUNCATE_EXISTING:
        am|=TRUNCATE_EXISTING;
        break;
    case FILE_AM_OPEN_EXISTING:
		am|=OPEN_EXISTING;
        break;
    case FILE_AM_OPEN_ALWAYS:
        am|=OPEN_ALWAYS;
        break;
    case FILE_AM_CREATE_ALWAYS:
		am|=CREATE_ALWAYS;
        break;
    default:
        return Error("File::accessFile()",
                       ERROR_LEVEL_NORMAL,
                       0,
                       ERROR_ADVISE_DONTKNOW,
                       "unknown access mode "+_name);
        break;
  } // switch

  if (mode & FILE_CM_WRITE)
	fatt=FILE_ATTRIBUTE_NORMAL;
  else
	fatt=FILE_ATTRIBUTE_READONLY;

  _handle=CreateFile(_name.c_str(),
	  om,
	  FILE_SHARE_READ | FILE_SHARE_WRITE,
	  NULL,
	  am,
	  fatt,NULL);
  if (_handle==FILE_NO_FILEHANDLE)
      return Error("File::openFile()",
                     ERROR_LEVEL_NORMAL,
                     0,
                     ERROR_ADVISE_DONTKNOW,
                     _lastErrorString(),
                     "error on open() "+_name);
  return Error();
}


Error File::closeFile(){
  if (!CloseHandle(_handle))
    return Error("File::closeFile()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   _lastErrorString(),
                   "error on close() "+_name);
  _handle=FILE_NO_FILEHANDLE;
  return Error();
}


Error File::createFile(int access,
                           int mode){
  return openFile(access,mode);
}


Error File::statFile(s_filestat &s){
  HANDLE h;
  WIN32_FIND_DATA wd;
  unsigned long long t;
  SYSTEMTIME st;

  h=FindFirstFile(_name.c_str(),&wd);
  if (_handle==FILE_NO_FILEHANDLE)
    return Error("File::statFile()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   _lastErrorString(),
                   "error on FindFirstFile() "+_name);
  t=wd.nFileSizeHigh;
  t=t<<32;
  t+=wd.nFileSizeLow;
  s.size=t;
  s.mode=0;
  // access time
  FileTimeToSystemTime(&(wd.ftLastAccessTime),&st);
  s.atime=DateTime(st);
  // modification time
  FileTimeToSystemTime(&wd.ftLastWriteTime,&st);
  s.mtime=DateTime(st);
  // change time
  FileTimeToSystemTime(&wd.ftCreationTime,&st);
  s.ctime=DateTime(st);
  FindClose(h);
  return Error();
}


Error File::renameFile(string newname){
  string nn;

  nn=_convPath(newname);

  if (!MoveFile(_name.c_str(),nn.c_str()))
    return Error("File::renameFile()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   _lastErrorString(),
                   "error on rename()"+_name+" to "+nn);
  _name=nn;
  return Error();
}


Error File::deleteFile(){
  if (!DeleteFile(_name.c_str()))
    return Error("File::deleteFile()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   _lastErrorString(),
                   "error on delete() "+_name);
  return Error();
}


Error File::lockFile(int mode,
                         t_offset start,
                         t_offset length,
                         bool wait){
	bool b;

	if (!wait || mode!=FILE_LOCK_EXCLUSIVE) {
		DWORD fl;
		OVERLAPPED ov;

		fl=0;
		ov.Internal=0;
		ov.InternalHigh=0;
		ov.Offset=start;
		ov.OffsetHigh=0;
		ov.hEvent=FILE_NO_FILEHANDLE;
		if (!wait)
			fl|=LOCKFILE_FAIL_IMMEDIATELY;
		if (mode==FILE_LOCK_EXCLUSIVE)
			fl|=LOCKFILE_EXCLUSIVE_LOCK;
		b=LockFileEx(_handle,
			fl,
			0,
			length,
			0,
			&ov);
	}
	else
		b=LockFile(_handle,start,0,length,0);

	if (!b)
    return Error("File::lockFile()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   _lastErrorString(),
                   "error on lock() "+_name);
  return Error();
}


Error File::unlockFile(t_offset start,
                           t_offset length){
	if (!UnlockFile(_handle,start,0,length,0))
    return Error("File::unlockFile()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   _lastErrorString(),
                   "error on UnlockFile() "+_name);
  return Error();
}


Error File::filePos(t_offset &p){
	p=SetFilePointer(_handle,
		0,
		NULL,
		FILE_CURRENT);
	if (p==0xffffffff)
    return Error("File::filePos()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   _lastErrorString(),
                   "error on SetFilePointer() "+_name);
  return Error();
}


Error File::setFilePos(t_offset p, int whence){
	if (SetFilePointer(_handle,
		p,
		NULL,
		whence)==0xffffffff)
    return Error("File::setFilePos()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   _lastErrorString(),
                   "error on SetFilePointer() "+_name);
  return Error();
}


Error File::changeMode(int mode){
    return Error("File::changeMode()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   "not implemented");
  return Error();
}


Error File::readData(string &data, unsigned int size){
  DWORD bytesread;
  char buffer[8192];

  // Now read the bytes
  if (size>sizeof(buffer))
    size=sizeof(buffer);

  if (!ReadFile(_handle,buffer,size,&bytesread,NULL))
    return Error("File::readData()",
                   ERROR_LEVEL_NORMAL,
                   0,
                   ERROR_ADVISE_DONTKNOW,
                   _lastErrorString(),
                   "error on ReadFile() "+_name);
  data.assign(buffer,bytesread);
  return Error();
}


Error File::writeData(string &buffer){
  int bytesleft;
  DWORD byteswritten;
  const char *bytespos;

  bytesleft=buffer.length();
  bytespos=buffer.c_str();

  while (bytesleft) {
    // Now write the bytes
    if (!WriteFile(_handle,bytespos,bytesleft,&byteswritten,NULL))
      return Error("File::writeData()",
                     ERROR_LEVEL_NORMAL,
                     0,
                     ERROR_ADVISE_DONTKNOW,
                     _lastErrorString(),
                     "error on WriteFile() "+_name);
    if (!byteswritten)
      return Error("File::writeData()",
                     ERROR_LEVEL_NORMAL,
                     0,
                     ERROR_ADVISE_DONTKNOW,
                     "Could not write bytes",
                     "no bytes written on write() "+_name);
    bytesleft-=byteswritten;
    bytespos+=byteswritten;
  } // while bytesleft
  return Error();
}

} /* namespace  HBCI */
