(** A table object displaying data in scrollable grid. **)

MODULE VO:TableView;

(*
    A table object displaying data in scrollable grid.
    Copyright (C) 1998  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D   := VO:Base:Display,
       E   := VO:Base:Event,
       F   := VO:Base:Frame,
       O   := VO:Base:Object,
       U   := VO:Base:Util,

       TM  := VO:Model:Table,

       G   := VO:Object,

       str := Strings;

CONST
  textOffset = 2; (*
                     If a header is drawn, should be the same as the left width
                     of the frame object used for drawing a header cell
                   *)

  (* Events *)
  mouseSelectionMsg* = 0;
  doubleClickMsg*    = 1;


TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 focusFrame* : LONGINT;
               END;

  (**
    Object handed to the RedrawCell/RedrawRow methods.
  **)

  DrawInfoDesc* = RECORD
                    y,
                    height,
                    row     : LONGINT;
                    draw    : D.DrawInfo;
                    font    : D.Font;
                  END;

  Table*     = POINTER TO TableDesc;
  TableDesc* = RECORD (G.ScrollableGadgetDesc)
                 focusFrame   : F.Frame;

                 model-       : TM.TableModel;

                 totalWidth,
                 rowHeight    : LONGINT;
               END;

  MouseSelectionMsg*     = POINTER TO MouseSelectionMsgDesc;
  MouseSelectionMsgDesc* = RECORD (O.MessageDesc)
                           END;

  DoubleClickMsg*     = POINTER TO DoubleClickMsgDesc;
  DoubleClickMsgDesc* = RECORD (O.MessageDesc)
                           END;

VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.none;
    p.focusFrame:=F.dottedFocus;
  END Init;

  PROCEDURE (t : Table) Init*;

  BEGIN
    t.Init^;

    t.SetBackground(D.tableBackgroundColor);

    t.SetPrefs(prefs);

    t.rowHeight:=0;

    t.SetFlags({G.canFocus}); (* We can show a focus frame *)
    t.RemoveFlags({G.stdFocus}); (* we do the displaying of the focus frame ourself *)

    NEW(t.focusFrame);
    t.focusFrame.Init;
    t.focusFrame.SetFrame(t.prefs(Prefs).focusFrame);

    NEW(t.vAdjustment);
    t.vAdjustment.Init;

    NEW(t.hAdjustment);
    t.hAdjustment.Init;

    t.AttachModel(t.vAdjustment.GetTopModel());
    t.AttachModel(t.hAdjustment.GetTopModel());

    t.model:=NIL;
  END Init;

  PROCEDURE (t : Table) SetRowHeight*(height : LONGINT);

  BEGIN
    t.rowHeight:=height;
  END SetRowHeight;

  PROCEDURE (t : Table) CalcSize*;

  VAR
    font  : D.Font;

  BEGIN
    t.width:=10*D.display.spaceWidth;
    t.height:=10*D.display.spaceHeight;

    t.minWidth:=t.width;
    t.minHeight:=t.height;

    font:=D.display.GetFont(D.normalFont);

    IF t.rowHeight=0 THEN
      t.rowHeight:=font.height+t.focusFrame.minHeight;
    END;

    t.CalcSize^;
  END CalcSize;

  (**
    Caculate width and height of the columns. Create a textobject for
    column header if necessary.
  **)

  PROCEDURE (t : Table) CalcCol;

  VAR
    x : LONGINT;

  BEGIN
    t.totalWidth:=0;
    IF t.model#NIL THEN
      FOR x:=0 TO t.model.GetColumns()-1 DO
        INC(t.totalWidth,t.model.GetColumnWidth(x));
      END;
    END;
  END CalcCol;

  PROCEDURE (t : Table) DrawRow(VAR info : DrawInfoDesc);

  VAR
    x,column,
    columnWidth : LONGINT;
    text        : U.Text;
    object      : G.Object;
    extent      : D.FontExtentDesc;
    rowSelect,
    cSelect     : BOOLEAN;

  BEGIN
    x:=t.x-t.hAdjustment.GetTop()+1;

    rowSelect:=(t.model.GetSelectionType()=TM.singleLineSelect)
               & (t.model.IsRowSelected(info.row));


    FOR column:=1 TO t.model.GetColumns() DO
      columnWidth:=t.model.GetColumnWidth(column-1);
      IF ~((x>=t.x+t.width) OR (x+columnWidth<t.x)) THEN
        text:=t.model.GetText(column,info.row);
        IF text=NIL THEN
          object:=t.model.GetObject(column,info.row);
        END;

        cSelect:=~rowSelect & (t.model.GetSelectionType()=TM.cellSelect)
                 & t.model.IsCellSelected(column,info.row);

        IF rowSelect OR cSelect THEN
          info.draw.PushForeground(D.fillColor);
          info.draw.FillRectangle(x,info.y,columnWidth,info.height);
          info.draw.PopForeground;
        ELSE
          t.DrawBackground(x,info.y,columnWidth,info.height);
        END;

        IF text#NIL THEN
          IF rowSelect OR cSelect THEN
            info.draw.PushForeground(D.fillTextColor);
          ELSE
            info.draw.PushForeground(D.tableTextColor);
          END;
          info.font.TextExtent(text^,str.Length(text^),{},extent);
          info.draw.DrawString(x-extent.lbearing+textOffset,
                               info.y+info.font.ascent+(info.height-info.font.height) DIV 2,
                               text^,str.Length(text^));

          info.draw.PopForeground;
        ELSIF object#NIL THEN
          IF ~(G.inited IN object.flags) OR (object.GetWindow()#t.GetWindow()) THEN
            object.SetParent(t);
            object.CalcSize;
          END;

          object.MoveResize(x+textOffset,
                            info.y+(info.height-object.oHeight) DIV 2,
                            columnWidth,info.height);
          object.Draw(t.oX,t.oY,t.oWidth,t.oHeight);
        END;

        IF cSelect & t.HasFocus() THEN
          (* Draw a frame arond one cell *)
          t.focusFrame.Draw(info.draw,
                            U.MaxLong(x,t.x),info.y,
                            U.MinLong(columnWidth,t.x+t.width-U.MaxLong(x,t.x)),info.height);
        END;

      END;
      INC(x,t.model.GetColumnWidth(column-1));
    END;

    IF x<t.x+t.width THEN
      (* Fill space behind last cell in row *)
      IF rowSelect THEN
        info.draw.PushForeground(D.fillColor);
        info.draw.FillRectangle(x,info.y,t.x+t.width-x+1,info.height);
        info.draw.PopForeground;
      ELSE
        t.DrawBackground(x,info.y,t.x+t.width-x+1,info.height);
      END;
    END;

    IF rowSelect & t.HasFocus() THEN
      (* Draw a frame all over the table width *)
      t.focusFrame.Draw(info.draw,t.x,info.y,t.width,info.height);
    END;
  END DrawRow;

  (**
    Redraw the given row if visible.
  **)

  PROCEDURE (t : Table) RedrawRow(row : LONGINT);

  VAR
    lastRow : LONGINT;
    draw    : D.DrawInfo;
    info    : DrawInfoDesc;

  BEGIN
    IF t.model#NIL THEN
      draw:=t.GetDrawInfo();

      lastRow:=U.MinLong(t.vAdjustment.GetTop()+t.vAdjustment.GetVisible(),t.model.GetRows());

      IF (row<t.vAdjustment.GetTop()) OR (row>lastRow) THEN
        RETURN;
      END;

      (* Clip complete content of table *)
      draw.InstallClip(t.x,t.y,t.width,t.height);

      info.font:=D.display.GetFont(D.normalFont);
      info.draw:=draw;
      info.y:=t.y+row*t.rowHeight-t.vAdjustment.GetTop()*t.rowHeight;
      info.height:=t.rowHeight;
      info.row:=row;

      t.DrawRow(info);

      draw.FreeLastClip;
    END;
  END RedrawRow;

  (**
    Redraw complete table.
  **)

  PROCEDURE (t : Table) DrawText;

  VAR
    row,
    lastRow : LONGINT;
    draw    : D.DrawInfo;
    info    : DrawInfoDesc;

  BEGIN
    draw:=t.GetDrawInfo();

    t.CalcCol;

    (* TODO: Fix setting of visible areas *)
    IF t.model#NIL THEN
      t.hAdjustment.SetDimension(t.width,t.totalWidth);
      t.vAdjustment.SetDimension(t.height DIV t.rowHeight,t.model.GetRows());
    ELSE
      t.hAdjustment.SetInvalid;
      t.vAdjustment.SetInvalid;
    END;

    IF t.model#NIL THEN
      (* Clip complete content of table *)
      draw.InstallClip(t.x,t.y,t.width,t.height);

      info.font:=D.display.GetFont(D.normalFont);
      info.draw:=draw;

      lastRow:=U.MinLong(t.vAdjustment.GetTop()+t.vAdjustment.GetVisible(),t.model.GetRows());

      FOR row:=t.vAdjustment.GetTop() TO lastRow DO
        info.y:=t.y+row*t.rowHeight-t.vAdjustment.GetTop()*t.rowHeight;
        info.height:=t.rowHeight;
        info.row:=row;

        t.DrawRow(info);
      END;


      (* Clear the area under the last row, if visible *)
      IF (lastRow-t.vAdjustment.GetTop()+1)*t.rowHeight<t.height THEN
        t.DrawBackground(t.x,
                         t.y+(lastRow-t.vAdjustment.GetTop()+1)*t.rowHeight,
                         t.width,
                         t.height-(lastRow-t.vAdjustment.GetTop()+1)*t.rowHeight);
      END;

      draw.FreeLastClip;
    END;
  END DrawText;

  (**
    Make the given cell visible. Handle an negative number, if the you want
    the method to ignore the column or row.
  **)

  PROCEDURE (t : Table) MakeVisible*(x,y : LONGINT);

  VAR
    width,i : LONGINT;

  BEGIN
    IF t.model#NIL THEN
      t.vAdjustment.MakeVisible(y);
      IF (x>0) & (x<=t.model.GetColumns()) THEN
        width:=0;
        FOR i:=0 TO x-2 DO
          INC(width,t.model.GetColumnWidth(i));
        END;

        t.hAdjustment.MakeVisible(width);
      END;
    END;
  END MakeVisible;

  (**
    Return the cell under the given coordinates. Returns FALSE if there is no
    cell at the given position.
  **)

  PROCEDURE (t : Table) GetCell(mx,my : LONGINT; VAR x,y : LONGINT):BOOLEAN;

  VAR
    start : LONGINT;

  BEGIN
    IF t.model=NIL THEN
      RETURN FALSE;
    END;

    y:=(my-t.y) DIV t.rowHeight+1;
    IF (y<1) OR (y+t.vAdjustment.GetTop()-1>t.vAdjustment.GetTotal()) THEN
      RETURN FALSE;
    END;

    INC(y,t.vAdjustment.GetTop()-1);
    DEC(mx,t.x-t.hAdjustment.GetTop());

    start:=0;
    FOR x:=0 TO t.model.GetColumns()-1 DO
      IF (mx>=start) & (mx<start+t.model.GetColumnWidth(x)) THEN
        RETURN TRUE;
      END;
      INC(start,t.model.GetColumnWidth(x));
    END;

    RETURN FALSE;
  END GetCell;

  PROCEDURE (t : Table) OnMouseSelection*;

  VAR
    selection : MouseSelectionMsg;

  BEGIN
    NEW(selection);

    t.Send(selection,mouseSelectionMsg);
  END OnMouseSelection;

  PROCEDURE (t : Table) OnDoubleClick*;

  VAR
    msg : DoubleClickMsg;

  BEGIN
    NEW(msg);

    t.Send(msg,doubleClickMsg);
  END OnDoubleClick;

  PROCEDURE (t : Table) Up;

  BEGIN
    CASE t.model.GetSelectionType() OF
      TM.noSelect:
        t.vAdjustment.DecTop;
    | TM.cellSelect:
        IF t.model.sy>1 THEN
          t.model.SelectCell(t.model.sx,t.model.sy-1);
          t.MakeVisible(t.model.sx,t.model.sy);
        END;
    | TM.singleLineSelect:
        IF t.model.sy>1 THEN
          t.model.SelectRow(t.model.sy-1);
          t.MakeVisible(t.model.sx,t.model.sy);
        END;
    ELSE
    END;
  END Up;

  PROCEDURE (t : Table) Down;

  BEGIN
    CASE t.model.GetSelectionType() OF
      TM.noSelect:
        t.vAdjustment.IncTop;
    | TM.cellSelect:
        IF t.model.sy>1 THEN
          t.model.SelectCell(t.model.sx,t.model.sy+1);
          t.MakeVisible(t.model.sx,t.model.sy);
        END;
    | TM.singleLineSelect:
        IF t.model.sy<t.model.GetRows() THEN
          t.model.SelectRow(t.model.sy+1);
          t.MakeVisible(t.model.sx,t.model.sy);
        END;
    ELSE
    END;
  END Down;

  PROCEDURE (t : Table) Left;

  BEGIN
    CASE t.model.GetSelectionType() OF
      TM.noSelect:
        t.hAdjustment.DecTop;
    | TM.cellSelect:
        IF t.model.sx>1 THEN
          t.model.SelectCell(t.model.sx-1,t.model.sy);
          t.MakeVisible(t.model.sx,t.model.sy);
        END;
    | TM.singleLineSelect:
        t.hAdjustment.DecTop;
    ELSE
    END;
  END Left;

  PROCEDURE (t : Table) Right;

  BEGIN
    CASE t.model.GetSelectionType() OF
      TM.noSelect:
        t.hAdjustment.IncTop;
    | TM.cellSelect:
        IF t.model.sx<t.model.GetColumns() THEN
          t.model.SelectCell(t.model.sx+1,t.model.sy);
          t.MakeVisible(t.model.sx,t.model.sy);
        END;
    | TM.singleLineSelect:
        t.hAdjustment.IncTop;
    ELSE
    END;
  END Right;

  PROCEDURE (t : Table) HandleMouseEvent*(event : E.MouseEvent;
                                          VAR grab : G.Object):BOOLEAN;

  VAR
    x,y : LONGINT;

  BEGIN
    IF ~t.visible OR t.disabled OR (t.model=NIL) THEN
      RETURN FALSE;
    END;

    WITH event : E.ButtonEvent DO
      IF (event.type=E.mouseDown) & t.PointIsIn(event.x,event.y) THEN
        IF (event.button=E.button1) THEN
          IF t.GetCell(event.x,event.y,x,y) THEN
            CASE t.model.GetSelectionType() OF
              TM.cellSelect:
                t.model.SelectCell(x,y);
                t.OnMouseSelection;
            | TM.singleLineSelect:
                t.model.SelectRow(y);
                t.OnMouseSelection;
            ELSE
            END;

            IF D.display.IsDoubleClicked() THEN (* TODO: Define double click more precisly *)
              t.OnDoubleClick;
            END;
          END;
        ELSIF (event.button=E.button4) THEN
          t.Up;
        ELSIF (event.button=E.button5) THEN
          t.Down;
        END;

        grab:=t;

        RETURN TRUE;

      ELSIF (event.type=E.mouseUp) THEN
        grab:=NIL;

        RETURN TRUE;

      END;
    ELSE
    END;

    RETURN FALSE;
  END HandleMouseEvent;

  PROCEDURE (t : Table) HandleKeys(event :  E.KeyEvent):BOOLEAN;

  VAR
    keysym : LONGINT;
    select : LONGINT;

  BEGIN
    keysym:=event.GetKey();
    select:=t.model.GetSelectionType();

    CASE keysym OF
    | E.left:
      t.Left;
    | E.right:
      t.Right;
    | E.up:
      t.Up;
    | E.down:
      t.Down;
    | E.home:
      CASE select OF
        TM.noSelect,
        TM.singleLineSelect:
          t.vAdjustment.SetTop(1);
      | TM.cellSelect:
          IF t.model.sx#1 THEN
            t.model.SelectCell(1,t.model.sy);
            t.MakeVisible(t.model.sx,t.model.sy);
          END;
      ELSE
      END;
    | E.end:
      CASE select OF
        TM.noSelect,
        TM.singleLineSelect:
          t.vAdjustment.SetTop(U.MaxLong(1,t.vAdjustment.GetTotal()-t.vAdjustment.GetVisible()+1));
      | TM.cellSelect:
          IF t.model.sx#t.model.GetColumns() THEN
            t.model.SelectCell(t.model.GetColumns(),t.model.sy);
            t.MakeVisible(t.model.sx,t.model.sy);
          END;
      ELSE
      END;
    | E.pageUp:
      CASE select OF
        TM.noSelect,
        TM.singleLineSelect,
        TM.cellSelect:
          t.vAdjustment.PageBack;
      ELSE
      END;
    | E.pageDown:
      CASE select OF
        TM.noSelect,
        TM.singleLineSelect,
        TM.cellSelect:
          t.vAdjustment.PageForward;
      ELSE
      END;
    ELSE
    END;
    RETURN FALSE;
  END HandleKeys;

  PROCEDURE (t : Table) HandleKeyEvent*(event : E.KeyEvent):BOOLEAN;

  BEGIN
    IF event.type=E.keyDown THEN
      RETURN t.HandleKeys(event);
    ELSE
      RETURN FALSE;
    END;
  END HandleKeyEvent;

  (**
    Completely reinitialize the table.
  **)

  PROCEDURE (t : Table) ReInit;

  BEGIN
    IF t.visible THEN
      IF t.model#NIL THEN
        t.hAdjustment.SetDimension(t.width,t.totalWidth);
        t.vAdjustment.SetDimension(t.height DIV t.rowHeight,t.model.GetRows());
      ELSE
        t.hAdjustment.SetDimension(0,0);
        t.vAdjustment.SetDimension(0,0);
      END;

      t.DrawText;
    END;
  END ReInit;

  (**
    Set the table model for the table object.
  **)

  PROCEDURE (t : Table) SetModel*(model : O.Model);

  BEGIN
    IF t.model#NIL THEN
      t.UnattachModel(t.model);
    END;
    IF (model#NIL) & (model IS TM.TableModel) THEN
      t.model:=model(TM.TableModel);
      t.AttachModel(model);
      t.ReInit;
    ELSE
      t.model:=NIL
    END;
  END SetModel;

  (**
    This function is used to check if an argument to SetModel
    was successfully accepted.
   **)

  PROCEDURE (t : Table) ModelAccepted * (m : O.Model):BOOLEAN;

  BEGIN
    RETURN m=t.model;
  END ModelAccepted;

  PROCEDURE (t : Table) Draw*(x,y,w,h : LONGINT);

  BEGIN
    t.Draw^(x,y,w,h);

    IF ~t.Intersect(x,y,w,h) THEN
      RETURN;
    END;

    t.DrawText;
  END Draw;

  PROCEDURE (t : Table) Hide*;

  BEGIN
    IF t.visible THEN

      t.DrawHide;
      t.Hide^;
    END;
  END Hide;

  (**
    Draw the keyboard focus.
  **)

  PROCEDURE (t : Table) DrawFocus*;

  BEGIN
    t.DrawText;
  END DrawFocus;

  (**
    Hide the keyboard focus.
  **)

  PROCEDURE (t : Table) HideFocus*;

  BEGIN
    t.DrawText;
  END HideFocus;

  PROCEDURE (t : Table) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF model=t.model THEN
      IF msg#NIL THEN
        WITH
          msg : TM.RefreshCell DO
            IF t.visible THEN
              t.RedrawRow(msg.y);
            END;
        | msg : TM.RefreshRow DO
            IF t.visible THEN
              t.RedrawRow(msg.y);
            END;
        | msg : TM.InsertRow DO
            IF t.visible THEN
              t.ReInit;
            END;
        | msg : TM.DeleteRow DO
            IF t.visible THEN
              t.ReInit;
            END;
        ELSE
          t.ReInit;
        END;
      ELSE
        t.ReInit;
      END;
    ELSIF t.visible THEN (* scrolling event *)
      t.DrawText;
    END;
  END Resync;

  PROCEDURE CreateTable*():Table;

  VAR
    table : Table;

  BEGIN
    NEW(table);
    table.Init;

    RETURN table;
  END CreateTable;

BEGIN
  NEW(prefs);
  prefs.Init;
END VO:TableView.