(**
   Implements a tab groupobject.

  TODO
  * Hide is suboptimal.
**)

MODULE VO:Tab;

(*
    Implements a tab gadget.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)


IMPORT D  := VO:Base:Display,
       E  := VO:Base:Event,
       O  := VO:Base:Object,
       U  := VO:Base:Util,

       G  := VO:Object,
       T  := VO:Text;


CONST
  changedMsg * = 0;

  (* internal *)

  horizFrameThick = 2;
  vertFrameThick  = 2;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 hilightLabel* : BOOLEAN;
               END;

  TabEntry     = POINTER TO TabEntryDesc;
  TabEntryDesc = RECORD
                   next   : TabEntry;
                   label  : G.Object;
                   object : G.Object;
                 END;

  Tab*     = POINTER TO TabDesc;
  TabDesc* = RECORD (G.GadgetDesc)
               mw       : LONGINT; (* greatest width of all tab-objects *)
               mh       : LONGINT; (* greatest height of all tab-object *)

               tabList,
               lastTab  : TabEntry;
               current  : G.Object;
               selected-: LONGINT; (* The selected tab (1..n) *)
               curCount : LONGINT; (* the currently selected tab during clicking *)
               isIn     : BOOLEAN; (* used during selection action *)
               count    : LONGINT; (* number of tabs *)
             END;

  (* messages *)

  ChangedMsg*     = POINTER TO ChangedMsgDesc;

  (**
    The PressedMsg generated everytime the button get clicked.
  **)

  ChangedMsgDesc* = RECORD (O.MessageDesc)
                      pos* : LONGINT;
                    END;

VAR
  prefs* : Prefs;


  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.hilightLabel:=FALSE;
  END Init;

  PROCEDURE (t : Tab) Init*;

  BEGIN
    t.Init^;

    t.SetFlags({G.canFocus});
    t.RemoveFlags({G.stdFocus});

    t.SetPrefs(prefs); (* We set the prefs *)

    t.tabList:=NIL;
    t.curCount:=0;
    t.selected:=1;
  END Init;

  PROCEDURE (t : Tab) AddTab*(label, object : G.Object);

  VAR
    entry : TabEntry;

  BEGIN
    label.SetParent(t);
    object.SetParent(t);

    NEW(entry);
    entry.label:=label;
    entry.object:=object;

    IF t.tabList=NIL THEN
      t.tabList:=entry;
    ELSE
      t.lastTab.next:=entry;
    END;
    t.lastTab:=entry;
    INC(t.count);
  END AddTab;

  PROCEDURE (t : Tab) AddStringTab*(string : ARRAY OF CHAR; object : G.Object);

  BEGIN
    t.AddTab(T.MakeCenterText(string),object);
  END AddStringTab;

  PROCEDURE (t : Tab) CalcSize*;

  VAR
    entry : TabEntry;
    ow,oh : LONGINT;

  BEGIN
    t.width:=0;
    t.height:=0;
    t.mw:=0;
    t.mh:=0;
    ow:=0;
    oh:=0;

    entry:=t.tabList;
    WHILE entry#NIL DO
      IF entry.label.StdFocus() THEN
        t.SetFlags({G.stdFocus});
      END;
      entry.label.CalcSize;
      t.mw:=U.MaxLong(t.mw,entry.label.oWidth);
      t.mh:=U.MaxLong(t.mh,entry.label.oHeight);
      entry.object.CalcSize;
      ow:=U.MaxLong(ow,entry.object.oWidth);
      oh:=U.MaxLong(oh,entry.object.oHeight);
      entry:=entry.next;
    END;

    IF ~t.StdFocus() & t.MayFocus() THEN
      entry:=t.tabList;
      WHILE entry#NIL DO
        entry.label.SetFlags({G.mayFocus});
        entry:=entry.next;
      END;
    END;

    INC(t.mh,D.display.spaceHeight+2*horizFrameThick);
    INC(t.mw,D.display.spaceWidth+2*vertFrameThick);
    INC(ow,2*D.display.spaceWidth);
    INC(oh,2*D.display.spaceHeight);

    t.height:=t.mh+oh+horizFrameThick;
    t.width:=U.MaxLong(ow+2*vertFrameThick,(*D.display.spaceWidth+*)t.count*t.mw(*+D.display.spaceWidth*));

    t.minWidth:=t.width;
    t.minHeight:=t.height;

    t.CalcSize^;
  END CalcSize;

  PROCEDURE (t : Tab) GetEntry(pos :LONGINT): TabEntry;

  VAR
    entry : TabEntry;

  BEGIN
    entry:=t.tabList;
    WHILE pos>1 DO
      entry:=entry.next;
      DEC(pos);
    END;
    RETURN entry;
  END GetEntry;

  PROCEDURE (t : Tab) GetLabel(pos :LONGINT): G.Object;

  VAR
    entry : TabEntry;

  BEGIN
    entry:=t.tabList;
    WHILE pos>1 DO
      entry:=entry.next;
      DEC(pos);
    END;
    RETURN entry.label;
  END GetLabel;

  PROCEDURE (t : Tab) GetEntryAtPoint(x,y :LONGINT):LONGINT;

  VAR
    entry : TabEntry;
    pos   : LONGINT;

  BEGIN
    entry:=t.tabList;
    pos:=1;
    WHILE entry#NIL DO
      IF entry.label.PointIsIn(x,y) THEN
        RETURN pos;
      END;
      entry:=entry.next;
      INC(pos);
    END;
    RETURN 0;
  END GetEntryAtPoint;

  PROCEDURE (t : Tab) DrawTab(pos : LONGINT; selected : BOOLEAN);

  VAR
    x      : LONGINT;
    entry  : TabEntry;
    object : G.Object;
    draw   : D.DrawInfo;

  BEGIN
    draw:=t.GetDrawInfo();

    entry:=t.GetEntry(pos);
    object:=entry.label;

    x:=t.x+(pos-1)*t.mw;

    IF selected & t.prefs(Prefs).hilightLabel THEN
      draw.mode:={D.selected};
    END;
    object.SetFlags({G.horizontalFlex,G.verticalFlex});
    object.Resize(t.mw-2*horizFrameThick,
                  t.mh-2*vertFrameThick);
    object.Move(x  +(t.mw-object.oWidth) DIV 2,
                t.y+(t.mh-object.oHeight) DIV 2);
    object.Draw(t.oX,t.oY,t.oWidth,t.oHeight);
    draw.mode:={};

    IF selected THEN
      draw.PushForeground(D.shadowColor);
    ELSE
      draw.PushForeground(D.shineColor);
    END;
    draw.DrawLine(x+3,t.y  ,x+t.mw-4,t.y);
    draw.DrawLine(x+1,t.y+1,x+t.mw-3,t.y+1);
    draw.DrawLine(x  ,t.y+3,x       ,t.y+t.mh-1-horizFrameThick);
    draw.DrawLine(x+1,t.y+1,x+1     ,t.y+t.mh-1-horizFrameThick);
    draw.PopForeground;

    IF selected THEN
      draw.PushForeground(D.shineColor);
    ELSE
      draw.PushForeground(D.shadowColor);
    END;
    draw.DrawLine(x+t.mw-1,t.y+3,x+t.mw-1,t.y+t.mh-1-horizFrameThick);
    draw.DrawLine(x+t.mw-2,t.y+1,x+t.mw-2,t.y+t.mh-1-horizFrameThick);
    draw.PopForeground;
  END DrawTab;

  PROCEDURE (t : Tab) DrawTabs;

  VAR
    x : LONGINT;

  BEGIN
    FOR x:=1 TO t.count DO
      t.DrawTab(x,FALSE);
    END;

    IF t.parent#NIL THEN
      t.parent.DrawBackground(t.x+t.count*t.mw,
                              t.y,
                              t.width-t.count*t.mw,
                              t.mh-horizFrameThick);
    ELSE
      t.DrawBackground(t.x+t.count*t.mw,
                       t.y,
                       t.width-t.count*t.mw,
                       t.mh-horizFrameThick);
    END;
  END DrawTabs;

  PROCEDURE (t : Tab) DrawTop;

  VAR
    draw : D.DrawInfo;

  BEGIN
    draw:=t.GetDrawInfo();

    draw.PushForeground(D.shineColor);
    draw.DrawLine(t.x,t.y+t.mh-2,
                  t.x+t.width-1-vertFrameThick,t.y+t.mh-2);
    draw.DrawLine(t.x,t.y+t.mh-1,
                  t.x+t.width-1-vertFrameThick,t.y+t.mh-1);
    draw.PopForeground;

    t.DrawBackground(t.x+(t.selected-1)*t.mw+vertFrameThick,
                     t.y+t.mh-2,
                     t.mw-2*vertFrameThick,
                     2);

    IF t.disabled THEN
      draw.PushForeground(D.disabledColor);
      draw.PushPattern(D.disablePattern,D.disableWidth,D.disableHeight,D.fgPattern);
      draw.FillRectangle(t.x,t.y,t.width,t.mh-horizFrameThick);
      draw.PopPattern;
      draw.PopForeground;
    END;
  END DrawTop;

  PROCEDURE (t : Tab) DrawObject(pos : LONGINT);

  VAR
    entry : TabEntry;
    draw  : D.DrawInfo;

  BEGIN
    entry:=t.GetEntry(pos);
    IF (t.current#entry.object) & (t.current#NIL) THEN
      t.current.Hide;
    END;

    t.current:=entry.object;
    t.selected:=pos;

    t.current.Resize(t.width-2*vertFrameThick-2*D.display.spaceWidth,
                     t.height-t.mh-horizFrameThick-2*D.display.spaceWidth);

    draw:=t.GetDrawInfo();

    draw.InstallClip(t.x+2,t.y+t.mh,t.width-4,t.height-t.mh-2);
    draw.SubRegion(t.x+(t.width-t.current.oWidth) DIV 2,
                   t.y+t.mh+(t.height-t.mh-t.current.oHeight) DIV 2,
                   t.current.oWidth,t.current.oHeight);

    t.DrawBackground(t.x+2,t.y+t.mh,t.width-4,t.height-t.mh-2);

    draw.FreeLastClip;

    t.current.Move(t.x+(t.width-t.current.oWidth) DIV 2,
                   t.y+t.mh+(t.height-t.mh-t.current.oHeight) DIV 2);
    t.current.Draw(t.oX,t.oY,t.oWidth,t.oHeight);
  END DrawObject;

  PROCEDURE (t : Tab) HandleMouseEvent*(event : E.MouseEvent;
                                        VAR grab : G.Object):BOOLEAN;

  VAR
    changed : ChangedMsg;
    object  : G.Object;
    pos     : LONGINT;

  BEGIN
    IF ~t.visible OR t.disabled THEN
      RETURN FALSE;
    END;

    WITH event : E.ButtonEvent DO
      IF (event.type=E.mouseDown) & t.PointIsIn(event.x,event.y)
      & (event.button=E.button1) THEN
        pos:=t.GetEntryAtPoint(event.x,event.y);
        IF pos>0 THEN
          t.curCount:=pos;
          IF t.curCount#t.selected THEN
            t.DrawTab(t.curCount,TRUE);
            t.isIn:=TRUE;

            grab:=t;
            RETURN TRUE;
          END;
        END;
      ELSIF (event.type=E.mouseUp) & (event.button=E.button1) THEN
        object:=t.GetLabel(t.curCount);
        IF object.PointIsIn(event.x,event.y) THEN
          t.HideFocus;
          t.DrawTab(t.curCount,FALSE);
          t.DrawObject(t.curCount);
          t.DrawTop;
          t.DrawFocus;

          NEW(changed);
          changed.pos:=t.selected;
          t.Send(changed,changedMsg);
        END;

        grab:=NIL;
        RETURN TRUE;
      END;
    | event : E.MotionEvent DO
      IF grab=t THEN
        object:=t.GetLabel(t.curCount);
        IF object.PointIsIn(event.x,event.y) THEN
          IF ~t.isIn THEN
            t.DrawTab(t.curCount,TRUE);
            t.isIn:=TRUE;
          END;
        ELSIF t.isIn THEN
          t.DrawTab(t.curCount,FALSE);
          t.isIn:=FALSE;
        END;
      END;

      RETURN TRUE;
    ELSE
    END;

    IF t.current#NIL THEN
      RETURN t.current.HandleMouseEvent(event,grab);
    ELSE
      RETURN FALSE;
    END;
  END HandleMouseEvent;

  PROCEDURE (t : Tab) HandleKeyEvent*(event : E.KeyEvent):BOOLEAN;

  VAR
    keysym : LONGINT;

  BEGIN
    IF event.type=E.keyDown THEN
      keysym:=event.GetKey();
      IF keysym=E.left THEN
        IF t.count>1 THEN
          t.HideFocus;
          DEC(t.selected);
          IF t.selected<1 THEN
            t.selected:=t.count;
          END;
          t.DrawObject(t.selected);
          t.DrawTop;
          t.DrawFocus;
        END;
        RETURN TRUE;
      ELSIF keysym=E.right THEN
        IF t.count>1 THEN
          t.HideFocus;
          INC(t.selected);
          IF t.selected>t.count THEN
            t.selected:=1;
          END;
          t.DrawObject(t.selected);
          t.DrawTop;
          t.DrawFocus;
        END;
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END HandleKeyEvent;

  PROCEDURE (t : Tab) GetPosObject*(x,y : LONGINT; type : LONGINT):G.Object;

  VAR
    object : TabEntry;
    return : G.Object;

  BEGIN
    IF t.current=NIL THEN
      RETURN NIL;
    END;

    return:=t.current.GetPosObject(x,y,type);
    IF return#NIL THEN
      RETURN return;
    END;

    object:=t.tabList;
    WHILE object#NIL DO
      return:=object.label.GetPosObject(x,y,type);
      IF return#NIL THEN
        RETURN return;
      END;
      object:=object.next;
    END;
    RETURN t.GetPosObject^(x,y,type);
  END GetPosObject;

  PROCEDURE (t : Tab) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;

  VAR
    object : TabEntry;
    return : G.Object;

  BEGIN
    IF t.current=NIL THEN
      RETURN NIL;
    END;

    return:=t.current.GetDnDObject(x,y,drag);
    IF return#NIL THEN
      RETURN return;
    END;

    object:=t.tabList;
    WHILE object#NIL DO
      return:=object.label.GetDnDObject(x,y,drag);
      IF return#NIL THEN
        RETURN return;
      END;
      object:=object.next;
    END;
    RETURN NIL;
  END GetDnDObject;

  PROCEDURE (t : Tab) Draw*(x,y,w,h : LONGINT);

  VAR
    draw : D.DrawInfo;

  BEGIN
    t.Draw^(x,y,w,h);

    IF ~t.Intersect(x,y,w,h) THEN
      RETURN;
    END;

    draw:=t.GetDrawInfo();

    t.DrawTabs;

    draw.PushForeground(D.shadowColor);

    (* right line *)
    draw.DrawLine(t.x+t.width-2,t.y+t.mh  ,t.x+t.width-2,t.y+t.height-1-2);
    draw.DrawLine(t.x+t.width-1,t.y+t.mh-1,t.x+t.width-1,t.y+t.height-1-1);

    (* bottom line *)
    draw.DrawLine(t.x,  t.y+t.height-1,t.x+t.width-1,t.y+t.height-1);
    draw.DrawLine(t.x+1,t.y+t.height-2,t.x+t.width-2,t.y+t.height-2);
    draw.PopForeground;

    draw.PushForeground(D.shineColor);

    (* left line *)
    draw.DrawLine(t.x  ,t.y+t.mh,t.x  ,t.y+t.height-1-1);
    draw.DrawLine(t.x+1,t.y+t.mh,t.x+1,t.y+t.height-1-2);
    draw.PopForeground;

    t.DrawObject(t.selected);
    t.DrawTop;
  END Draw;

  (**
    Draw the keyboard focus.
  **)

  PROCEDURE (t : Tab) DrawFocus*;

  VAR
    entry : TabEntry;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    entry:=t.GetEntry(t.selected);
    IF (entry#NIL) & ~entry.label.StdFocus() THEN
      entry.label.DrawFocus;
    ELSE
      (* Delegate drawing to the baseclass *)
      t.DrawFocus^;
    END;
  END DrawFocus;

  (**
    Hide the keyboard focus.
  **)

  PROCEDURE (t : Tab) HideFocus*;

  VAR
    entry : TabEntry;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    entry:=t.GetEntry(t.selected);
    IF (entry#NIL) & ~entry.label.StdFocus() THEN
      entry.label.HideFocus;
    ELSE
      (* Delegate drawing to the baseclass *)
      t.HideFocus^;
    END;
  END HideFocus;

  PROCEDURE (t : Tab) Hide*;

  VAR
    entry : TabEntry;

  BEGIN
    IF t.visible THEN
      entry:=t.tabList;
      WHILE entry#NIL DO
        entry.label.Hide;
        entry:=entry.next;
      END;
      t.current.Hide;
      t.DrawHide;  (* That is suboptimal :-( We should only clear our own frames *)
      t.Hide^;
    END;
  END Hide;

  PROCEDURE CreateTab*():Tab;

  VAR
    tab : Tab;

  BEGIN
    NEW(tab);
    tab.Init;

    RETURN tab;
  END CreateTab;

BEGIN
  NEW(prefs);
  prefs.Init;
END VO:Tab.